#!/usr/bin/env python

# From data of anchors, we construct a C2-continuous Gimp path.
# This is an implementation of the ideas in Stuart Kent's text:
# https://www.stkent.com/2015/07/03/building-smooth-paths-using-bezier-curves.html
# with some modifications: equations extended to produce G2-continuous curves;
# closed strokes are included.

# History:
# v0.1: 2022-02-24: First, experimental plugin.
# v0.2: 2022-03-01: Restrict the effect according to selection.
# v0.3: 2022-03-07: Better restriction. Option to straighten unprocessed segments.
# v0.4: 2022-03-09: Open strokes: option: keep the first or the last segment.
# v0.5: 2022-03-13: Open strokes: option: keep the first and last direction.
#                   Abandon "keep the first or the last segment".
# v0.6: 2022-03-17: Tunings. First published version.
# v0.7: 2022-03-28: Added option to preserve non-zero handles.
#                   Removed option to preserve tangent directions.
#                   Removed option to straighten unprocessed segments.
#                   Unpublished
# v0.8: 2022-04-02: G1-continuity at anchors where precisely one preserved handle.

# (c) Markku Koppinen 2021
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published
#   by the Free Software Foundation; either version 3 of the License, or
#   (at your option) any later version.
#
#   This very file is the complete source code to the program.
#
#   If you make and redistribute changes to this code, please mark it
#   in reasonable ways as different from the original version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   The GPL v3 licence is available at: https://www.gnu.org/licenses/gpl-3..en.html'

from __future__ import division, print_function
from gimpfu import *
from math import *
from cmath import exp as cexp
from copy import copy, deepcopy

#==============================================================
#             class BCurve       
#==============================================================

# Note: The plane is treated as the complex plane.

class BCurve(object):
    """Data structures to handle Gimp vectors objects by means of individual
    Bezier arcs.
    By a "Bezier arc" we mean here a Bezier curve determined by 4 control points
    p0,p1,p2,p3.
    This class contains data structures needed to do computations on
    a Bezier curve which are done arc by arc.
    To this end, Gimp data structures
    - gimp.Vectors
    - gimp.VectorsBezierStroke
    are here represented as data structures (classes)
    - GimpVectors
    - GimpStroke
    - BezierCurve
    - BezierArc
    - ControlPoint = complex
    with the aim that
    - each control point [x,y] is represented as an instance of ControlPoint;
    - each Bezier arc (four control points, see above) is represented as an instance
      of BezierArc.
    """
    
    ControlPoint = complex
    
    class GimpVectors(object):
        """Essentially same data as in a gimp.Vectors object except that
        strokes are instances of GimpStroke rather than
        gimp.VectorsBezierStroke.
        Attributes:
        - stroke_list: [GimpStroke]
        - name:        string
        """
        def __init__(self,
                     stroke_list,            # [GimpStroke]
                     name='GimpVectors'      # string
                     ):
            self.stroke_list = stroke_list
            self.name = name
        def __str__(self):
            s = 'GimpVectors '+self.name
            count = 0
            for stroke in self.stroke_list:
                s += '\n  Stroke '+str(count)+': GimpStroke'
                s += '\n    '+str(stroke)
                count += 1
            return s
        def gv2vectors_object(self, image, name=None):
            """Conversion GimpVectors -> gimp.Vectors
            (The converse: vectors_object2gv)
            """
            def cp_list2xy_list(cp_list):
                xy = []
                for cp in cp_list:
                    xy += [cp.real, cp.imag]
                return xy
            if name is None:
                name = self.name
            vectors = pdb.gimp_vectors_new(image, name)
            for stroke in self.stroke_list:
                closed = stroke.closed
                xy_list = cp_list2xy_list(stroke.cp_list)
                stroke_id = pdb.gimp_vectors_stroke_new_from_points(
                             vectors, 0, len(xy_list), xy_list, closed)
            return vectors
    
    class GimpStroke(object):
        """Essentially same data as in a gimp.VectorsBezierStroke.points,
        except that control points list [x,y,x,y,...] is arranged as
        a list [ControlPoint] by joining successive x,y.
        Attributes:
        - cp_list:     [ControlPoint]
        - closed:      boolean
        - stroke_name: string
        """
        def __init__(self,
                     cp_list,                   # [ControlPoint]
                     closed,                    # boolean
                     stroke_name = 'GimpStroke' # string
                     ):
            self.stroke_name = stroke_name
            self.cp_list = cp_list
            self.closed = closed
        def __str__(self):
            s = self.stroke_name + '; Control points:'
            for cp in self.cp_list:
                s += '\n    '+str(cp)
            s += '\n    closed: '+str(self.closed)
            return s
        def gs2bc(self):
            """Conversion GimpStroke -> BezierCurve
            """
            head = self.cp_list[0]
            tail = self.cp_list[-1]
            inner = self.cp_list[1:-1]
            if len(inner) > 1: # Must be divisible by 4
                ba_list = []
                count = 0
                for i in range(0,len(inner)-1,3):
                    ba_list.append(BCurve.BezierArc(inner[i:i+4], 'arc '+str(count)))
                    count += 1
            elif len(inner) == 1: # Stroke has only one anchor
                ba_list = [BCurve.BezierArc([inner[0]], 'arc 0')]
            else:
                raise Exception("BCurve.GimpStroke.gs2bc: No anchors in stroke?")
            return BCurve.BezierCurve(
                               bezier_arcs = ba_list,
                               head_handle = head,
                               tail_handle = tail,
                               closed = self.closed,
                               curve_name = self.stroke_name
                               )
    
    class BezierCurve(object):
        """BezierCurve is a list of butting Bezier arcs with some extra data.
        Attributes:
        - curve_name:  string
        - bezier_arcs: [BezierArc];
        - head_handle: ControlPoint;
        - tail_handle: ControlPoint;
        - closed:      boolean.
        Note: "Butting" means that for any pair of successive arcs, the last
              control point of the former equals the first control point of
              the latter.
              In initialization no checks are done about this condition.
        """
        def __init__(self,
                     bezier_arcs=[],           # [BezierArc]
                     head_handle=None,         # ControlPoint
                     tail_handle=None,         # ControlPoint
                     closed=False,             # boolean
                     curve_name='BezierCurve', # string
                     ):
            self.curve_name = curve_name
            self.bezier_arcs = bezier_arcs
            if head_handle is None:
                self.head_handle = bezier_arcs[0].cp4[0]
            else:
                self.head_handle = head_handle
            if tail_handle is None:
                self.tail_handle = bezier_arcs[-1].cp4[-1]
            else:
                self.tail_handle = tail_handle
            self.closed = closed
        def __str__(self):
            s = self.curve_name + ': Bezier curve' + '; Bezier arcs:'
            count = 0
            for arc in self.bezier_arcs:
                s += '\n  arc '+ str(count)+ ': ' +str(arc)
                count += 1
            s += '\n  Head handle:    '+str(self.head_handle)
            s += '\n  Tail handle:    '+str(self.tail_handle)
            s += '\n  Closed:'    +str(self.closed)
            return s
        def bc2gs(self):
            """Conversion BezierCurve -> GimpStroke
            """
            cp_list = [self.head_handle]
            if len(self.bezier_arcs[0].cp4) > 1:
                for ba in self.bezier_arcs:
                    cp_list += ba.cp4[:3]
                cp_list.append(self.bezier_arcs[-1].cp4[-1])
            else:
                # Only one BezierArc and it has cp4 = [p0]
                # (may rise from Gimp stroke with only one anchor).
                cp_list.append(self.bezier_arcs[0].cp4[0])
            cp_list.append(self.tail_handle)
            return BCurve.GimpStroke(cp_list, self.closed, self.curve_name)
        def bc2vectors_object(self, image, name=None):
            """Conversion BezierCurve -> gimp.Vectors
            Makes vectors_object with 1 stroke.
            """
            if name is None:
                name = self.curve_name
            gs = self.bc2gs()             # BCurve.GimpStroke
            gv = BCurve.GimpVectors([gs]) # BCurve.GimpVectors
            return gv.gv2vectors_object(image, name)
    
    class BezierArc(object):
        """Data structure for one Bezier arc: arc determined by 4 control
           points.
        Attributes:
        - cp4:      [p0,p1,p2,p3] where each pi:ControlPoint, or
                    [p0] if only one control point (may rise from
                    a stroke with only one anchor);
        - arc_name: string
        """
        def __init__(self,
                     cp4,                 # cp4=[p0,p1,p2,p3] with pi:ControlPoint
                                          # or [p0]
                     arc_name='BezierArc' # string
                     ):
            self.cp4 = cp4
            self.arc_name = arc_name
        def reverse(self):
            self.cp4.reverse()
        def __str__(self):
            s = 'BezierArc, control points:'
            for cp in self.cp4:
                s += '\n  '+ str(cp)
            return s

# -----------------
# vectors_object2gv
# -----------------
# Conversion gimp.Vectors -> GimpVectors
# Args:
# - vectors_object: gimp.Vectors (Gimp vectors object)
# Returns:
# - GimpVectors
#
# Note: The inverse conversion is GimpVectors.gv2vectors_object(...).
#       See also BezierCurve.bc2vectors_object(...).
def vectors_object2gv(vectors_object):
    def xy_list2cp_list(xy_list): # xy_list = [x,y,x,y,...,]
        cp = []
        for i in range(0,len(xy_list),2):
            cp.append(BCurve.ControlPoint(xy_list[i], xy_list[i+1]))
        return cp
    stroke_list = []
    count = 0
    for stroke in vectors_object.strokes:
        xy_list = stroke.points[0]
        closed = stroke.points[1]
        cp_list = xy_list2cp_list(xy_list)
        stroke_name = vectors_object.name + ' stroke '+str(count)
        stroke_list.append(BCurve.GimpStroke(cp_list, closed, stroke_name))
        count += 1
    return BCurve.GimpVectors(stroke_list, vectors_object.name)

# ---------------------
# completed_bezier_arcs
# ---------------------
# Give bc:BCurve.BezierCurve, get the list of bezier arcs where, in the case of
# closed bc, a gap-closing arc is added at the end, with necessary modification
# for the case of closed 1-anchor stroke.
# Only copies of the original arcs are returned.
# Args:
# - bc:BCurve.BezierCurve
# Returns:
# - [BCurve.BezierArc]
def completed_bezier_arcs(bc):
    bas = deepcopy(bc.bezier_arcs)
    if not bc.closed:
        return bas
    gap_ba=BCurve.BezierArc(cp4=[bas[-1].cp4[-1],
                            bc.tail_handle,
                            bc.head_handle,
                            bas[0].cp4[0]])
    if len(bas[0].cp4) == 1: # 1-anchor stroke
        return [gap_ba]
    else:
        return bas + [gap_ba]

#==============================================================
#                         Draw
#==============================================================
# ------------------------
# gimp_draw_vectors_object
# ------------------------
# The Gimp version of the routine 'drawing_routine_for_bezier':
# Insert into Gimp the input vectors object, and make it visible if
# visible=True.
# Args:
# - image: Gimp's image (obtained usually from GUI);
# - vectors_object: a vectors object in Gimp;
# - visible: Boolean.
# Returns the vectors object in Gimp, and makes it visible if visible=True.
def gimp_draw_vectors_object(image, vectors_object, visible=True):
    # Insert a Gimp vectors object in Gimp.
    # Adapted from http://registry.gimp.org/files/lqr_wpset-0.14.py.txt
    # by Mike Kazantsev.
    def gimp_insert_vectors(image, vectors_object, position=0):
        if gimp.version >= (2, 8, 0):
            # 2.7.0 has gimp-image-insert-vectors, but fails with
            # parent=None
            parent = None
            return pdb.gimp_image_insert_vectors(
                     image, vectors_object, parent, position)
        else:
            return pdb.gimp_image_add_vectors(
                     image, vectors_object, position)# Deprecated!
    gimp_insert_vectors(image, vectors_object)
    vectors_object.visible = visible
    return vectors_object

#==============================================================
#             Matrix inversion
#==============================================================
# Matrix invertion in Python, code taken from
# https://stackoverflow.com/questions/32114054/matrix-inversion-without-numpy

class SingularMatrixError(Exception):
    def __init__(self, message):
        self.message = message
    def __str__(self):
        return self.message

def eliminate(r1, r2, col, target=0):
    fac = (r2[col]-target) / r1[col]
    for i in range(len(r2)):
        r2[i] -= fac * r1[i]

def gauss(a):
    for i in range(len(a)):
        if a[i][i] == 0:
            for j in range(i+1, len(a)):
                if a[i][j] != 0:
                    a[i], a[j] = a[j], a[i]
                    break
            else:
                #raise ValueError("Matrix is not invertible")
                raise SingularMatrixError("Matrix is not invertible")
        for j in range(i+1, len(a)):
            eliminate(a[i], a[j], i)
    for i in range(len(a)-1, -1, -1):
        for j in range(i-1, -1, -1):
            eliminate(a[i], a[j], i)
    for i in range(len(a)):
        eliminate(a[i], a[i], i, target=1)
    return a

def inverse(a):
    tmp = [[] for _ in a]
    for i,row in enumerate(a):
        assert len(row) == len(a)
        tmp[i].extend(row + [0]*i + [1] + [0]*(len(a)-i-1))
    gauss(tmp)
    ret = []
    for i in range(len(tmp)):
        ret.append(tmp[i][len(tmp[i])//2:])
    return ret
    
#==============================================================
#             class ProcessArc
#==============================================================

def point_in_selection(x,y,image):
    if (0 <= x < image.width) and (0 <= y < image.height):
        _,pixel = pdb.gimp_drawable_get_pixel(image.selection, x, y)
        return (pixel[0] > 127)
    else:
        return False

class ProcessArc(object):
    """Record a Bezier arc together with the info whether it
    should be processed or not, and whether its start and/or end point
    should be a cut point. Record also the info if the arc is the starting
    or ending arc of a stroke.
    Attributes:
    - cp4: [complex,complex,complex,complex] (control point quadruple)
    - process: boolean
    - cut_at_start: boolean
    - cut_at_end: boolean
    """
    def __init__(self,
                 cp4,
                 process=True,
                 cut_at_start=False,
                 cut_at_end=False,
                 ):
        self.cp4 = cp4
        self.process = process
        self.cut_at_start = cut_at_start
        self.cut_at_end = cut_at_end
    def __str__(self):
        s = "ProcessArc:"
        s += "\n  cp4 = "+str(self.cp4)
        s += "\n  process      = "+str(self.process)
        s += "\n  cut_at_start = "+str(self.cut_at_start)
        s += "\n  cut_at_end   = "+str(self.cut_at_end)
        return s

# ---------------
# get_ProcessArcs
# ---------------
# Given a stroke, get the Bezier arcs and the info whether they should be
# processed by the G2 algorithm or not.
# Mark also cut points: points where the stroke must be cut prior the G2 algorithm.
# The cut points are determined by the following rule:
# The anchor between two arcs is a cut point if at least one of the
# following is true:
# - left_arc.cut_at_end = True
# - right_arc.cut_at_start = True
# - left_arc.process != right_arc.process
# Args:
# - image
# - gs:                        BCurve.BezierStroke
# - selection_case:            string
# Returns:
# - [ProcessArc]
def get_ProcessArcs(image,
                    gs,
                    selection_case,
                    preserve_nonzero_handles,
                    print_info=False):
    ZERO = 1e-3
    if print_info:
        print('-------------------')
        print("Starting get_ProcessArcs")
    bc = gs.gs2bc()
    pas = [] # [ProcessArc]
    bas = completed_bezier_arcs(bc)
    for cnt,ba in enumerate(bas):
        p0,p1,p2,p3 = ba.cp4
        p0_in_selection = point_in_selection(p0.real, p0.imag, image)
        p3_in_selection = point_in_selection(p3.real, p3.imag, image)
        if p0_in_selection != p3_in_selection:
            process = False
        elif (p0_in_selection == (selection_case == 'inside')):
            process = True
        else:
            process = False
        if preserve_nonzero_handles:
            cut_at_start = (abs(p0-p1) > ZERO)
            cut_at_end = (abs(p2-p3) > ZERO)
        else:
            cut_at_start = False
            cut_at_end = False
        pas.append(ProcessArc(cp4=[p0,p1,p2,p3], # copy
                              process=process,
                              cut_at_start = cut_at_start,
                              cut_at_end = cut_at_end,
                              ))
    # Make cut points consistent:
    for i in range(len(pas)-1):
        first, second = pas[i], pas[i+1]
        cut_point = (first.cut_at_end or second.cut_at_start
                     or (first.process != second.process))
        first.cut_at_end = cut_point
        second.cut_at_start = cut_point
    if gs.closed:
        first, second = pas[-1], pas[0]
        cut_point = (first.cut_at_end or second.cut_at_start
                     or (first.process != second.process))
        first.cut_at_end = cut_point
        second.cut_at_start = cut_point
    if print_info:
        print("get_ProcessArcs returns:")
        for i,pa in enumerate(pas):
            print(str(i))
            print(str(pa))
        print('-------------------')
    return pas

#==============================================================
#             Bezier
#==============================================================

# ----------------
# collapse_handles
# ----------------
# Given gv:BCurve.GimpVectors, collapse to exactly 0:
# 1. either all handles, or
# 2. all handles below the threshold.
# Args:
# - gv:        BCurve.GimpVectors
# - threshold: float
# - collapse_all: boolean
# Returns:
# - gv: BCurve.GimpVectors
def collapse_handles(gv, threshold, collapse_all=False):
    new_stroke_list = []
    for gs in gv.stroke_list:
        new_cp_list = [gs.cp_list[0]] # head handle
        for i in range(1, len(gs.cp_list)-2, 3):
            p0,p1,p2,p3 = gs.cp_list[i:i+4]
            if collapse_all or (abs(p0-p1) < threshold):
                p1 = p0
            if collapse_all or (abs(p2-p3) < threshold):
                p2 = p3
            new_cp_list += [p0,p1,p2]
        new_cp_list += gs.cp_list[-2:] # last anchor and tail handle
        new_stroke_list.append(BCurve.GimpStroke(cp_list = new_cp_list,
                                                 closed = gs.closed))
    return BCurve.GimpVectors(stroke_list = new_stroke_list)


#==============================================================
#             G2-continuity of a stroke
#==============================================================

# --------------------
# G2_continuity_stroke
# --------------------
# Given a stroke as a list of ProcessArcs
# 1. subdivide it to sections to be processed and sections not to be processed;
# 2. call G2_continuity_anchorlist to process those that should be processed;
# 3. stitch the sections back together;
# 4. make a new stroke.
# Args:
# - image
# - process_arcs: [ProcessArc]
# - closed:       boolean
# - K_tuning:     callable (float->float)
# - L_param:      float
# - print_info:   boolean
# Returns:
# - BCurve.GimpStroke
def G2_continuity_stroke(image,
                         process_arcs,
                         closed,
                         K_tuning,
                         L_param,
                         print_info = False):
    ZERO = 1e-3
    if print_info:
        print("=======================")
        print("G2_continuity_stroke called with process_arcs: [ProcessArc]:")
        for i,pa in enumerate(process_arcs):
            print(str(i))
            print(str(pa))
        print("--------------------")
    class Section(object):
        """Section is a maximal continuous part of a stroke such that
        all its segments should be processed by the G2 algorithm, or
        all its segments should be left unprocessed by the G2 algorithm.
        It consists of
        - a list of objects of class ProcessArc, each having equal 'process' field
        - a boolean attribute: process or not
        - closed_NN_HH_NH_HN: string (one of 'closed','NN','HH','NH','HN')
              Meanings:
              'closed': use G2-algorithm for closed curves
              'NN': open curve, "Natural boundary condition" at both ends
              'HH': open curve, boundary condition "Pre-determined handle" at both ends
              'NH': Different boundary condition at start and end
              'HN': Different boundary condition at start and end
              The pre-determined handles are the following two:
        - target_start_handle 
        - target_end_handle
        - input_2_vectors (the two input vectors for G2-algorithm)
        """
        def __init__(self,
                     process_arc_list,
                     process=True,
                     closed_NN_HH_NH_HN=None,
                     target_start_handle=None,
                     target_end_handle=None,
                     input_2_vectors=None
                     ):
            self.process_arc_list = process_arc_list
            self.process = process
            self.closed_NN_HH_NH_HN = closed_NN_HH_NH_HN
            self.target_start_handle = target_start_handle
            self.target_end_handle = target_end_handle
            self.input_2_vectors = input_2_vectors
        def __str__(self):
            s = "Section: process_arc_list:"
            for i,pa in enumerate(self.process_arc_list):
                s += "\n  Arc "+str(i)+", control points:"
                for j,cp in enumerate(pa.cp4):
                        s += "\n      "+str(j)+":"+str(pa.process)+":"+str(cp)
            s += "\n    process = "+str(self.process)
            s += "\n    closed_NN_HH_NH_HN  = "+str(self.closed_NN_HH_NH_HN)
            s += "\n    target_start_handle = "+str(self.target_start_handle)
            s += "\n    target_end_handle   = "+str(self.target_end_handle)
            s += "\n    input_2_vectors   = "+str(self.input_2_vectors)
            return s
    # Step 1: Cut the list process_arcs in sections according the field
    # cut_at_end. (They should now be compatible with cut_at_start.)
    sections = []
    current_pas = [process_arcs[0]]
    current_process = process_arcs[0].process
    for next_pa in process_arcs[1:]:
        if not(current_pas[-1].cut_at_end): # Append next_pa to current_pas
            current_pas.append(next_pa)
            if next_pa.process != current_process:
                raise Exception("G2_continuity_stroke: Something wrong with process_arcs")
        else: # End this section, start new
            sections.append(Section(process_arc_list=current_pas,
                                    process=current_process))
            current_pas = [next_pa]
            current_process = next_pa.process
    sections.append(Section(process_arc_list=current_pas,
                            process=current_process))
    no_cuts = (len(sections) == 1)
    if print_info:
            print("\nStep 1: "+str(len(sections))+" sections: subdivide in sections:")
            for j,section in enumerate(sections):
                print("Section "+str(j))
                print(str(section))
    # Step 2: If closed, and there were cut(s), and
    # - the first and last section have the same 'process' field, and
    # - the anchor where the joining would happen is itself not a cut point,
    # join the first and last section to one section:
    joined_first_last = False
    if closed and not(no_cuts):
        first, last = sections[0], sections[-1]
        if (first.process == last.process) and not(last.process_arc_list[-1].cut_at_end):
            joined = Section(process_arc_list = last.process_arc_list+first.process_arc_list,
                             process = first.process)
            sections = [joined] + sections[1:-1]
            joined_first_last = True
        no_cuts = (len(sections) == 1) # Update this
    if print_info:
            print("\nStep 2: "+str(len(sections))+" sections, joined_first_last = "
                         +str(joined_first_last))
            for j,section in enumerate(sections):
                print("Section "+str(j))
                print(str(section))
    # Step 3: Update fields closed_NN_HH_NH_HN:
    for section in sections:
        if closed and no_cuts:
            if not section.process_arc_list[-1].cut_at_end:
                section.closed_NN_HH_NH_HN = 'closed'
                continue
            else: # The section must be processed as open
                pass
        # Assume: preserve_handle_threshold is already applied by collapsing short
        # handles, so handles to be preserved are those whose length exceeds ZERO.
        p0,p1,p2,p3 = section.process_arc_list[0].cp4
        q0,q1,q2,q3 = section.process_arc_list[-1].cp4
        if abs(p0-p1) < ZERO:
            case1 = 'N'
        else:
            case1 = 'H'
        if abs(q2-q3) < ZERO:
            case2 = 'N'
        else:
            case2 = 'H'
        section.closed_NN_HH_NH_HN = case1+case2
    if print_info:
            print("\nStep 3: "+str(len(sections))+" sections: update closed_NN_HH_NH_HN")
            for j,section in enumerate(sections):
                print("Section "+str(j))
                print(str(section))
    # Step 4: Update fields target_start_handle and target_end_handle:
    for section in sections:
        cp4list = [pa.cp4 for pa in section.process_arc_list]
        cp4_start = cp4list[0]
        cp4_end = cp4list[-1]
        if section.closed_NN_HH_NH_HN == 'closed':
            section.target_start_handle = None
            section.target_end_handle = None
        elif section.closed_NN_HH_NH_HN == 'NN':
            section.target_start_handle = None
            section.target_end_handle = None
        elif section.closed_NN_HH_NH_HN == 'HH':
            p0,p1,p2,p3 = cp4_start
            q0,q1,q2,q3 = cp4_end
            section.target_start_handle = -p0+p1
            section.target_end_handle = -q2+q3
        elif section.closed_NN_HH_NH_HN == 'NH':
            q0,q1,q2,q3 = cp4_end
            section.target_start_handle = None
            section.target_end_handle = -q2+q3
        elif section.closed_NN_HH_NH_HN == 'HN':
            p0,p1,p2,p3 = cp4_start
            section.target_start_handle = -p0+p1
            section.target_end_handle = None
        else:
            raise Exception("Something wrong")
    if print_info:
            print("\nStep 4: "+str(len(sections))+" sections: update target_start_handle, target_end_handle")
            for j,section in enumerate(sections):
                print("Section "+str(j))
                print(str(section))
    # Step 5: If two successive sections are xN-Hy or xH-Ny, make them xH-Hy and
    # update the relevant target handle using the neighbouring handle and K_tuning:
    # Note: This step causes that the path will be smoothed (G1-continuous)
    # at anchors where one handle is non-zero and the other is zero.
    # Collect left-right pairs of sections:
    if len(sections) > 1:
        left_right_pairs = [[sections[i], sections[i+1]]
                                            for i in range(len(sections)-1)]
        if closed:
            left_right_pairs.append([sections[-1], sections[0]])
    elif closed and no_cuts:
        left_right_pairs = [[sections[0],sections[0]]]
    else:
        left_right_pairs = []
    for left,right in left_right_pairs:
        if left.closed_NN_HH_NH_HN == 'closed': #closed stroke
            break
        left_case1, left_case2 = left.closed_NN_HH_NH_HN
        right_case1, right_case2 = right.closed_NN_HH_NH_HN
        if left_case2 == right_case1:                  # xH-Hy or xN-Ny
            continue
        left_cp4 = left.process_arc_list[-1].cp4
        right_cp4 = right.process_arc_list[0].cp4
        left_chord = abs(left_cp4[0] - left_cp4[-1])
        right_chord = abs(right_cp4[0] - right_cp4[-1])
        if left_case2 == 'N':                          # xN-Hy
            # Make target_end_handle for the left section
            K_factor = K_tuning(left_chord / right_chord)
            left.target_end_handle = K_factor * right.target_start_handle
            left.closed_NN_HH_NH_HN = left_case1+'H'   # change xN to xH
        elif right_case1 == 'N':                       # xH-Ny
            # Make target_start_handle for the right section
            K_factor = K_tuning(right_chord / left_chord)
            right.target_start_handle = K_factor * left.target_end_handle
            right.closed_NN_HH_NH_HN = 'H'+right_case2 # change Ny to Hy
        else:
            raise Exception("Something very wrong") ##############
    if print_info:
            print("\nStep 5: "+str(len(sections))+" sections: xN-Hy, xH-Ny")
            for j,section in enumerate(sections):
                print("Section "+str(j))
                print(str(section))
    # Step 6: Update field input_2_vectors:
    for section in sections:
            section.input_2_vectors = [section.target_start_handle, section.target_end_handle]
    if print_info:
            print("\nStep 6: "+str(len(sections))+" sections: update target_start_handle, target_end_handle")
            for j,section in enumerate(sections):
                print("Section "+str(j))
                print(str(section))
    # Step 7: Do the G2 processing:
    single_closed_section = (section.closed_NN_HH_NH_HN == 'closed')
    new_gs_list = []
    for section in sections:
        cp4list = [pa.cp4 for pa in section.process_arc_list]
        if section.process and (len(cp4list) > 1): # Process with the G2 algorithm
            anchorlist = [cp4[0] for cp4 in cp4list] + [cp4list[-1][-1]]
            new_gs = G2_continuity_anchorlist(anchorlist, # Main call
                                              single_closed_section,
                                              K_tuning,
                                              L_param,
                                              section.closed_NN_HH_NH_HN,
                                              section.input_2_vectors)
        else: # Do not process; copy the relevant parts from the original stroke:
            cplist = [cp4list[0][0]]
            for cp4 in cp4list:
                cplist += cp4[:-1]
            cplist += [cp4list[-1][-1]]*2
            new_gs = BCurve.GimpStroke(cp_list=cplist, closed=single_closed_section)
        new_gs_list.append(new_gs)
    if print_info:
        print("\nStep 7: new_gs_list:")
        for i,gs in enumerate(new_gs_list): 
            print("gs "+str(i))
            print(str(gs))
    # Step 8: stitch the sections of each newly created stroke together:
    new_cp_list = []
    for i,gs in enumerate(new_gs_list):
        if i == 0:
            new_cp_list += gs.cp_list[:-1]
        else:
            new_cp_list += gs.cp_list[2:-1]
    new_cp_list.append(new_gs_list[-1].cp_list[-1])
    if print_info:
        print("\nStep 8: new_cp_list: "+str(len(new_cp_list)))
        for cp in new_cp_list: 
            print(str(cp))
    if print_info:
        print("=======================")
    return BCurve.GimpStroke(cp_list=new_cp_list, closed=closed)

# ------------------------
# G2_continuity_anchorlist
# ------------------------
# From a list of anchors, construct a G2-continuous BCurve.GimpStroke
# (a composite Bezier curve).
# This routine contains the mathematics of the work.
# Args:
# - anchors:            [complex]
# - closed:             boolean
# - K_tuning:           callable (float->float)
# - L_param:            float
# - closed_NN_HH_NH_HN: string (one of 'closed', 'NN', 'HH', 'NH', 'HN')
# - input_2_vectors:    [complex,complex] (for boundary conditions in some cases)
# Returns:
# - BCurve.GimpStroke
def G2_continuity_anchorlist(anchors,
                             closed,
                             K_tuning,
                             L_param,
                             closed_NN_HH_NH_HN,
                             input_2_vectors):
    A = anchors
    n = len(A)-1 # A[0] .. A[n]
    if closed:
        a = [abs(-A[i]+A[i+1]) for i in range(n-1)] + [abs(-A[n-1]+A[0])]
        k = [K_tuning(a[n-1]/a[0])] + [K_tuning(a[i-1]/a[i]) for i in range(1,n)]
        A_tilde = [-A[i]+A[i+1] for i in range(n-1)] + [-A[n-1]+A[0]]
    else:
        a = [abs(-A[i]+A[i+1]) for i in range(n)]
        k = [0] + [K_tuning(a[i-1]/a[i]) for i in range(1,n)] # 0 is placeholder, no effect
        A_tilde = [-A[i]+A[i+1] for i in range(n)]
    # Build the matrix equation:
    # The matrix M and the column vector Y of the closed case 
    # serve as the prototype for the other cases, so we build them first:
    Y = ([A_tilde[n-1] + (k[0]**2)*A_tilde[0]]
         + [A_tilde[i-1] + (k[i]**2)*A_tilde[i] for i in range(1,n)]
        )
    L1 = 2 + L_param/2
    N = [L1*ki + L1*(ki**2) for ki in k]
    matrix = [] # Build matrix M row by row
    matrix.append([N[0], (k[0]**2)*k[1]] + [0]*(n-3) + [1])     # row 0
    for i in range(1,n-1):                                      # rows 1,...,n-2
        matrix.append([0]*(i-1)
                    +[1, N[i], (k[i]**2)*k[i+1]]
                    +[0]*(n-i-2)
                    )
    matrix.append([(k[n-1]**2)*k[0]] + [0]*(n-3) + [1, N[n-1]]) # row n-1
    if closed: # Closed stroke: matrix equation already ok
        pass
    else: # Open stroke: matrix equation a little different
        if closed_NN_HH_NH_HN == 'NN':
            matrix[0] = [2, k[1]] + [0]*(n-2)
            matrix[n-1] = [0]*(n-2) + [1, N[n-1] - (1./2)*(k[n-1]**2)]
            Y[0] = A_tilde[0]
            Y[n-1] = A_tilde[n-2] + (1./2)*(k[n-1]**2)*A_tilde[n-1]
        elif closed_NN_HH_NH_HN == 'HH':
            X0, Ylast = input_2_vectors
            matrix[0] = [1] + [0]*(n-1)
            matrix[n-1] = [0]*(n-2) + [1, N[n-1]]
            Y[0] = X0
            Y[n-1] = A_tilde[n-2] + (k[n-1]**2)*A_tilde[n-1] - (k[n-1]**2)*Ylast
        elif closed_NN_HH_NH_HN == 'NH':
            _, Ylast = input_2_vectors
            matrix[0] = [2, k[1]] + [0]*(n-2)
            matrix[n-1] = [0]*(n-2) + [1, N[n-1]]
            Y[0] = A_tilde[0]
            Y[n-1] = A_tilde[n-2] + (k[n-1]**2)*A_tilde[n-1] - (k[n-1]**2)*Ylast
        elif closed_NN_HH_NH_HN == 'HN':
            X0,_ = input_2_vectors
            matrix[0] = [1] + [0]*(n-1)
            matrix[n-1] = [0]*(n-2) + [1, N[n-1] - (1./2)*(k[n-1]**2)]
            Y[0] = X0
            Y[n-1] = A_tilde[n-2] + (1./2)*(k[n-1]**2)*A_tilde[n-1]
        else:
            raise Exception("Unknown closed_NN_HH_NH_HN: "+closed_NN_HH_NH_HN)
    #print("Y:")
    #for Yi in Y:
    #    print(str(Yi))
    #print("Matrix M:")
    #for row in matrix:
    #    print(str(row))
    # Solve:
    try:
        inverse_matrix = inverse(matrix) # Inverse matrix
    except SingularMatrixError as e:
        #print(str(e))
        #print("Anchors: "+str(anchors))
        #print("Matrix:")
        #for row in matrix:
        #    print(str(row))
        raise SingularMatrixError(e.message)
    # pre_x is the solution of MX = Y:
    pre_x = [sum(inverse_matrix[i][j]*Y[j] for j in range(n)) for i in range(n)]
    if closed:
        x = pre_x # [x0,x1,x2,...,x(n-1)]
        y = [k[i+1]*x[i+1] for i in range(n-1)] + [k[0]*x[0]]
    else: # Open stroke: connection between x's and y's a little different
        if closed_NN_HH_NH_HN == 'NN':
            x = pre_x # [x0,x1,x2,...,x(n-1)]
            y = [k[i+1]*x[i+1] for i in range(n-1)] + [(1./2)*(A_tilde[n-1] - x[n-1])]
        elif closed_NN_HH_NH_HN == 'HH':
            x = pre_x # = [x0,x1,x2,...,x(n-1)]
            y = [k[i+1]*x[i+1] for i in range(n-1)] + [Ylast]
        elif closed_NN_HH_NH_HN == 'NH':
            x = pre_x # [x0,x1,x2,...,x(n-1)]
            y = [k[i+1]*x[i+1] for i in range(n-1)] + [Ylast]
        if closed_NN_HH_NH_HN == 'HN':
            x = pre_x # [x0,x1,x2,...,x(n-1)]
            y = [k[i+1]*x[i+1] for i in range(n-1)] + [(1./2)*(A_tilde[n-1] - x[n-1])]
    # Make the stroke:
    if closed: 
        cp_list = [A[0]-y[n-1]]
        for i in range(n-1):
            Ai = A[i]
            Bi = A[i+1]
            xi = x[i]
            yi = y[i]
            cp_list += [Ai, Ai + xi, Bi - yi]
        cp_list += [A[n-1], A[n-1]+x[n-1]]
    else:
        cp_list = [A[0]]
        for i in range(n):
            Ai = A[i]
            Bi = A[i+1]
            xi = x[i]
            yi = y[i]
            cp_list += [Ai, Ai + xi, Bi - yi]
        cp_list += [A[-1], A[-1]]
    return BCurve.GimpStroke(cp_list = cp_list, closed = closed)


#==============================================================
#             Main
#==============================================================

selection_options = [ # (description, identifier)
        ('Ignore any selection',                            'ignore'),
        ('Restrict the effect to segments with both ends inside the selection',     'inside'),
        ('Restrict the effect to segments with both ends outside of the selection', 'outside'),
        ]

def G2_continuity_main(image,
                       path,
                       K_tuning_parameter,
                       L_tuning_parameter,
                       force_C2,
                       preserve_nonzero_handles, # boolean
                       preserve_handle_threshold, # float
                       selection_option,
                       ):
    if force_C2: # C2-continuity: K=1, L=0
        def K_tuning(x):
            return 1 # Different edge lengths not taken into account in any way.
        L_param = 0
    else: # More general G2-continuity
        # The function 'K_tuning' gives a one-parameter tuning option.
        # It will be called as 'tuning(a/b)' where a and b are lengths of
        # adjacent edges in the input path, hence the effect is to take into
        # account the different lengths of adjacent edges.
        # The following is a simple choice. Any function should give G2-continuity.
        def K_tuning(x):
            return pow(x, K_tuning_parameter/4)     # ad hoc
        # The parameter 'L_param' determines how strongly the path will bend around
        # each anchor.
        L_param = (exp(L_tuning_parameter/4) - 1)*3 # ad hoc
    
    selection_case = selection_options[selection_option][1]
    gv = vectors_object2gv(path)
    if preserve_nonzero_handles: # Collapse all handles below the threshold
        gv = collapse_handles(gv, preserve_handle_threshold)
    else:                        # Collapse all handles
        gv = collapse_handles(gv, preserve_handle_threshold, collapse_all=True)
    new_gslist = []
    for i,gs in enumerate(gv.stroke_list):
        if len(gs.cp_list) < 6:
            new_gslist.append(deepcopy(gs))
            continue
        else:
            gs_work = gs
        process_arcs = get_ProcessArcs(image,
                                       gs_work,
                                       selection_case,
                                       preserve_nonzero_handles
                                       )
        # Main call
        new_gs = G2_continuity_stroke(image,
                                      process_arcs,
                                      gs_work.closed,
                                      K_tuning,
                                      L_param,
                                      #preserve_handle_threshold
                                      )
        new_gslist.append(new_gs)
    new_gv = BCurve.GimpVectors(stroke_list = new_gslist)
    new_path = new_gv.gv2vectors_object(image)
    if force_C2:
        new_path.name = path.name+"|C2"
    else:
        new_path.name = path.name+"|G2"+str((K_tuning_parameter,L_tuning_parameter))
    # Draw
    pdb.gimp_image_undo_group_start(image)
    gimp_draw_vectors_object(image, new_path, visible=True)
    pdb.gimp_image_undo_group_end(image)
    return new_path

#======================================================
#                    Registrations
#======================================================

versionnumber = "0.8"
procedure_author = "Markku Koppinen"
procedure_copyright = procedure_author
procedure_date = "2022"
image_types = "*"
menupath = '<Vectors>/Tools/Modify path'

procedure_name  = "G2_continuity"
procedure_blurb = ("A G2-continuous path is generated"
                   +"\nthrough the anchors of the input path."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = ("Given a path, a G2-continuous path is generated "
                   +"through its anchors. Optionally C2-continuity can be forced."
                   +"\n- Two tuning parameters are provided, both rather ad hoc:"
                   +"\nThe first adjusts how strongly differences in distances "
                   +"between successive anchors are taken into account; the second "
                   +"adjusts how tightly the path will bend at anchors."
                   +"\n- The plugin allows preserving handles whose length exceeds "
                   +"some given threshold. This enables the user to preserve shapes"
                   +"\nof sharp corners. The idea is that"
                   +"\n(1) the purpose is to smooth the path almost everywhere; but"
                   +"\n(2) at some anchors a sharp corner and its shape are to be preserved."
                   +"\nThis is achieved by shaping such corners with long handles (over the "
                   +"threshold) which the plugin then preserves, and having all other handles zero length "
                   +"or very short (below the threshold). "
                   +"The threshold has also the useful consequence that very short handles will be ignored "
                   +"which may inadvertently be present in the path."
                   +"\n- At anchors where both handles are preserved the path will be "
                   +"continuous; at anchors where one of the handles is preserved the path will be "
                   +"G1-continuous; everywhere else it will be G2-continuous."
                   +"\n- The plugin allows to restrict the effect by means of a selection; "
                   +"however, to have any effect, at least three consecutive anchors are needed."
                   )

procedure_label = "G2-continuity"

procedure_function = G2_continuity_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path", "The input path", None),
      (PF_FLOAT, "K_tuning_parameter",
                 ("First tuning parameter (recommendation: keep in -2..3)"
                  +"\nAdjusts how strongly segment lengths are taken into account."
                  ),
                  1),
      (PF_FLOAT, "L_tuning_parameter",
                 ("Second tuning parameter (recommendation: keep in -3..4)"
                  +"\nAdjusts how tightly the path will bend at anchors."
                  ),
                  0),
      (PF_BOOL, "force_C2", 
                   ('Force C2-continuity?'
                  +"\nIf 'Yes', the two tuning parameters will have no effect."
                  ),
                  False),
      (PF_BOOL, "preserve_nonzero_handles", 
                   (
                   "Preserve long handles (overriding G2-continuity locally)?"
                   "\nIf 'Yes', set the threshold for \"long\" handles below."
                  ),
                  False),
      (PF_FLOAT, "preserve_handle_threshold",
                   ("The threshold for \"long\" handles (pixels)"
                  #+"\n(Use this to avoid very short handles that were perhaps made accidentally!)"
                   ),
                  5.0),
      (PF_OPTION, 'selection_option','How to use the selection?',
                   0,
                  [case[0] for case in selection_options]), # Descriptions of cases
    ],
    [
        (PF_VECTORS, "new_path", "G2-continuous path"),
    ],
    procedure_function,
    menu=menupath)

main()
