#!/usr/bin/env python

# Experimental Gimp plugin to simplify paths
#
# History:
# v0.1: 2021-28-04: Experimental plugin "Simplify"
# v0.2: 2021-01-05: Developing
# v0.3: 2021-04-05: Developing
# v0.4: 2021-09-05: Developing
# v0.5: 2021-10-05: Developing
# v0.6: 2021-12-05: Developing
# v0.7: 2021-14-05: First published version
# v0.8: 2021-19-05: Plugin made faster
# v0.9: 2021-31-05: Plugin made faster
# v0.10: 2021-27-06: Bug fix (using selection + anchor outside of selection)
# v0.11: 2021-28-06: Better wording in the GUI.

# (c) Markku Koppinen 2020, 2021
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published
#   by the Free Software Foundation; either version 3 of the License, or
#   (at your option) any later version.
#
#   This very file is the complete source code to the program.
#
#   If you make and redistribute changes to this code, please mark it
#   in reasonable ways as different from the original version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   The GPL v3 licence is available at: https://www.gnu.org/licenses/gpl-3..en.html'

from __future__ import division, print_function
from gimpfu import *
from math import *
from copy import deepcopy

#==============================================================
#             class BCurve       
#==============================================================

# Note: The plane is treated as the complex plane.

class BCurve(object):
    """Data structures to handle Gimp vectors objects by means of individual
    Bezier arcs.
    By a "Bezier arc" we mean here a Bezier curve determined by 4 control points
    p0,p1,p2,p3.
    This class contains data structures needed to do computations on
    a Bezier curve which are done arc by arc.
    To this end, Gimp data structures
    - gimp.Vectors
    - gimp.VectorsBezierStroke
    are here represented as data structures (classes)
    - GimpVectors
    - GimpStroke
    - BezierCurve
    - BezierArc
    - ControlPoint = complex
    with the aim that
    - each control point [x,y] is represented as an instance of ControlPoint;
    - each Bezier arc (four control points, see above) is represented as an instance
      of BezierArc.
    """
    
    ControlPoint = complex
    
    class GimpVectors(object):
        """Essentially same data as in a gimp.Vectors object except that
        strokes are instances of GimpStroke rather than
        gimp.VectorsBezierStroke.
        Attributes:
        - stroke_list: [GimpStroke]
        - name:        string
        """
        def __init__(self,
                     stroke_list,            # [GimpStroke]
                     name='GimpVectors'      # string
                     ):
            self.stroke_list = stroke_list
            self.name = name
        def __str__(self):
            s = 'GimpVectors '+self.name
            count = 0
            for stroke in self.stroke_list:
                s += '\n  Stroke '+str(count)+': GimpStroke'
                s += '\n    '+str(stroke)
                count += 1
            return s
        def gv2vectors_object(self, image, name=None):
            """Conversion GimpVectors -> gimp.Vectors
            (The converse: vectors_object2gv)
            """
            def cp_list2xy_list(cp_list):
                xy = []
                for cp in cp_list:
                    xy += [cp.real, cp.imag]
                return xy
            if name is None:
                name = self.name
            vectors = pdb.gimp_vectors_new(image, name)
            for stroke in self.stroke_list:
                closed = stroke.closed
                xy_list = cp_list2xy_list(stroke.cp_list)
                stroke_id = pdb.gimp_vectors_stroke_new_from_points(
                             vectors, 0, len(xy_list), xy_list, closed)
            return vectors
    
    class GimpStroke(object):
        """Essentially same data as in a gimp.VectorsBezierStroke.points,
        except that control points list [x,y,x,y,...] is arranged as
        a list [ControlPoint] by joining successive x,y.
        Attributes:
        - cp_list:     [ControlPoint]
        - closed:      boolean
        - stroke_name: string
        """
        def __init__(self,
                     cp_list,                   # [ControlPoint]
                     closed,                    # boolean
                     stroke_name = 'GimpStroke' # string
                     ):
            self.stroke_name = stroke_name
            self.cp_list = cp_list
            self.closed = closed
        def __str__(self):
            s = self.stroke_name + '; Control points:'
            for cp in self.cp_list:
                s += '\n    '+str(cp)
            s += '\n    closed: '+str(self.closed)
            return s
        def gs2bc(self):
            """Conversion GimpStroke -> BezierCurve
            """
            head = self.cp_list[0]
            tail = self.cp_list[-1]
            inner = self.cp_list[1:-1]
            if len(inner) > 1: # Must be divisible by 4
                ba_list = []
                count = 0
                for i in range(0,len(inner)-1,3):
                    ba_list.append(BCurve.BezierArc(inner[i:i+4], 'arc '+str(count)))
                    count += 1
            elif len(inner) == 1: # Stroke has only one anchor
                ba_list = [BCurve.BezierArc([inner[0]], 'arc 0')]
            else:
                raise Exception("BCurve.GimpStroke.gs2bc: No anchors in stroke?")
            return BCurve.BezierCurve(
                               bezier_arcs = ba_list,
                               head_handle = head,
                               tail_handle = tail,
                               closed = self.closed,
                               curve_name = self.stroke_name
                               )
    
    class BezierCurve(object):
        """BezierCurve is a list of butting Bezier arcs with some extra data.
        Attributes:
        - curve_name:  string
        - bezier_arcs: [BezierArc];
        - head_handle: ControlPoint;
        - tail_handle: ControlPoint;
        - closed:      boolean.
        Note: "Butting" means that for any pair of successive arcs, the last
              control point of the former equals the first control point of
              the latter.
              In initialization no checks are done about this condition.
        """
        def __init__(self,
                     bezier_arcs=[],           # [BezierArc]
                     head_handle=None,         # ControlPoint
                     tail_handle=None,         # ControlPoint
                     closed=False,             # boolean
                     curve_name='BezierCurve', # string
                     ):
            self.curve_name = curve_name
            self.bezier_arcs = bezier_arcs
            if head_handle is None:
                self.head_handle = bezier_arcs[0].cp4[0]
            else:
                self.head_handle = head_handle
            if tail_handle is None:
                self.tail_handle = bezier_arcs[-1].cp4[-1]
            else:
                self.tail_handle = tail_handle
            self.closed = closed
        def __str__(self):
            s = self.curve_name + ': Bezier curve' + '; Bezier arcs:'
            count = 0
            for arc in self.bezier_arcs:
                s += '\n  arc '+ str(count)+ ': ' +str(arc)
                count += 1
            s += '\n  Head handle:    '+str(self.head_handle)
            s += '\n  Tail handle:    '+str(self.tail_handle)
            s += '\n  Closed:'    +str(self.closed)
            return s
        def bc2gs(self):
            """Conversion BezierCurve -> GimpStroke
            """
            cp_list = [self.head_handle]
            if len(self.bezier_arcs[0].cp4) > 1:
                for ba in self.bezier_arcs:
                    cp_list += ba.cp4[:3]
                cp_list.append(self.bezier_arcs[-1].cp4[-1])
            else:
                # Only one BezierArc and it has cp4 = [p0]
                # (may rise from Gimp stroke with only one anchor).
                cp_list.append(self.bezier_arcs[0].cp4[0])
            cp_list.append(self.tail_handle)
            return BCurve.GimpStroke(cp_list, self.closed, self.curve_name)
        def bc2vectors_object(self, image, name=None):
            """Conversion BezierCurve -> gimp.Vectors
            Makes vectors_object with 1 stroke.
            """
            if name is None:
                name = self.curve_name
            gs = self.bc2gs()             # BCurve.GimpStroke
            gv = BCurve.GimpVectors([gs]) # BCurve.GimpVectors
            return gv.gv2vectors_object(image, name)
    
    class BezierArc(object):
        """Data structure for one Bezier arc: arc determined by 4 control
           points.
        Attributes:
        - cp4:      [p0,p1,p2,p3] where each pi:ControlPoint, or
                    [p0] if only one control point (may rise from
                    a stroke with only one anchor);
        - arc_name: string
        """
        def __init__(self,
                     cp4,                 # cp4=[p0,p1,p2,p3] with pi:ControlPoint
                                          # or [p0]
                     arc_name='BezierArc' # string
                     ):
            self.cp4 = cp4
            self.arc_name = arc_name
        def reverse(self):
            self.cp4.reverse()
        def __str__(self):
            s = 'BezierArc, control points:'
            for cp in self.cp4:
                s += '\n  '+ str(cp)
            return s

# -----------------
# vectors_object2gv
# -----------------
# Conversion gimp.Vectors -> GimpVectors
# Args:
# - vectors_object: gimp.Vectors (Gimp vectors object)
# Returns:
# - GimpVectors
#
# Note: The inverse conversion is GimpVectors.gv2vectors_object(...).
#       See also BezierCurve.bc2vectors_object(...).
def vectors_object2gv(vectors_object):
    def xy_list2cp_list(xy_list): # xy_list = [x,y,x,y,...,]
        cp = []
        for i in range(0,len(xy_list),2):
            cp.append(BCurve.ControlPoint(xy_list[i], xy_list[i+1]))
        return cp
    stroke_list = []
    count = 0
    for stroke in vectors_object.strokes:
        xy_list = stroke.points[0]
        closed = stroke.points[1]
        cp_list = xy_list2cp_list(xy_list)
        stroke_name = vectors_object.name + ' stroke '+str(count)
        stroke_list.append(BCurve.GimpStroke(cp_list, closed, stroke_name))
        count += 1
    return BCurve.GimpVectors(stroke_list, vectors_object.name)


#======================================================
#                 Zeroes of a polynomial
#======================================================

# A function to find zeroes of a polynomial, and as by-products the
# x-values of the extrema and of the inflection points.
# At the final phase the zeroes are found by binary search from monotonous
# segments.
# 
# Written just for fun. Other algorithms exist in abundance.
#
# In addition, as a simple wrap-up, a function to do the same job when the
# polynomial is of degree 3 and given in the Bernstein basis.

# --------------------
# zeroes_of_polynomial
# --------------------
# Given a real polynomial, find the real zeroes, the x values of extrema,
# and the x values of the inflection points. May restrict to an interval.
#
# Args:
# - coefficients: list [c0,c1,...,cn] of the coefficients of the
#                     polynomial c0 + c1*x + ... + cn*(x**n);
#                     the ci are integer or float.
# - interval:     None or [float,float]
# Returns:
# 1. None in the case of a zero polynomial.
# 2. None if interval is given (not None) with interval=[a,b] where a>b.
# 3. If interval=[a,a] is given, if a is a root then [a],[],[] is returned,
#    otherwise [],[],[]. So, no effort is made to give any extrema or inflection
#    points.
# 3. Otherwise:
#    - the list of real zeroes of the polynomial, in increasing order
#      (possibly []) with multiple roots repeated;
#    - the extrema (the x values);
#    - the inflection points (the x values).
# 4. Because of the chosen accuracy and tolerance in the code, choosing different
#    interval may change the returned values slightly.
def zeroes_of_polynomial(coefficients, interval=None):
    from math import sqrt
    from functools import partial
    def f(x, coeff): # Horner
        s = 0.
        for ci in coeff[::-1]:
            s = ci + x*s
        return s
    if not (interval is None):
        LO,HI = interval
        if LO > HI:
            return None
        if LO == HI:
            if f(LO) == 0:
                return [LO],[],[]
            else:
                return [],[],[]
    coefficients = [float(ci) for ci in coefficients]
    # Find largest n with coefficients[n] non-zero and
    # discard all higher terms:
    n = -1
    for i in range(len(coefficients)):
        if coefficients[i] != 0.:
            n = i
    if n < 0: # Polynomial is identically zero.
        return None
    c = coefficients[:n+1] 
    # Make c[n] positive:
    if c[n] < 0:
        c = [-ci for ci in c]
    if n <= 2:
        if n == 0: # Polynomial is a non-zero constant.
            results = [[],[],[]]
        if n == 1: # Polynomial is c[0] + c[1]*x
            results = [[-c[0] / c[1]], [], []]
        if n == 2: # Polynomial is c[0] + c[1]*x + c[2]*(x**2).
            discr = c[1]**2 - 4*c[0]*c[2]
            try:
                root = sqrt(discr)
                x1 = (-c[1] - root) / (2*c[2])
                x2 = (-c[1] + root) / (2*c[2])
                #return [x1,x2], [-c[1]/(2*c[2])], []
                results = [[x1,x2], [-c[1]/(2*c[2])], []]
            except ValueError:
                #return [], [-c[1]/(2*c[2])], []
                results = [[], [-c[1]/(2*c[2])], []]
        if interval is None:
            return results
        else:
            results0 = [t for t in results[0] if LO <= t <= HI]
            results1 = [t for t in results[1] if LO <= t <= HI]
            results2 = [t for t in results[2] if LO <= t <= HI]
            return results0, results1, results2
    # Build subdivision such that in each subinterval
    # the polynomial is monotonous:
    derivative = [e*i for e,i in enumerate(c)][1:]
    extrema, inflection, _ = zeroes_of_polynomial(derivative, interval)
    if interval is None:
        x_far = sum([abs(ci/c[-1]) for ci in c])
        subdiv = [-x_far] + extrema + [x_far]
    else:
        subdiv = [LO] + [t for t in extrema if LO < t < HI] + [HI]
    # Solve and collect the zeroes of each monotonous segment:
    fc = partial(f, coeff=c)
    zeroes = []
    for i in range(len(subdiv)-1):
        lo, hi = subdiv[i], subdiv[i+1]
        x = zero_of_monotonous_function(fc, lo, hi,
                                    accuracy=1e-10, tolerance=1e-10)
        if not(x is None):
            zeroes.append(x)
    return zeroes, extrema, inflection

# ---------------------------
# zero_of_monotonous_function
# ---------------------------
# Find the zero, if any, of a monotonous function f(x)
# in the interval [lo,hi]. If none is found, None is returned.
# Args:
# - f, lo, hi: the monotonous function with start and end value;
# - accuracy:  accuracy for x.
# - tolerance: float>=0; if no zeroes exist in the interval, then
#              if f(lo) or f(hi) is less than 'tolerance' from 0,
#              then lo or hi is returned, respectively.
# Returns: the zero or None.
def zero_of_monotonous_function(f, lo, hi, accuracy=1e-10, tolerance=1e-10):
    MAX_ROUNDS = 50
    lo, hi = float(lo), float(hi)
    flo, fhi = float(f(lo)), float(f(hi))
    if (flo > 0.) == (fhi > 0.):
        if abs(flo) <= tolerance:
            return lo
        elif abs(fhi) <= tolerance:
            return hi
        else:
            return None
    if flo == 0.:
        return lo
    if fhi == 0.:
        return hi
    count = 0
    while hi-lo > accuracy:
        mid = (lo+hi)/2.
        fmid = float(f(mid))
        if fmid == 0.:
            return mid
        if (flo > 0.) == (fmid > 0.):
            lo, flo = mid, fmid
        else:
            hi, fhi = mid, fmid
        count += 1
        if count > MAX_ROUNDS:
            break
    x = (fhi*lo - flo*hi) / (fhi-flo)
    return x

# -------------------------
# solve_bernstein2_equation
# -------------------------
# Find real zeroes of the equation c0*b0(t) + c1*b1(t) + c2*b2(t) = 0 where
# b0,b1,b2 are the Bernstein polynomials of degree 2 and
# c0,c1,c2 are real coefficients.
# Return the zeroes as a list (0, 1, or 2 items).
# If the function is constant 0, return None.
# Args:
# - c0,c1,c2: float
# Returns:
# - None or list of floats (possibly empty)
def solve_bernstein2_equation(c0,c1,c2):
    from math import sqrt
    ZERO = 1e-14
    if abs(c0 - 2.*c1 + c2) < ZERO: # equation of degree <= 1
        try:
            return [c0 / (2.*(c0 - c1))]
        except ZeroDivisionError: # equation of degree 0
            if abs(c0) < ZERO: # constant 0
                return None
            else: # constant not 0
                return []
    try:
        root = sqrt(c1**2 - c0*c2)
    except ValueError: # No real roots
        return []
    return [(c0 - c1 + root) / (c0 - 2.*c1 + c2),
            (c0 - c1 - root) / (c0 - 2.*c1 + c2)]


#==============================================================
#                Closest point on a Bezier curve
#==============================================================
# --------------------
# bezier_closest_point
# --------------------
# Find closest point (parameter value t and the point B(t)) to a given v on
# the Bezier curve arc B(t), 0<=t<=1. Control points are cp4.
# Args:
# - cp4: [complex,complex,complex,complex] (control points)
# - v:   complex
# - level: integer (working level: higher number, harder work)
# Returns:
# - [float,complex] (=[t,B(t)])
def bezier_closest_point(cp4, v, level):#, restrict01=True):
    accuracy = 1e-8
    linear_approx = True # Linear approximation
    if level <= 2:
        guesses = [1/3, 2/3]
    elif level <= 5:
        guesses = [1/4, 2/4, 3/4]
    elif level <= 8:
        guesses = [1/6, 2/6, 3/6, 4/6, 5/6]
    else:
        guesses = [1/8, 2/8, 3/8, 4/8, 5/8, 6/8, 7/8]
        linear_approx = False # Quadratic approximation
    candidates = [[0,cp4[0]], [1,cp4[-1]]] # [B(t),t]: First, store end points.
    for guess in guesses: # Second, points in open interval (tangent condition)
        t = guess
        Bv = bezier_rv(t,cp4) - v
        for tries in range(20):
            Bd = bezier_dot_rv(t,cp4)
            Bdd = bezier_dot_dot_rv(t,cp4)
            b = (Bv.real * Bdd.real + Bv.imag * Bdd.imag     # b = (B-v).B'' + B'.B'
                 + Bd.real * Bd.real + Bd.imag * Bd.imag)
            c = Bv.real * Bd.real + Bv.imag * Bd.imag        # c = (B-v).B'
            if not linear_approx: # Quadratic: solve a(dt)^2 + b(dt) + c = 0
                try:
                    a = Bd.real * Bdd.real + Bd.imag * Bdd.imag  # a = B'.B''
                    inv_2a = 1/(2*a)
                    dt = inv_2a * (-b + sqrt(b*b - 4*a*c))
                except ZeroDivisionError:
                    linear_approx = True # Try linear
                except ValueError:
                    linear_approx = True # Try linear
            if linear_approx:     # Linear: solve b(dt) + c = 0
                try:
                    dt = -c / b
                except ZeroDivisionError:
                    break
            tdt = t+dt
            if tdt < 0: tdt = 0 # Restrict the parameter to [0,1]
            elif tdt > 1: tdt = 1
            if abs(t - tdt) < accuracy:
                t = (t + tdt)/2
                candidates.append([t, bezier_rv(t,cp4)])
                break
            else:
                t = tdt
                Bv = bezier_rv(t,cp4) - v
        else:
            candidates.append([t, Bv])
    return min(candidates, key=(lambda x: abs(x[1]-v)))

# -------------------------------
# bezier_critical_point_candidates
# -------------------------------
# Candidates for closest or farthest points:
# Solutions to (B(t)-v).B'(t) = 0.
# Args:
# - cp4: [complex,complex,complex,complex] (control points)
# - v:   complex
# Returns:
# - [[float,complex]] (=[[t,B(t)]])
def bezier_critical_point_candidates(cp4, v, interval=None):
    p0,p1,p2,p3 = cp4
    # Control points w0,w1,w2,w3 when the basis {1, t, t^2/2, t^3/6}
    # is used instead of the Bernstein polynomials:
    w0 = p0
    w1 = 3*(-p0 + p1)
    w2 = 6*(p0 - 2*p1 + p2)
    w3 = 6*(-p0 + 3*p1 - 3*p2 + p3)
    # Dot products:
    w0w1 = w0.real * w1.real + w0.imag * w1.imag
    w0w2 = w0.real * w2.real + w0.imag * w2.imag
    w0w3 = w0.real * w3.real + w0.imag * w3.imag
    w1w1 = w1.real * w1.real + w1.imag * w1.imag
    w1w2 = w1.real * w2.real + w1.imag * w2.imag
    w1w3 = w1.real * w3.real + w1.imag * w3.imag
    w2w2 = w2.real * w2.real + w2.imag * w2.imag
    w2w3 = w2.real * w3.real + w2.imag * w3.imag
    w3w3 = w3.real * w3.real + w3.imag * w3.imag
    #
    vw1 = v.real * w1.real + v.imag * w1.imag
    vw2 = v.real * w2.real + v.imag * w2.imag
    vw3 = v.real * w3.real + v.imag * w3.imag
    # Polynomial coefficients:
    c0 = w0w1 - vw1
    c1 = w0w2 + w1w1 - vw2
    c2 = w0w3/2 + 3*w1w2/2 - vw3/2
    c3 = 2*w1w3/3 + w2w2/2
    c4 = 5*w2w3/12
    c5 = w3w3/12
    # Solve polynomial zeroes:
    zeroes = zeroes_of_polynomial([c0,c1,c2,c3,c4,c5], interval)
    if zeroes is None:
        return []
    ts = zeroes[0]
    # Compute B(t)'s:
    t_Bs = [[t, w0 + t*w1 + t*t*w2/2 + t*t*t*w3/6] for t in ts]
    return t_Bs

#==============================================================
#                         Affine map
#==============================================================
# Note: The plane is treated as the complex plane: #

class MakeAffineMapError(Exception):
    def __init__(self,message):
        self.message = message
    def __str__(self):
        return self.message

# ---------------
# make_affine_map
# ---------------
# Affine map sending three given points u0,u1,u2 to another
# three given points v0,v1,v2. 
# Args:
# - base:   [complex,complex,complex] (at least 3 points)
# - target: [complex,complex,complex] (at least 3 points)
# Returns:
# - callable (complex -> complex)
def make_affine_map(base, target):
    def A_inv_B(A,B): # Two 2x2 matrices. Compute A^(-1)*B
        a,b,c,d = A
        x,y,z,u = B
        try:
            idet = 1./(a*d-b*c)
        except ZeroDivisionError:
            raise Exception("make_affine_map: Zero determinant")
        C = [d*x-b*z, d*y-b*u,
            -c*x+a*z, -c*y+a*u]
        return [idet*e for e in C]
    u0,u1,u2 = base[:3] # The first three anchors: complex
    v0,v1,v2 = target[:3] # The first three anchors: complex
    if collinear([u0,u1,u2]):
        raise MakeAffineMapError("make_affine_map: Anchors of Base are collinear.")
    u10 = u1-u0
    u20 = u2-u0
    v10 = v1-v0
    v20 = v2-v0
    A = [u10.real, u10.imag, # 2x2 matrix 
         u20.real, u20.imag]
    B = [v10.real, v10.imag, # 2x2 matrix 
         v20.real, v20.imag]
    try:
        AB = A_inv_B(A,B)
    except Exception as e:
        raise MakeAffineMapError("make_affine_map: "+str(e))
    def pm(x):
        return v0 + complex((x-u0).real*AB[0] + (x-u0).imag*AB[2],
                     (x-u0).real*AB[1] + (x-u0).imag*AB[3])
    return pm

#==============================================================
#                 Plane vectors (=complex numbers)
#==============================================================

# v dot (w perpendicular)
def cross(v,w):
    return (v*w.conjugate()).imag

def vdot(v,w):
    return v.real*w.real + v.imag*w.imag

def collinear(points): # points: [complex] (The plane is the complex plane.)
    ZERO = 1e-8
    n = len(points)
    if n < 3:
        return True
    for i in range(n):
        for j in range(i+1,n):
            for k in range(j+1,n):
                a,b,c = points[i], points[j], points[k]
                ab = -a+b
                ac = -a+c
                bc = -b+c
                if abs(cross(ab,ac)) > ZERO:
                    return False
                if abs(cross(ab,bc)) > ZERO:
                    return False
                if abs(cross(ac,bc)) > ZERO:
                    return False
    return True

# ---------------------
# intersection_of_lines
# ---------------------
# Intersection of two lines.
# The plane is viewed as the complex number plane.
# The lines are given as pairs of points: [complex,complex]
# Return None if
# - the lines are parallel (coincident or not), or
# - the points in the pair for line0 or line1 are identical.
# Args:
# - line0: [complex,complex]
# - line1: [complex,complex]
# Returns:
# - None or complex
def intersection_of_lines(line0,line1):
    a,b = line0
    c,d = line1
    try:
        t = cross(c-a,d-c) / cross(b-a,d-c)
    except ZeroDivisionError: # parallel lines or a=b or c=d
        return None
    return a + t*(b-a)

# ------------------
# distance_from_line
# ------------------
# Distance of p from the line through a,b,
# or the distance from a if a=b.
# Optionally signed distance when a!=b.
# All points are complex numbers.
# Args:
# - p,a,b: complex
# - signed: boolean
# Returns:
# - float
def distance_from_line(p,a,b, signed=False):
    try:
        if not signed:
            return abs(cross(p-a,b-a) / (b-a))
        else:
            return cross(p-a,b-a) / abs(b-a)
    except ZeroDivisionError:
        return abs(p-a)

#==============================================================
#                       Bezier curve
#==============================================================

# ---------
# bezier_rv
# ---------
# The usual Bezier curve from control points.
# Version where all points are complex numbers
# (the plane is the complex number plane).
# Args:
# - t:  float
# - cp4:[complex,complex,complex,complex]
# Returns:
# - complex
def bezier_rv(t, cp4):
    P0,P1,P2,P3 = cp4
    if t > 0.5:
        u = (1-t)/t
        t3 = t**3
        return t3*(u*(u*(u*P0+3*P1)+3*P2)+P3) # complex
    else:
        u = t/(1-t)
        t3 = (1-t)**3
        return t3*(u*(u*(u*P3+3*P2)+3*P1)+P0) # complex

# Derivative: Returns complex.
def bezier_dot_rv(t, control_points):
    P0,P1,P2,P3 = control_points
    P01 = -P0+P1
    P12 = -P1+P2
    P23 = -P2+P3
    if t > 0.5:
        u = (1-t)/t
        t2 = t**2
        return 3*t2*(u*(u*P01+2*P12)+P23) # complex
    else:
        u = t/(1-t)
        t2 = (1-t)**2
        return 3*t2*(u*(u*P23+2*P12)+P01) # complex


# Second derivative
def bezier_dot_dot_rv(t, control_points): # Could be made faster as above.
    p0,p1,p2,p3 = control_points
    return -6*(1-t)*(-p0+p1) + 6*(1-2*t)*(-p1+p2) + 6*t*(-p2+p3)


# ------------------------
# bezier_tangential_points
# ------------------------
# Given an infinite Bezier curve B(t) (4 control points), find the points on
# the curve where B'(t) is parallel with the given direction.
# Notes:
# 1. Cusps are included (B'(t)=0).
# 2. The plane is viewed as the complex number plane.
# 3. If the curve is a straight line parallel to the direction None is returned.
# Args:
# - cp4: [complex,complex,complex,complex]
# - direction: complex
# Returns: None or
# - [[float,complex]] (=[[t,B(t)]])
def bezier_tangential_points(cp4, direction):
    p0,p1,p2,p3 = cp4
    p01 = -p0+p1
    p12 = -p1+p2
    p23 = -p2+p3
    c0 = -p01.real * direction.imag + p01.imag * direction.real
    c1 = -p12.real * direction.imag + p12.imag * direction.real
    c2 = -p23.real * direction.imag + p23.imag * direction.real
    ts = solve_bernstein2_equation(c0,c1,c2)
    if ts is None:
        return None
    return [[t,bezier_rv(t,cp4)] for t in ts]

# -------------------------
# bezier_new_control_points
# -------------------------
# Given a Bezier arc B(t) and two distinct parameter values a and b,
# find a new parametrization C(t) of the same curve such that
# B(a) = C(0) and B(b) = C(1).
# Args:
# - a:  float;
# - b:  float;
# - cp4:[complex,complex,complex,complex] (control points [p0,p1,p2,p3]).
# Returns:
# - [complex,complex,complex,complex] (new control points [q0,q1,q2,q3])
# Notes:
# - There is an exception: If a=b, the returned control points
#   are equal, so the new curve is a single point, not the original
#   curve. The function gives no signal about such occurence.
def bezier_new_control_points(a, b, cp4):
    def bezier_2_rv(t, r0, r1, r2): # Bezier curve of degree two (complex)
        b20 = (1-t)**2
        b21 = 2*(1-t)*t
        b22 = t**2
        return b20*r0 + b21*r1 + b22*r2
    p0,p1,p2,p3 = cp4
    q0 = bezier_rv(a, cp4)
    q1 = (1-b)*bezier_2_rv(a, p0,p1,p2) + b*bezier_2_rv(a, p1,p2,p3)
    q2 = (1-a)*bezier_2_rv(b, p0,p1,p2) + a*bezier_2_rv(b, p1,p2,p3)
    q3 = bezier_rv(b, cp4)
    return [q0,q1,q2,q3]

# -----------------
# bezier_arc_length
# -----------------
# Given 4 control points, get an approximate length of the Bezier arc.
# Note: Only approximation: Implementation computes the length simply from
# three sides of the control polygon.
def bezier_arc_length(cp4):
    if len(cp4) == 1:
        return 0
    p0,p1,p2,p3 = cp4
    return abs(-p0+p1) + abs(-p1+p2) + abs(-p2+p3)

# -------------------------
# bezier_arc_end_directions
# -------------------------
# Given a Bezier arc (control points, 0<=t<=1), find, at each end,
# the unit direction vector of the arc.
# Note: this does not mean the derivatives of the parametric representation,
# but non-zero vectors which are tangential to the arc and point in the
# direction where the arc is located when viewed from that end point.
# Return the result as two complex numbers (= two vectors),
# or as None if the arc is one point.
# The plane is viewed as the complex number plane, so points are complex numbers.
# Args:
# - cp4: [complex,complex,complex,complex] (control points)
# Returns: either None (when the arc is one point), or
# - complex (unit direction vector at starting point)
# - complex (unit direction vector at ending point)
def bezier_arc_end_directions(cp4):
    ZERO = 1e-12
    if len(cp4) == 1: # May happen in a 1-anchor stroke
        return None
    try:
        p0,p1,p2,p3 = cp4
    except ValueError:
        raise Exception("bezier_arc_end_directions: Should not happen 1!")
    p01 = -p0+p1
    p12 = -p1+p2
    p23 = -p2+p3
    if vdot(p01,p01) > ZERO:
        head_dir = p01
    elif vdot(p12,p12) > ZERO:
        head_dir = p12
    elif vdot(p23,p23) > ZERO:
        head_dir = p23
    else: # Arc is one point
        return None
    if vdot(p23,p23) > ZERO:
        tail_dir = -p23
    elif vdot(p12,p12) > ZERO:
        tail_dir = -p12
    elif vdot(p01,p01) > ZERO:
        tail_dir = -p01
    else:
        raise Exception("bezier_arc_end_directions: Should never happen 2!")
    return head_dir/abs(head_dir), tail_dir/abs(tail_dir)

#==============================================================
#                           Drawing
#==============================================================
# ------------------------
# gimp_draw_vectors_object
# ------------------------
# The Gimp version of the routine 'drawing_routine_for_bezier':
# Insert into Gimp the input vectors object, and make it visible if
# visible=True.
# Args:
# - image: Gimp's image (obtained usually from GUI);
# - vectors_object: a vectors object in Gimp;
# - visible: Boolean.
# Returns the vectors object in Gimp, and makes it visible if visible=True.
def gimp_draw_vectors_object(image, vectors_object, visible=True):
    # Insert a Gimp vectors object in Gimp.
    # Adapted from http://registry.gimp.org/files/lqr_wpset-0.14.py.txt
    # by Mike Kazantsev.
    def gimp_insert_vectors(image, vectors_object, position=0):
        if gimp.version >= (2, 8, 0):
            # 2.7.0 has gimp-image-insert-vectors, but fails with
            # parent=None
            parent = None
            return pdb.gimp_image_insert_vectors(
                     image, vectors_object, parent, position)
        else:
            return pdb.gimp_image_add_vectors(
                     image, vectors_object, position)# Deprecated!
    gimp_insert_vectors(image, vectors_object)
    vectors_object.visible = visible
    return vectors_object

# For debugging and experimenting only
def draw_cp4list(image, cp4list, name='cp4list'):
    balist = [BCurve.BezierArc(cp4=cp4) for cp4 in cp4list]
    bc = BCurve.BezierCurve(bezier_arcs = balist)
    vectors = bc.bc2vectors_object(image, name=name)
    gimp_draw_vectors_object(image, vectors, visible=True)

#==============================================================
#                 Bezier arc from tangent data
#==============================================================

class NoSolutionError(Exception):
    def __init__(self,m):
        self.message = m

# ------------------------
# bezier_from_tangent_data
# ------------------------
# Given two points p0,p3, a point K, and three directions (0,1,2),
# find control points cp4=[p0,p1,p2,p3] for all Bezier curves:
# - running through K and having there tangent with direction 0,
# - having at p0,p3 tangents with directions 1 and 2.
# Return the solutions as a list, each Bezier curve as cp4, list of 4 control points.
# Return also the parameter values t at which K is reached.
# There are two extra conditions to reject too wild solutions:
# respect_directions and wild_handle_limit.
# Args:
# - p0:         complex # Start point
# - p3:         complex # End point
# - K:          complex # Mean point
# - direction1: complex # Tangent at start point
# - direction2: complex # Tangent at end point
# - direction0: complex # Tangent at mean point
# - restrict01: boolean # Accept only 0<=t<=1
# - respect_directions: boolean # The arc should point (about) in the directions 1,2
#                               # at the end points,
#                               # and the derivative should point (about) in
#                               # the direction0 at point K
# - wild_handle_limit: None or float # to reject solutions with too long handles
#                                    # (None means that this conditon is not used)
# - print_info: boolean
# Returns:
# - [[float, [complex,complex,complex,complex]] (=[[t,cp4]])
def bezier_from_tangent_data(p0, p3, K,
                             direction1, direction2, direction0,
                             restrict01=True,
                             respect_directions=True,
                             wild_handle_limit=2, # Ad hoc limit
                             print_info=False):
    ZERO = 1e-8
    D1,D2 = p0+direction1, p3+direction2
    # Cases:
    if p0 == p3:
        case = 'VI'      # stroke is one loop or one point
    elif abs(direction1.real*direction2.imag - direction1.imag*direction2.real) < ZERO:
        if collinear([p0,p3,p0+direction1]):
            case = 'V'   # all collinear
        else:
            case = 'IV'  # parallel lines
    elif collinear([p0,p3,D2]):
        case = 'II'      # anchors on one of the lines
    elif collinear([p0,p3,D1]):
        case = 'III'     # anchors on one of the lines
    else:
        case = 'I'       # proper triangle
    # Different procedures for different cases:
    if print_info:
        print("\nCase = "+case)
        print("p0 = "+str(p0))
        print("p3 = "+str(p3))
        print("K = "+str(K))
    if case == 'I':
        C = intersection_of_lines([p0,p0+direction1], [p3, p3+direction2]) # apex
        if print_info:
            print("C = "+str(C))
        t_cp4_list = case_I_triangle(p0,p3,C,K, direction0,
                                     restrict01=restrict01,
                                     print_info=print_info)
    elif case == 'II':
        D = p0 + direction1
        t_cp4_list = case_II_flat(p0,p3,D,K, direction0,
                                  restrict01=restrict01,
                                  print_info=print_info)
    elif case == 'III':
        D = p3 + direction2
        t_cp4_list = case_II_flat(p3,p0,D,K, direction0,
                                  restrict01=restrict01,
                                  print_info=print_info)
        t_cp4_list = [[1-t,cp4[::-1]] for t,cp4 in t_cp4_list]
    elif case == 'IV':
        D = p0 + direction1
        t_cp4_list = case_IV_parallel(p0,p3,D,K, direction0,
                                      restrict01=restrict01,
                                      print_info=print_info)
    elif case == 'V':
        result = t_cp4_list = case_V_collinear(p0,p3,K,
                                                print_info=print_info)
    elif case == 'VI':
        D1 = p0+direction1
        D2 = p0+direction2
        t_cp4_list = case_VI_zero_chord(p0,D1,D2,K, direction0,
                                        restrict01=restrict01,
                                        print_info=print_info)
    if len(t_cp4_list) == 0:
        raise NoSolutionError("bezier_from_tangent_data: empty t_cp4_list")
    if case != 'V': # Case V: zero handles
        # Remove solutions violating the extra requirements:
        if respect_directions:
            accept = []
            for t,cp4 in t_cp4_list:
                q01 = -cp4[0]+cp4[1]
                q32 = -cp4[3]+cp4[2]
                derivative_at_K = bezier_dot_rv(t, cp4)
                chord = -p0+p3
                if (
                    (vdot(q01,direction1) > 0)
                    and (vdot(q32,direction2) > 0)
                    and (vdot(chord,derivative_at_K) > 0)
                    ):
                    accept.append([t,cp4])
            t_cp4_list = accept
        if not (wild_handle_limit is None):
            accept = []
            for t,cp4 in t_cp4_list:
                q0,q1,q2,q3 = cp4
                q01 = -q0+q1
                q02 = -q0+q2
                q03 = -q0+q3
                q12 = -q1+q2
                q13 = -q1+q3
                q23 = -q2+q3
                if (wild_handle_limit*(abs(q01)+abs(q23))
                                   <= (abs(q02)+abs(q03)+abs(q12)+abs(q13))
                    ):
                    accept.append([t,cp4])
            t_cp4_list = accept
    if len(t_cp4_list) == 0:
        raise NoSolutionError("bezier_from_tangent_data: empty t_cp4_list after removals")
    return t_cp4_list

# ---------------
# case_I_triangle
# ---------------
# One case of bezier_from_tangent_data: The given data forms a proper triangle
# with vertices A,B,C.
# Args:
# - A,B,C,K:    complex
# - direction0: complex
# - restrict01: boolean
# - print_info: boolean
# Returns:
# - [[float, [complex,complex,complex,complex]]] (=[[t,cp4]])
def case_I_triangle(A,B,C,K, direction0, restrict01=True, print_info=False):
    # Transform to a standard position:
    # A -> (0,1)
    # B -> (1,0)
    # C -> (0,0)
    sA, sB, sC = 1j, 1, 0
    try:
        aff_map = make_affine_map([A,B,C], [sA,sB,sC])
        inv_map = make_affine_map([sA,sB,sC], [A,B,C])
    except MakeAffineMapError as e:
        if print_info:
            print("case_I_triangle: make_affine_map raised: "+str(e))
        return []
    sK = aff_map(K)
    sd0 = aff_map(K + direction0) - sK
    u,v = sd0.real, sd0.imag
    a,b = sK.real, sK.imag
    # Solve the parameter values t.
    # Coefficients:
    c3 = -u+v
    c2 = 3*u
    c1 = -3*u -3*a*v + 3*b*u
    c0 = u + 2*a*v - b*u
    #ts = solve_pol3_R(c3,c2,c1,c0)
    ts = zeroes_of_polynomial([c0,c1,c2,c3])[0]
    if len(ts) == 0:
        raise NoSolutionError("case_I_triangle: empty ts")
    # Compute the handle lengths:
    result = []
    for t in ts:
        if restrict01 and not(0<t<1):
            continue
        xi  = (t*t*(3-2*t) - a) / (3*(1-t)*t*t)
        eta = ((1-t)*(1-t)*(1+2*t) - b) / (3*(1-t)*(1-t)*t)
        scp4 = [1j, complex(0,1-eta), complex(1-xi,0), 1] # cp4 in standard position
        cp4 = [inv_map(z) for z in scp4]
        result.append([t,cp4])
    return result

# ------------
# case_II_flat
# ------------
# Another case of bezier_from_tangent_data: The tangent at p0 runs through p3,
# or vice versa.
# Args:
# - A,B,D,K:    complex
# - direction0: complex
# - restrict01: boolean
# - print_info: boolean
# Returns:
# - [[float, [complex,complex,complex,complex]]] (=[[t,cp4]])
def case_II_flat(A,B,D,K, direction0, restrict01=True, print_info=False):
    # Transform the control points to standard position:
    # p0 -> (0,0)
    # p3 -> (1,0)
    # D -> (0,1)
    try:
        aff_map = make_affine_map([A,B,D], [0,1,1j])
        inv_map = make_affine_map([0,1,1j], [A,B,D])
    except MakeAffineMapError as e:
        if print_info:
            print("case_II_flat: make_affine_map raised: "+str(e))
        return []
    sK = aff_map(K)
    sd0 = aff_map(K + direction0) - sK
    u,v = sd0.real, sd0.imag
    a,b = sK.real, sK.imag
    # Solve the parameter values t.
    # Coefficients:
    c3 = v
    c2 = 0
    c1 = -3*a*v + 3*b*u
    c0 = 2*a*v - b*u
    #ts = solve_pol3_R(c3,c2,c1,c0)
    ts = zeroes_of_polynomial([c0,c1,c2,c3])[0]
    # Compute the handle lengths:
    result = []
    for t in ts:
        if restrict01 and not(0<=t<=1):
            continue
        xi  = (t*t*(3-2*t) - a) / (3*(1-t)*t*t)
        eta = b / (3*(1-t)*(1-t)*t)
        scp4 = [0, complex(0,eta), complex(1-xi,0), 1] # cp4 in standard position
        cp4 = [inv_map(z) for z in scp4]
        result.append([t,cp4])
    return result

# ----------------
# case_IV_parallel
# ----------------
# Another case of bezier_from_tangent_data: parallel tangents at end points.
# Args:
# - A,B,D,K:    complex
# - direction0: complex
# - restrict01: boolean
# - print_info: boolean
# Returns:
# - [[float, [complex,complex,complex,complex]]] (=[[t,cp4]])
def case_IV_parallel(A,B,D,K, direction0, restrict01=True, print_info=False):
    # Transform the control points to standard position:
    # p0 -> (0,0)
    # p3 -> (1,0)
    # D -> (0,1)
    try:
        aff_map = make_affine_map([A,B,D], [0,1,1j])
        inv_map = make_affine_map([0,1,1j], [A,B,D])
    except MakeAffineMapError as e:
        if print_info:
            print("case_IV_parallel: make_affine_map raised: "+str(e))
        return []
    sK = aff_map(K)
    sd0 = aff_map(K + direction0) - sK
    u,v = sd0.real, sd0.imag
    a,b = sK.real, sK.imag
    # Solve the parameter values t.
    # Coefficients:
    c3 = 2
    c2 = -3
    c1 = 0
    c0 = a
    #ts = solve_pol3_R(c3,c2,c1,c0)
    ts = zeroes_of_polynomial([c0,c1,c2,c3])[0]
    # Compute the handle lengths:
    try:
        v_per_u = v/u
    except ZeroDivisionError:
        if a in (0,1) and b == 0:
            raise NoSolutionError("Infinitely many solutions, IV")
        else:
            raise NoSolutionError("No solutions, IV, 1")
    result = []
    for t in ts:
        if restrict01 and not(0<=t<=1):
            continue
        try:
            upper = 2*v_per_u*(1-t)*t
            lower = b / (3*(1-t)*t)
        except ZeroDivisionError:
            if b == 0:
                raise NoSolutionError("Infinitely many solutions, IV")
            else:
                raise NoSolutionError("No solutions, IV, 2")
        xi  = (upper - (1-3*t)*lower) / t
        eta = (-upper + (2-3*t)*lower) / (1-t)
        scp4 = [0, complex(0,eta), complex(1,xi), 1]
        cp4 = [inv_map(z) for z in scp4]
        result.append([t,cp4])
    return result

# ----------------
# case_V_collinear
# ----------------
# Another case of bezier_from_tangent_data: all collinear
# Notes:
# 1. Returns t = 0.5.
# 2. Fails regularly. Must have K on the same line.
# Args:
# - p0,p3,K:    complex
# - restrict01: boolean
# - print_info: boolean
# Returns:
# - [[float, [complex,complex,complex,complex]]] (=[[t,cp4]])
def case_V_collinear(p0,p3,K, print_info=False):
    if not collinear([p0,p3,K]):
        if print_info:
            print("p0 = "+str(p0))
            print("p3 = "+str(p3))
            print("K = "+str(K))
        raise NoSolutionError("No solutions, V, 1")
    return [[.5, [p0,p0,p3,p3]]]

# ------------------
# case_VI_zero_chord
# ------------------
# Another case of bezier_from_tangent_data: Zero chord (p0=p3).
# Args:
# - A,D1,D2,K:    complex
# - direction0: complex
# - restrict01: boolean
# - print_info: boolean
# Returns:
# - [[float, [complex,complex,complex,complex]]] (=[[t,cp4]])
def case_VI_zero_chord(A,D1,D2,K, direction0, restrict01=True, print_info=False):
    # Transform the control points to standard position:
    # p0 = p3 -> (0,0)
    # D1 -> (0,1)
    # D2 -> (1,0)
    try:
        aff_map = make_affine_map([A,D1,D2], [0,1j,1])
        inv_map = make_affine_map([0,1j,1], [A,D1,D2])
    except MakeAffineMapError as e:
        if print_info:
            print("case_VI_zero_chord: make_affine_map raised: "+str(e))
        return []
    sK = aff_map(K)
    sd0 = aff_map(K + direction0) - sK
    u,v = sd0.real, sd0.imag
    a,b = sK.real, sK.imag
    # Solve the parameter values t.
    try:
        t = (2*a*v - b*u) / (3*(a*v - b*u))
    except ZeroDivisionError:
        raise NoSolutionError("Should not happen: ZeroDivisionError when computing t (K=p0?)")
    # Compute the handle lengths:
    if restrict01 and not(0<=t<=1):
        return []
    xi  = a / (3*(1-t)*t*t)
    eta = b / (3*(1-t)*(1-t)*t)
    scp4 = [0, complex(0,eta), complex(xi,0), 0] # cp4 in standard position
    cp4 = [inv_map(z) for z in scp4]
    return [[t,cp4]]


#==============================================================
#                   class SplittableArcList
#==============================================================

class NoSplitError(Exception):
    def __init__(self, message):
        self.message = message
    def __str__(self):
        return self.message

class SplittableArcList(object):
    """Data structure for storing and splitting a list of butting Bezier arcs,
    (lists of 4 control points).
    The plane is viewed as the complex number plane.
    Attributes:
    - cp4_list: [[complex,complex,complex,complex]] (list of control point
                lists [p0,lp1,p2,p3])
    - start_point
    - end_point
    - start_direction (unit vector)
    - end_direction   (unit vector)
    - length
    Methods:
    - split (splits self at an extremal point)
    - concatenate (concatenates two SplittableArcList's, assumed to be butting)
    - __str__
    Init arguments:
    - cp4_list
    """
    def __init__(self,
                 cp4_list,
                 ):
        self.cp4_list = cp4_list
        try:
            self.start_point = cp4_list[0][0]
            self.end_point = cp4_list[-1][-1]
        except IndexError:
            raise Exception("SplittableArcList: init: len(cp4_list) = "+str(len(cp4_list)))
        try_start_direction = bezier_arc_end_directions(cp4_list[0])
        try_end_direction = bezier_arc_end_directions(cp4_list[-1])
        if None in (try_start_direction, try_end_direction):
            raise NoSplitError("Cannot create SplittableArcList: an 1-point arc?")
        self.start_direction = try_start_direction[0]
        self.end_direction = try_end_direction[1]
        self.length = sum([bezier_arc_length(cp4) for cp4 in cp4_list])
    def __str__(self):
        s = "SplittableArcList, cp4_list:"
        count = 0
        for cp4 in self.cp4_list:
            s += "\n"+str(count)+": "+str(cp4[count])
            count += 1
        return s
    def split(self):
        ZERO = 1e-8
        i,t,_ = extreme_for_cp4list(self.cp4_list) # splitting point
        if t < ZERO:
            left_cp4_list = self.cp4_list[:i]
            right_cp4_list = self.cp4_list[i:]
        elif t > 1-ZERO:
            left_cp4_list = self.cp4_list[:i+1]
            right_cp4_list = self.cp4_list[i+1:]
        else: # Split arc cp4_list[i]
            left_cp4  = bezier_new_control_points(0, t, self.cp4_list[i])
            right_cp4 = bezier_new_control_points(t, 1, self.cp4_list[i])
            left_cp4_list = self.cp4_list[:i] + [left_cp4]
            right_cp4_list = [right_cp4] + self.cp4_list[i+1:]
        left = SplittableArcList(cp4_list = left_cp4_list)
        right = SplittableArcList(cp4_list = right_cp4_list)
        return left,right
    def concatenate(sal1,sal2): # Assumes that the cp4_lists are butting: no checking
        return SplittableArcList(cp4_list = sal1.cp4_list + sal2.cp4_list)

# -------------------
# extreme_for_cp4list
# -------------------
# Given a list of (butting) Bezier arcs Bi, find the point Bi(t) which is the
# most distant from the chord. Return
# - index i of the most distant arc in the list
# - parameter value t
# - point Bi(t).
# Args:
# - [complex,complex,complex,complex]
# Returns:
# - [int,float,complex] (=[i,t,Bi(t)])
# Note: So far, a loop not allowed.
def extreme_for_cp4list(cp4list):
    a,b = cp4list[0][0], cp4list[-1][-1]
    direction = -a + b
    loop_case = (direction == 0)
    candidates = [] # [[int,float,complex]] = [[i,t,B]]
    for i in range(len(cp4list)):
        cp4 = cp4list[i]
        if loop_case:
            t_Bs = bezier_critical_point_candidates(cp4, a, interval=[0,1]) # [[t,B(t)]]
        else:
            t_Bs = bezier_tangential_points(cp4, direction) # [[t,B(t)]]
        if not (t_Bs is None):
            for t,B in t_Bs:
                if 0<=t<=1:
                    candidates.append([i,t,B])
            candidates.append([i,0,cp4[0]])
            candidates.append([i,1,cp4[-1]])
        else:
            candidates.append([i,.5,cp4[-1]])
    if loop_case:
        return max(candidates, key=(lambda x:abs(x[2]-a)))
    else:
        return max(candidates, key=(lambda x:distance_from_line(x[2],a,b)))


#==============================================================
#                       Preprocessing
#==============================================================

def point_in_selection(x,y,image):
    if (0 <= x < image.width) and (0 <= y < image.height):
        _,pixel = pdb.gimp_drawable_get_pixel(image.selection, x, y)
        return (pixel[0] > 127)
    else:
        return False

# --------------
# discard_stroke
# --------------
# Check if the stroke should be included or discarded.
# It is discarded if:
# - its arcs are all below detail_threshold, and
# - either selection_case = 'fix' and no anchor belongs to the selection,
#   or selection_case = 'simplify' and all anchors belong to the selection.
# Equivalently: Return False if
# - its arcs are all below detail_threshold; and
#   - 'ignore':   empty condition;
#   - 'fix':      there exists an anchor belonging to the selection;
#   - 'simplify': there exists an anchor outside of the selection;
# otherwise return True.
# Args:
# - image
# - cp4list: [[complex,complex,complex,complex]]
# - simplify_parameters: SimplifyParameters
# Returns:
# - boolean
def discard_cp4list(image, cp4list, simplify_parameters):
    width, height = image.width, image.height
    detail_threshold = simplify_parameters.detail_threshold
    selection_case = simplify_parameters.selection_case
    for cp4 in cp4list:
        if bezier_arc_length(cp4) >= detail_threshold:
            return False
    if selection_case not in ('simplify', 'fix'):
        return True
    elif selection_case == 'simplify':
        for anchor in [cp4[0] for cp4 in cp4list] + [cp4list[-1][-1]]:
            x,y = anchor.real, anchor.imag
            if not ((0 <= x < width) and (0 <= y < height)):
                return False  # Found anchor not in selection (outside of the image)
            #_,pixel = pdb.gimp_drawable_get_pixel(image.selection, x, y)
            #if (pixel[0] <= 127): # Found anchor not in selection
            if not point_in_selection(x,y,image): # Found anchor not in selection
                return False
    else: # 'fix':
        for anchor in [cp4[0] for cp4 in cp4list] + [cp4list[-1][-1]]:
            x,y = anchor.real, anchor.imag
            if not ((0 <= x < width) and (0 <= y < height)):
                continue  # Anchor not in selection (outside of the image)
            #_,pixel = pdb.gimp_drawable_get_pixel(image.selection, x, y)
            #if (pixel[0] > 127): # Found anchor in selection
            if point_in_selection(x,y,image): # Found anchor not in selection
                return False
    return True

# --------------
# uncurl_cp4list
# --------------
# Given a list of butting Bezier arcs (as list of cp4's), defluctuate curly parts.
# The error is guaranteed to be at most simplify_parameters.allow_error.
# Args:
# - cp4list:             [[complex,complex,complex,complex]]
# - simplify_parameters: SimplifyParameters
# - print_info:          boolean (for debugging)
# - draw_intermediate:   boolean (for debugging)
# Returns:
# - [[complex,complex,complex,complex]]
def uncurl_cp4list(cp4list, simplify_parameters,
                   print_info=False, draw_intermediate=False):
    if len(cp4list) <= 3: # Not only for closed!
        return cp4list
    detail_threshold = simplify_parameters.detail_threshold
    allow_error      = simplify_parameters.allow_error
    level = simplify_parameters.work_level
    # Find [start,end] of curly segments in cp4list
    # (where [start,end] = [starting index, 1 + ending index]):
    curl_start_ends = []
    doing_curl = False
    for i in range(len(cp4list)):
        cp4 = cp4list[i]
        if bezier_arc_length(cp4) > detail_threshold:
            if doing_curl:
                curl_start_ends.append([start_curl,i])
                doing_curl = False
        else: # <= detail_threshold:
            if not doing_curl:
                start_curl = i
                doing_curl = True
    if doing_curl:
        curl_start_ends.append([start_curl,len(cp4list)])
    if print_info:
        print("uncurl_cp4list: curl_start_ends = "+str(curl_start_ends)+ ", len(cp4list) = "+str(len(cp4list)))
    # Defluctuate the curly segments:
    if len(curl_start_ends) == 0:
        return cp4list
    new_cp4list = []
    current_end = 0
    for curl_start, curl_end in curl_start_ends:
        new_cp4list += cp4list[current_end: curl_start]
        original = cp4list[curl_start: curl_end]
        try_current = original # Try to uncurl this
        error = 0
        modified = False
        count = 0
        for tries in range(10):
            try_uncurled, try_modified = defluctuate_cp4list(try_current)
            if try_modified:
                try_error = hausdorff_distance(try_uncurled, original, level)
                if try_error <= allow_error: # Try next round
                    try_current = try_uncurled
                    error = try_error
                    modified = True
                    if draw_intermediate:
                       draw_cp4list(image, try_current, name='uncurled ' +str(count))
                    count += 1
                    if print_info:
                        print(str(tries)+":  ACCEPT: error = "+str(error)+", simplify_parameters.allow_error) = "+str(simplify_parameters.allow_error))
                else:
                    if print_info:
                        print(str(tries)+":  REJECT: error = "+str(try_error)+", simplify_parameters.allow_error) = "+str(simplify_parameters.allow_error))
                    break
            else:
                if print_info:
                    print(str(tries)+":  STOP: not modified")
                break
        if modified:
            if print_info:
                print("uncurl_cp4list: Uncurling happened")
            new_cp4list += try_current
        else:
            if print_info:
                print("uncurl_cp4list: No uncurling")
            new_cp4list += cp4list[curl_start: curl_end]
        current_end = curl_end
    new_cp4list += cp4list[current_end:]
    return new_cp4list

# -------------------
# defluctuate_cp4list
# -------------------
# Diminish fluctuation in a Bezier curve, represented by cp4list.
# (This means a list of butting Bezier arcs.)
# The extreme anchors and the neighbouring handles are preserved
# Hence, a Bezier curve with only one Bezier arc is returned as such.
# Otherwise, the number of anchors is diminished by one.
# Args:
# - cp4list: [[complex,complex,complex,complex]]
# Returns:
# - [[complex,complex,complex,complex]] (similar to cp4list)
# - boolean (was the list modified?)
def defluctuate_cp4list(cp4list):
    cp4list = deepcopy(cp4list)
    if len(cp4list) <= 1:
        return cp4list, False
    new_cp4list = []
    for i in range(len(cp4list)-1):
        cp_1, cp_2 = cp4list[i:i+2]
        p0,p1,p2,p3 = cp_1
        q0,q1,q2,q3 = cp_2
        new_cp4list.append([(p0+q0)/2,
                            (p0+q0+p1+q1)/4,
                            (p3+q3+p2+q2)/4,
                            (p3+q3)/2])
    new_cp4list[0][:2] = cp4list[0][:2]
    new_cp4list[-1][-2:] = cp4list[-1][-2:]
    return new_cp4list, True

# ---------
# subdivide
# ---------
# A preliminary  step in the simplifying algorithm: Subdivide the given
# cp4list at certain anchors.
# Later, each part will be processed separately. This causes that
# the subdivision points (anchors) will be fixed points in the process.
# The subdivision points are:
# 1. corner points (anchors) where
#   - the angle <= simplify_parameters.corner_angle_threshold, and
#   - the legs >= simplify_parameters.detail_threshold;
# 2. ends of straight line segments exceeding simplify_parameters.detail_threshold;
# 3. if simplify_parameters.selection_case is not 'ignore', either
#    - all anchors outside of the selection if selection_case is 'simplify', or
#    - all anchors in the selection if selection_case is 'fix'.
# Args:
# - cp4list:             [[complex,complex,complex,complex]] (butting Bezier arcs)
# - closed:              boolean
# - simplify_parameters: SimplifyParameters
# - print_info:          boolean
# Returns:
# - [[[complex,complex,complex,complex]]] (butting list of butting Bezier arcs)
def subdivide(image, cp4list, closed, simplify_parameters, print_info=False):
    ZERO = 1e-8
    # Make a list to record for each cp4 whether its start and end points
    # will be subdivision points
    # (This brings some redundancy since almost every anchor appears in two cp4's.)
    subdivision_anchors = [[False,False] for i in range(len(cp4list))] # Initially all False
    angle = simplify_parameters.corner_angle_threshold
    short = simplify_parameters.detail_threshold
    selection_case = simplify_parameters.selection_case
    # 1. Subdivision points: Corners
    cos_angle = cos(angle*pi/180)
    for i in range(len(cp4list)):
        try:
            left, right = cp4list[i],cp4list[i+1]
        except IndexError: # At end of cp4list
            if closed:
                left,right = cp4list[-1],cp4list[0]
            else:
                continue
        if (bezier_arc_length(left) < short) or (bezier_arc_length(right) < short):
            continue
        try:
            left_dir = bezier_arc_end_directions(left)[1]
            right_dir = bezier_arc_end_directions(right)[0]
        except TypeError: # arc one point?
            continue
        cos_left_right = vdot(left_dir,right_dir)
        if cos_left_right + ZERO >= cos_angle: # Mark the anchor between left and right
            subdivision_anchors[i][1] = True
            try:
                subdivision_anchors[i+1][0] = True
            except IndexError:
                subdivision_anchors[0][0] = True
    # 2. Subdivision points: Ends of straight line segments
    if simplify_parameters.preserve_straight:
        for i in range(len(cp4list)):
            cp4 = cp4list[i]
            if collinear(cp4): # Straight line segment
                subdivision_anchors[i] = [True,True]
    # 3. Subdivision points: Points in the selection or outside of it.
    if selection_case == 'ignore':
        pass
    elif selection_case == 'simplify':
        # Points outside of the selection are made subdivision points.
        for i in range(len(cp4list)):
            cp4 = cp4list[i]
            x,y = cp4[0].real, cp4[0].imag
            #_,pixel = pdb.gimp_drawable_get_pixel(image.selection, x, y)
            #if pixel[0] <= 127: # Not in selection
            if not point_in_selection(x,y,image):
                subdivision_anchors[i][0] = True
            x,y = cp4[-1].real, cp4[-1].imag
            #_,pixel = pdb.gimp_drawable_get_pixel(image.selection, x, y)
            #if pixel[0] <= 127: # Not in selection
            if not point_in_selection(x,y,image):
                subdivision_anchors[i][1] = True
    elif selection_case == 'fix':
        # Points in the selection are made subdivision points.
        for i in range(len(cp4list)):
            cp4 = cp4list[i]
            x,y = cp4[0].real, cp4[0].imag
            #_,pixel = pdb.gimp_drawable_get_pixel(image.selection, x, y)
            #if pixel[0] > 127: # In selection
            if point_in_selection(x,y,image):
                subdivision_anchors[i][0] = True
            x,y = cp4[-1].real, cp4[-1].imag
            #_,pixel = pdb.gimp_drawable_get_pixel(image.selection, x, y)
            #if pixel[0] > 127: # In selection
            if point_in_selection(x,y,image):
                subdivision_anchors[i][1] = True
    # Now the subdivision points are marked. Do the subdivision:
    split_happens = False
    cp4list_list = []
    cumulate_cp4list = []
    for i in range(len(cp4list)):
        cp4 = cp4list[i]
        split_at_start, split_at_end = subdivision_anchors[i]
        if split_at_start and split_at_end:
            split_happens = True
            cp4list_list.append(cumulate_cp4list)
            cp4list_list.append([cp4])
            cumulate_cp4list = []
        elif split_at_start:
            split_happens = True
            cp4list_list.append(cumulate_cp4list)
            cumulate_cp4list = [cp4]
        elif split_at_end:
            split_happens = True
            cp4list_list.append(cumulate_cp4list + [cp4])
            cumulate_cp4list = []
        else:
            cumulate_cp4list.append(cp4)
    if subdivision_anchors[-1][1] == False:
        cp4list_list.append(cumulate_cp4list)
    if closed:
        if split_happens:
            if subdivision_anchors[-1][1] == False:
                cp4list_list = cp4list_list[1:-1] + [cp4list_list[-1]+cp4list_list[0]]
    cp4list_list = [x for x in cp4list_list if len(x) > 0]
    if print_info:
        print("\n\nsubdivide: end points of cp4list:")
        print(str((cp4list[0][0], cp4list[-1][-1])))
        print("subdivide: end points of subdivision:")
        for cp4l in cp4list_list:
            print(str((cp4l[0][0], cp4l[-1][-1])))
    return cp4list_list

#==============================================================
#            Approximation of a SplittableArcList
#==============================================================

class SimplifyParameters(object):
    def __init__(self,
                 selection_case,         # One of: 'ignore', 'simplify', 'fix'
                 allow_error,            # float
                 corner_angle_threshold, # max corner angle to preserve (degrees)
                 preserve_straight,      # preserve straight line segments
                 detail_threshold,       # shorted arcs probably just abolished (pixels)
                 work_level,             # higher level, harder working
                 do_smoothing            # boolean (smooth relevant remaining corners?)
                 ):
        work_level = max(0, min(10, work_level))
        self.selection_case = selection_case
        self.allow_error = allow_error
        self.corner_angle_threshold = corner_angle_threshold
        self.preserve_straight = preserve_straight
        self.detail_threshold = detail_threshold
        self.work_level = work_level
        self.do_smoothing = do_smoothing
    def __str__(self):
        s = 'SimplifyParameters:'
        s += '\n  selection_case: '+self.selection_case
        s += '\n  allow_error: '+str(self.allow_error)
        s += '\n  corner_angle_threshold: '+str(self.corner_angle_threshold)
        s += '\n  preserve_straight: '+str(self.preserve_straight)
        s += '\n  detail_threshold: '+str(self.detail_threshold)
        s += '\n  work_level: '+str(self.work_level)
        s += '\n  do_smoothing: '+str(self.do_smoothing)
        return s

# --------------
# sal_approx_cp4
# --------------
# Given a sal:SplittableArcList, find an approximate Bezier arc.
# Return errors and a list of 4 control points.
# Args:
# - sal:                 SplittableArcList
# - simplify_parameters: SimplifyParameters
# - print_info:          boolean
# Returns:
# - float (absolute error)
# - [complex,complex,complex,complex]
def sal_approx_cp4(sal, simplify_parameters, print_info=False):
    from cmath import exp as cexp
    level = allow_deviation = simplify_parameters.work_level
    N = 10
    p0,p3 = sal.start_point, sal.end_point
    _,_,K = extreme_for_cp4list(sal.cp4_list)
    d1,d2 = sal.start_direction, sal.end_direction
    try:
        d0 = (-p0+p3) / abs(-p0+p3)
    except ZeroDivisionError:
        p0K = -p0+K
        d0 = complex(-p0K.imag, p0K.real) # Orthogonal vector
    #
    try:
        bftd = bezier_from_tangent_data(p0, p3, K=K, # [[t,cp4]] list of solutions
                                        direction1=d1,
                                        direction2=d2,
                                        direction0=d0)
    except NoSolutionError as e:
        # Make new tries by allowing small deviations for K and direction0
        # (the tangent direction at K):
        # Deviations in angle:
        angles = [0]  # angle deviations, degrees
        for a in range(1, 1+allow_deviation):
            angles += [a,-a]
        # Deviations in K:
        ratios = [1-i/20. for i in range(1+allow_deviation)]
        m = (p0+p3)/2
        Ks = [k*K + (1-k)*m for k in ratios]
        # Deviations combined:
        try_K_angles = []  # [K,angle] deviations, angle in degrees
        for KK in Ks:
            try_K_angles += [[KK,a] for a in angles]
        try_K_angles = try_K_angles[1:] # Drop [K,0] that was already tried.
        if print_info:
            print("\nsal_approx_cp4: bezier_from_tangent_data: "+str(e))
            print("Trying deviations.")
        bftd = []
        for try_K, try_angle in try_K_angles:
            try_d0 = d0 * cexp(1j*try_angle*pi/180)
            try:
                bftd += bezier_from_tangent_data(p0, p3, K=try_K,
                                                 direction1=d1,
                                                 direction2=d2,
                                                 direction0=try_d0)
            except NoSolutionError as e:
                continue
    if len (bftd) == 0:
        raise NoSolutionError("No solutions")
    # Compute errors and choose best:
    error_cp4s = [[half_hausdorff_cp4(sal.cp4_list, cp4, level), cp4]
                      for t,cp4 in bftd]
    best_error, best_cp4 = min(error_cp4s, key=(lambda x:x[0])) # [error,cp4]
    # Ad hoc: Make new tries by shortening the handles in the same proportion:
    M = 20
    count = 0
    for x in [i/M for i in range(1,M)]:
        k = -2*x*x*x + 3*x*x # floats in (0,1), densest close to 0 and 1
        try_cp4 = [cp4[0],
                  k*cp4[0] + (1-k)*cp4[1],
                  k*cp4[3] + (1-k)*cp4[2],
                  cp4[3]]
        try_error = half_hausdorff_cp4(sal.cp4_list, try_cp4, level)
        if try_error >= best_error - 0.001:
            if print_info and (count == 0):
                print("sal_approx_cp4: NO IMPROVEMENT from handle shortening found")
            break # Break at first point of no improvement
        else:
            if print_info:
                relative = (best_error - try_error)# /  best_error
                print("sal_approx_cp4: IMPROVEMENT from handle shortening, count = "+str(count)+", "+str(relative))
            best_error, best_cp4 = try_error, try_cp4
            count += 1
    else:
        if print_info:
            print("sal_approx_cp4: LOOP for handle shortening run to end")
    if print_info:
        print("\nBEST:")
        print("abs_error = "+str(best_error))
        print("cp4:")
        for cp in best_cp4: print(cp)
    return best_error, best_cp4

# ------------------
# sal_approx_cp4list
# ------------------
# Given a sal:SplittableArcList, find an approximate Bezier curve.
# Return as a list of control point lists of butting Bezier arcs
# Args:
# - sal:                 SplittableArcList
# - simplify_parameters: SimplifyParameters
# - print_info:          boolean (for debugging)
# - draw_intermediate:   boolean (for debugging)
# Returns:
# - [[complex,complex,complex,complex]]
# Note: Non-recursive version.
def sal_approx_cp4list(sal, simplify_parameters,
                       print_info=False, draw_intermediate=False):
    if print_info:
        print("\nStarting sal_approx_cp4list with anchors of sal:")
        for cp4 in sal.cp4_list: print(str(cp4[0]))
        print(sal.cp4_list[-1][-1])
        print("= "*15)
    #if len(sal.cp4_list) == 1:
    #    return sal.cp4_list
    MAX_ABS_ERROR = simplify_parameters.allow_error
    MAX_ROUNDS = 500
    if len(sal.cp4_list) == 1:
        if print_info:
            print("sal_approx_cp4list: No splitting of a 1-arc list with")
            print("start and end: "+str((sal.cp4_list[0][0], sal.cp4_list[-1][-1])))
        raise NoSplitError("No splitting of a 1-arc list")
    if sal.length <= simplify_parameters.detail_threshold:
        if print_info:
            print("sal_approx_cp4list: detail_threshold reached")
        raise NoSplitError("detail_threshold reached")
    result_cp4list = []
    work_sal = sal         # work piece: SplittableArcList
    pending_sal_list = []  # pending: [SplittableArcList]
    for work in range(MAX_ROUNDS):
        if print_info:
            print("\nStarting work round "+str(work)+" with start and end:")
            print(str((work_sal.cp4_list[0][0], work_sal.cp4_list[-1][-1])))
        try:
            abs_err, cp4 = sal_approx_cp4(work_sal, simplify_parameters) # Main call
            if abs_err <= MAX_ABS_ERROR:
                if print_info:
                    print("=> abs_err = "+str(abs_err)+" <= "+str(MAX_ABS_ERROR)+". OK")
                if draw_intermediate:
                    draw_cp4list(image, [cp4], name='work round '+str(work))
                # Good. Store cp4. Update the work piece and the pending list:
                result_cp4list.append(cp4)
                try:
                    work_sal = pending_sal_list[0]
                    pending_sal_list = pending_sal_list[1:]
                    continue # Next round
                except IndexError: # pending_sal_list empty. Work done.
                    if print_info:
                        print("Work done. Returning.")
                        print("= "*15)
                    break
            if print_info:
                print("=> abs_err = "+str(abs_err)+" > "+str(MAX_ABS_ERROR)+". NOT OK. Going to split")
        except NoSolutionError as e:
            if print_info:
                print("\nNo solution: "+e.message+". Going to split")
        # Not good. Split. Update the work piece and the pending list:
        if work_sal.length <= simplify_parameters.detail_threshold:
            if print_info:
                print("sal_approx_cp4list: Should split but not done: detail_threshold reached")
            raise NoSplitError("detail_threshold reached")
        sal1, sal2 = work_sal.split()
        work_sal = sal1
        pending_sal_list = [sal2] + pending_sal_list
    else: # If come here, MAX_ROUND was not enough to get the work done.
        raise Exception("MAX_ROUND "+str(MAX_ROUNDS)+" was not enough to get the work done.")
    if len(result_cp4list) < len(sal.cp4_list):
        result = result_cp4list
    else:
        result = sal.cp4_list
    if print_info:
        print("\nsal_approx_cp4list returning:")
        for cp4 in result: print(str(cp4[0]))
        print(sal.cp4_list[-1][-1])
        print("= "*15)
    return result

# ------------------
# hausdorff_distance
# ------------------
# Hausdorff distance between cp4list1 and cp4list2, two lists of butting Bezier
# arcs (each arc represented as a list of 4 control points).
# To be used as an error measure for approximation.
# Args:
# - cp4list1: [[complex,complex,complex,complex]]
# - cp4list2: [[complex,complex,complex,complex]]
# - level: integer (working level)
# Returns:
# - float
def hausdorff_distance(cp4list1, cp4list2, level):
    #N = 10
    N = 5
    sample_list1 = [] # Sample points on cp4list1
    sample_list2 = [] # Sample points on cp4list2
    for cp4 in cp4list1:
        sample_list1 += [bezier_rv(i/N, cp4) for i in range(N+1)]
    for cp4 in cp4list2:
        sample_list2 += [bezier_rv(i/N, cp4) for i in range(N+1)]
    distances12 = [] # Distances from points in sample_list1 to arcs in cp4list2
    distances21 = [] # Distances from points in sample_list2 to arcs in cp4list1
    for p in sample_list1:
        p_dist = min([abs(p - bezier_closest_point(cp4, p, level)[1])
                      for cp4 in cp4list2])
        distances12.append(p_dist)
    for p in sample_list2:
        p_dist = min([abs(p - bezier_closest_point(cp4, p, level)[1])
                      for cp4 in cp4list1])
        distances21.append(p_dist)
    return max(distances12 + distances21)

# ------------------
# half_hausdorff_cp4
# ------------------
# One-sided Hausdorff distance from cp4list (list of butting Bezier arcs)
# to cp4 (one Bezier arc as a list of 4 control points).
# To be used as an error measure for approximation.
# Args:
# - cp4list: [[complex,complex,complex,complex]]
# - cp4:     [complex,complex,complex,complex]
# - level:   integer
# Returns:
# - float
def half_hausdorff_cp4(cp4list, cp4, level):
    if level <= 5:
        N = 3 # Sample points are the anchors and 2 points on each arc.
    elif level <= 8:
        N = 4 # Sample points are the anchors and 3 points on each arc.
    else:
        N = 5 # Sample points are the anchors and 4 points on each arc.
    sample_list = [] # Sample points on cp4list
    for cp41 in cp4list:
        sample_list += [bezier_rv(i/N, cp41) for i in range(N)]
    sample_list.append(cp4list[-1][-1])
    return max([abs(p - bezier_closest_point(cp4, p, level)[1])
                       for p in sample_list])

#==============================================================
#                       Smoothing corners
#==============================================================
# ----------------------
# smooth_corners_cp4list
# ----------------------
# Smooth corners > simplify_parameters.corner_angle_threshold
# Args:
# - cp4list:             [[complex,complex,complex,complex]] (butting Bezier arcs)
# - simplify_parameters: SimplifyParameters
# Returns:
# - [[complex,complex,complex,complex]] (butting Bezier arcs)
def smooth_corners_cp4list(cp4list, simplify_parameters):
    ZERO = 1e-5
    MAX_ROUNDS = 10
    cos_threshold = cos((pi/180)*simplify_parameters.corner_angle_threshold)
    level = simplify_parameters.work_level
    smoothed = cp4list
    count = 0
    while count < MAX_ROUNDS:
        count += 1
        modified = False
        # Run through 'smoothed', arcs as [left,right]-pairs.
        # As soon as found one pair where smoothing can be done,
        # do so, modify 'smoothed', and break the loop.
        # Then start anew with the modified 'smoothed'.
        for i in range(len(smoothed)-1):
            left  = smoothed[i]
            right = smoothed[i+1]
            left_dir  = bezier_arc_end_directions(left)[1] # Unit vector
            right_dir = bezier_arc_end_directions(right)[0] # Unit vector
            cos_angle = left_dir.real * right_dir.real + left_dir.imag * right_dir.imag
            if cos_angle < -1+ZERO: # straight: no corner
                continue # Try next pair
            if cos_angle >=  cos_threshold:
                continue # Try next pair
            # First method to smooth: sal_approx_cp4 (removes one arc)
            sal = SplittableArcList(cp4_list = [left,right])
            try:
                try_error, try_cp4 = sal_approx_cp4(sal, simplify_parameters, print_info=False)
                if try_error <= simplify_parameters.allow_error:
                    modified = True
                    smoothed = smoothed[:i] + [try_cp4] + smoothed[i+2:]
            except NoSolutionError as e:
                pass
            if modified:
                break # Modified: break for-loop
            # If come here, no modification was done.
            # Second method: smooth_corner (keeps arcs but changes handles)
            for tight_bend in range(3,20):
                try_cp4_pair = smooth_corner(left, right, tight_bend)
                try_error = hausdorff_distance(try_cp4_pair, [left,right], level)
                if try_error <= simplify_parameters.allow_error:
                    modified = True
                    smoothed = smoothed[:i] + try_cp4_pair + smoothed[i+2:]
                    break # Break at smallest tight_bend possible
        if modified:
            continue # Modified: start anew (new round in while-loop)
        else:
            break
    return smoothed

# -------------
# smooth_corner
# -------------
# Given two Bezier arcs forming a corner, smooth it.
# Args:
# - left_cp4:   [complex,complex,complex,complex]
# - right_cp4:  [complex,complex,complex,complex]
# - tight_bend: float (>= 0.1) (the larger number, the tighter bend)
# Returns:
# - [complex,complex,complex,complex]
# - [complex,complex,complex,complex]
def smooth_corner(left_cp4, right_cp4, tight_bend=3):
    from cmath import exp as cexp
    from cmath import polar,phase
    ZERO = 1e-6
    tight_bend = max(.1, tight_bend)
    p0,p1,p2,p3 = left_cp4
    q0,q1,q2,q3 = right_cp4 # p3 = q0
    left_directions = bezier_arc_end_directions(left_cp4)
    if left_directions is None:
        raise Exception("smooth_corner: left_cp4 is one point")
    else:
        left_dir = left_directions[1] # unit vector
    right_directions = bezier_arc_end_directions(right_cp4)
    if right_directions is None:
        raise Exception("smooth_corner: right_cp4 is one point")
    else:
        right_dir = right_directions[0] # unit vector
    if abs(left_dir.real * right_dir.imag - left_dir.imag * right_dir.real) < ZERO:
        return [left_cp4, right_cp4] # Collinear => already smooth
    left_len = bezier_arc_length(left_cp4)
    right_len = bezier_arc_length(right_cp4)
    
    middle = (left_dir + right_dir) / 2
    mperp = complex(-middle.imag, middle.real)
    
    x = (vdot(left_dir, mperp) / vdot(mperp,mperp)) * mperp
    
    new_left  =  x + ((left_len - right_len) / (left_len + right_len)) * middle
    new_right = -new_left
    
    P32 = left_len * new_left / tight_bend
    Q01 = right_len * new_right / tight_bend
    P2 = p3+P32
    Q1 = q0+Q01
    new_left_cp4  = [p0, p1, P2, p3]
    new_right_cp4 = [q0, Q1, q2, q3]
    return [new_left_cp4, new_right_cp4]

#==============================================================
#                       Simplifying a path
#==============================================================

# ----------------
# simplify_cp4list
# ----------------
# Args:
# - image
# - cp4list:             [[complex,complex,complex,complex]] (butting Bezier arcs)
# - closed:              boolean
# - simplify_parameters: SimplifyParameters
# Returns:
# - [[complex,complex,complex,complex]] (butting Bezier arcs)
def simplify_cp4list(image, cp4list, closed, simplify_parameters):
    #print("simplify_cp4list: starting with len(cp4list) = "+str(len(cp4list)))
    if len(cp4list[0]) == 1: # 1-anchor stroke: return as such
        return cp4list
    simplified_cp4list = []
    ### Preprocessing ###
    # Subdivide to sublists:
    cp4list_list = subdivide(image, cp4list, closed, simplify_parameters)
    # Uncurl "dense" places:
    cp4list_list = [uncurl_cp4list(cp4l, simplify_parameters)
                    for cp4l in cp4list_list]
    ### Process the sublists ###
    for cl in cp4list_list:
        try:
            sal = SplittableArcList(cp4_list=cl)
            simplified_sublist = sal_approx_cp4list(sal, simplify_parameters) # Main call
        except NoSplitError as e:
            #print('\n'+str(e))
            simplified_sublist = cl
        # Postprocessing: smoothing?
        if simplify_parameters.do_smoothing:
            postprocessed = smooth_corners_cp4list(simplified_sublist,
                                                   simplify_parameters)
        else:
            postprocessed = simplified_sublist
        simplified_cp4list += postprocessed
    #print("simplify_cp4list: returning len(simplified_cp4list) = "+str(len(simplified_cp4list)))
    return simplified_cp4list

# ---------------
# simplify_stroke
# ---------------
# Simplify a stroke. If the stroke is discarded in the process, return None.
# Args:
# - image
# - gs:                  BCurve.GimpStroke
# - simplify_parameters: SimplifyParameters
# Returns: None or
# - BCurve.GimpStroke
def simplify_stroke(image, gs, simplify_parameters):
    closed = gs.closed
    bc = gs.gs2bc()
    cp4_list = [ba.cp4 for ba in bc.bezier_arcs]
    if closed: # Close the gap
        gap_cp4 = [cp4_list[-1][-1],
                   bc.tail_handle,
                   bc.head_handle,
                   cp4_list[0][0]
                   ]
        if len(cp4_list[0]) == 1:
            cp4_list = [gap_cp4]
        else:
            cp4_list.append(gap_cp4)
    
    if discard_cp4list(image, cp4_list, simplify_parameters):
        return None
    simplified_cp4list = simplify_cp4list(image, cp4_list, closed, simplify_parameters) # Main call
    if closed:
        if len(simplified_cp4list) > 1:
            balist = [BCurve.BezierArc(cp4=cp4) for cp4 in simplified_cp4list[:-1]]
            bc = BCurve.BezierCurve(bezier_arcs = balist,
                                    head_handle = simplified_cp4list[-1][2],
                                    tail_handle = simplified_cp4list[-1][1],
                                    closed = True)
        else:
            p0,p1,p2,p3 = simplified_cp4list[0]
            bc = BCurve.BezierCurve(bezier_arcs = [BCurve.BezierArc(cp4=[p0])],
                                    head_handle = p2,
                                    tail_handle = p1,
                                    closed = True)
    else:
        balist = [BCurve.BezierArc(cp4=cp4) for cp4 in simplified_cp4list]
        bc = BCurve.BezierCurve(bezier_arcs = balist)
    return bc.bc2gs()

# -------------
# simplify_path
# -------------
# Simplify a path. If all strokes are discarded in the process, return None.
# Args:
# - image
# - path:                gimp.Vectors
# - simplify_parameters: SimplifyParameters
# Returns: None or
# - gimp.Vectors
def simplify_path(image, path, simplify_parameters):
    gv = vectors_object2gv(path) # BCurve.GimpVectors
    new_strokes = []
    for gs in gv.stroke_list:
        simplified_stroke = simplify_stroke(image, gs, simplify_parameters)
        if simplified_stroke is None:
            continue
        new_strokes.append(simplified_stroke)
    if len(new_strokes) == 0: # All strokes discarded
        return None
    gv = BCurve.GimpVectors(stroke_list = new_strokes)
    simplified_path = gv.gv2vectors_object(image, name = path.name+'|simplified')
    return simplified_path

# ------------------
# simplify_path_main
# ------------------
# The main procedure. Simplify a path. Draw the result.
# Args:
# - image:                  gimp.Image
# - path:                   gimp.Vectors 
# - allow_error:            float (pixels)
# - detail_threshold:       float (pixels)
# - corner_angle_threshold: degrees
# - preserve_straight:      boolean
# - selection_option:       integer
def simplify_path_main(image,
                       path, 
                       allow_error,
                       detail_threshold,
                       corner_angle_threshold,
                       preserve_straight,
                       work_level,
                       selection_option,
                       ):
    do_smoothing = True # No option for the user!
    work_level = max(0, min(10, work_level))
    corner_angle_threshold = max(0, min(180, corner_angle_threshold))
    selection_case = selection_options[selection_option][1]
    if selection_case != 'ignore':
        bounds = pdb.gimp_selection_bounds(image)
        if bounds[0] == 0:
            raise Exception("No active selection. (You may wish to try \"Ignore the selection\"?)")
    simplify_parameters = SimplifyParameters(
                            selection_case = selection_case,
                            allow_error = allow_error,
                            corner_angle_threshold = corner_angle_threshold,
                            preserve_straight = preserve_straight,
                            detail_threshold = detail_threshold,
                            work_level = work_level,
                            do_smoothing = do_smoothing
                            )
    simplified_path = simplify_path(image, path, simplify_parameters)
    # Draw
    if not (simplified_path is None): # May happen that all strokes are discarded.
        pdb.gimp_image_undo_group_start(image)
        gimp_draw_vectors_object(image, simplified_path, visible=True)
        pdb.gimp_image_undo_group_end(image)
    return simplified_path

selection_options = [ # (description, identifier)
        ('Ignore any selection',                                      'ignore'),
        ('Simplify the part inside the selection, preserve the rest', 'simplify'),
        ('Preserve the part inside the selection, simplify the rest', 'fix'),
        ]

#======================================================
#                    Registrations
#======================================================

versionnumber = "0.11"
procedure_author = "Markku Koppinen"
procedure_copyright = procedure_author
procedure_date = "2021"
image_types = "*"
menupath = '<Vectors>/Tools/Modify path'


#####################  Simplify path  #####################

procedure_name  = "simplify_path"
procedure_blurb = ("Simplify a path: Re-construct the path with fewer anchors and some smoothing."
                   #+"\nRe-construct the path with fewer anchors,"
                   #+"\nand some smoothing."
                   #+"\n(Slow for complicated paths.)"
                   +"\nNote: If you use a selection, out-of-canvas anchors"
                   +"\nare taken to be outside of the selection."
                   +"\nIf you need to include out-of-canvas anchors in the selection,"
                   +"\nyou have to enlarge the canvas temporarily first."
                   +"\n(Version "+versionnumber+")"
                   )
procedure_help  = ("Diminish number of anchors in a path,"
                   +"\nallowing some distortion."
                   +"\nVery experimental."
                   )
procedure_label = "Simplify"

procedure_function = simplify_path_main

register(
    procedure_name,
    procedure_blurb,
    procedure_help,
    procedure_author,
    procedure_copyright,
    procedure_date,
    procedure_label,
    image_types,
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_VECTORS, "path", "The path to simplify", None),
      (PF_FLOAT, "allow_error",
                 ("Allowed deviation from the original path (pixels, float)"
                  +"\n(only suggestive - actual value may locally become a little larger)"
                  )
                  , 3),
      (PF_FLOAT, "detail_threshold",
                 ("Detail threshold (pixels, float)"
                  +"\n(Smaller details may vanish.)"
                  ),
                  5),
      (PF_FLOAT, "corner_angle_threshold",
                  ("Corner angle threshold (degrees 0..180, float)"
                  +"\n(Corners to be preserved"
                  +" - larger corners may be smoothed out.)"
                  ),
                  120),
      (PF_BOOL,  "preserve_straight",
                 ("Preserve straight edges?"
                  #+"\n(Only those exceeding the detail threshold.)"
                  ),
                  False),
      (PF_INT, "allow_deviation",
                  ("Working level (integer 0..10)"
                  #+"\n(Larger value gives the plugin more freedom to work"
                  #+"\nwith the cost of longer running time.)"
                  +"\n(Higher level => harder work, longer running time.)"
                  ),
                  5),
      (PF_OPTION, 'selection_option','How to use the selection?',
                   0,
                  [case[0] for case in selection_options]), # Descriptions of cases
    ],
    [
        (PF_VECTORS, "simplified_path", "simplified path"),
    ],
    procedure_function,
    menu=menupath)


main()
