#!/usr/bin/env python
# -*- coding: iso-8859-15 -*-

# GIMP plugin replicate a layer on a path
# (c) J.F.Garcia 2013-2014 www.arakne.es

# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc., 59 Temple
# Place - Suite 330, Boston, MA 02111-1307, USA.

import os, sys, gtk, random


import string as _string
import math
import gimp
import gimpcolor
import re
from gimpenums import *
pdb = gimp.pdb

import gettext
t = gettext.translation("gimp20-python", gimp.locale_directory, fallback=True)
_ = t.ugettext

class error(RuntimeError): pass
class CancelError(RuntimeError): pass

PF_INT8        = PDB_INT8
PF_INT16       = PDB_INT16
PF_INT32       = PDB_INT32
PF_INT         = PF_INT32
PF_FLOAT       = PDB_FLOAT
PF_STRING      = PDB_STRING
PF_VALUE       = PF_STRING
#PF_INT8ARRAY   = PDB_INT8ARRAY
#PF_INT16ARRAY  = PDB_INT16ARRAY
#PF_INT32ARRAY  = PDB_INT32ARRAY
#PF_INTARRAY    = PF_INT32ARRAY
#PF_FLOATARRAY  = PDB_FLOATARRAY
#PF_STRINGARRAY = PDB_STRINGARRAY
PF_COLOR       = PDB_COLOR
PF_COLOUR      = PF_COLOR
PF_ITEM        = PDB_ITEM
PF_DISPLAY     = PDB_DISPLAY
PF_IMAGE       = PDB_IMAGE
PF_LAYER       = PDB_LAYER
PF_CHANNEL     = PDB_CHANNEL
PF_DRAWABLE    = PDB_DRAWABLE
PF_VECTORS     = PDB_VECTORS
#PF_SELECTION   = PDB_SELECTION
#PF_BOUNDARY    = PDB_BOUNDARY
#PF_PATH        = PDB_PATH
#PF_STATUS      = PDB_STATUS

PF_TOGGLE      = 1000
PF_BOOL        = PF_TOGGLE
PF_SLIDER      = 1001
PF_SPINNER     = 1002
PF_ADJUSTMENT  = PF_SPINNER

PF_FONT        = 1003
PF_FILE        = 1004
PF_BRUSH       = 1005
PF_PATTERN     = 1006
PF_GRADIENT    = 1007
PF_RADIO       = 1008
PF_TEXT        = 1009
PF_PALETTE     = 1010
PF_FILENAME    = 1011
PF_DIRNAME     = 1012
PF_OPTION      = 1013
PF_TAB         = 1014
PF_DRAW        = 1015
PF_TITLE       = 1016

_type_mapping = {
    PF_INT8        : PDB_INT8,
    PF_INT16       : PDB_INT16,
    PF_INT32       : PDB_INT32,
    PF_FLOAT       : PDB_FLOAT,
    PF_STRING      : PDB_STRING,
    PF_COLOR       : PDB_COLOR,
    PF_ITEM        : PDB_ITEM,
    PF_DISPLAY     : PDB_DISPLAY,
    PF_IMAGE       : PDB_IMAGE,
    PF_LAYER       : PDB_LAYER,
    PF_CHANNEL     : PDB_CHANNEL,
    PF_DRAWABLE    : PDB_DRAWABLE,
    PF_VECTORS     : PDB_VECTORS,

    PF_TOGGLE      : PDB_INT32,
    PF_SLIDER      : PDB_FLOAT,
    PF_SPINNER     : PDB_INT32,

    PF_FONT        : PDB_STRING,
    PF_FILE        : PDB_STRING,
    PF_BRUSH       : PDB_STRING,
    PF_PATTERN     : PDB_STRING,
    PF_GRADIENT    : PDB_STRING,
    PF_RADIO       : PDB_STRING,
    PF_TEXT        : PDB_STRING,
    PF_PALETTE     : PDB_STRING,
    PF_FILENAME    : PDB_STRING,
    PF_DIRNAME     : PDB_STRING,
    PF_OPTION      : PDB_INT32,
    PF_TAB         : PDB_STRING,
    PF_DRAW        : PDB_INT32,
    PF_TITLE       : PDB_STRING,
}

_obj_mapping = {
    PF_INT8        : int,
    PF_INT16       : int,
    PF_INT32       : int,
    PF_FLOAT       : float,
    PF_STRING      : str,
    PF_COLOR       : gimpcolor.RGB,
    PF_ITEM        : int,
    PF_DISPLAY     : gimp.Display,
    PF_IMAGE       : gimp.Image,
    PF_LAYER       : gimp.Layer,
    PF_CHANNEL     : gimp.Channel,
    PF_DRAWABLE    : gimp.Drawable,
    PF_VECTORS     : gimp.Vectors,

    PF_TOGGLE      : bool,
    PF_SLIDER      : float,
    PF_SPINNER     : int,

    PF_FONT        : str,
    PF_FILE        : str,
    PF_BRUSH       : str,
    PF_PATTERN     : str,
    PF_GRADIENT    : str,
    PF_RADIO       : str,
    PF_TEXT        : str,
    PF_PALETTE     : str,
    PF_FILENAME    : str,
    PF_DIRNAME     : str,
    PF_OPTION      : int,
    PF_TAB         : str,
    PF_DRAW        : int,
    PF_TITLE       : str,
}

_registered_plugins_ = {}

def register(proc_name, blurb, help, author, copyright, date, label, imagetypes, params, results, function, menu=None, domain=None, on_query=None, on_run=None):
    """This is called to register a new plug-in."""

    # First perform some sanity checks on the data
    def letterCheck(str):
        allowed = _string.letters + _string.digits + "_" + "-"
        for ch in str:
            if not ch in allowed:
		return 0
        else:
            return 1

    if not letterCheck(proc_name):
        raise error, "procedure name contains illegal characters"

    for ent in params:
        if len(ent) < 4:
            raise error, ("parameter definition must contain at least 4 "
                          "elements (%s given: %s)" % (len(ent), ent))

        if type(ent[0]) != int:
            raise error, "parameter types must be integers"

        if not letterCheck(ent[1]):
            raise error, "parameter name contains illegal characters"

    for ent in results:
        if len(ent) < 3:
            raise error, ("result definition must contain at least 3 elements (%s given: %s)" % (len(ent), ent))

        if type(ent[0]) != type(42):
            raise error, "result types must be integers"

        if not letterCheck(ent[1]):
            raise error, "result name contains illegal characters"

    plugin_type = PLUGIN

    if (not proc_name.startswith("python-") and
        not proc_name.startswith("python_") and
        not proc_name.startswith("extension-") and
        not proc_name.startswith("extension_") and
        not proc_name.startswith("plug-in-") and
        not proc_name.startswith("plug_in_") and
        not proc_name.startswith("file-") and
        not proc_name.startswith("file_")):
           proc_name = "python-fu-" + proc_name

    # if menu is not given, derive it from label
    need_compat_params = False
    if menu is None and label:
        fields = label.split("/")
        if fields:
            label = fields.pop()
            menu = "/".join(fields)
            need_compat_params = True

            import warnings
            message = ("%s: passing the full menu path for the menu label is "
                       "deprecated, use the 'menu' parameter instead" % (proc_name))
            warnings.warn(message, DeprecationWarning, 3)

        if need_compat_params and plugin_type == PLUGIN:
            file_params = [(PDB_STRING, "filename", "The name of the file", ""),
                           (PDB_STRING, "raw-filename", "The name of the file", "")]

            if menu is None:
                pass
            elif menu.startswith("<Load>"):
                params[0:0] = file_params
            elif menu.startswith("<Image>") or menu.startswith("<Save>"):
                params.insert(0, (PDB_IMAGE, "image", "Input image", None))
                params.insert(1, (PDB_DRAWABLE, "drawable", "Input drawable", None))
                if menu.startswith("<Save>"):
                    params[2:2] = file_params

    _registered_plugins_[proc_name] = (blurb, help, author, copyright, date, label, imagetypes, plugin_type, params, results, function, menu, domain, on_query, on_run)

def _query():
    for plugin in _registered_plugins_.keys():
        (blurb, help, author, copyright, date, label, imagetypes, plugin_type, params, results, function, menu, domain, on_query, on_run) = _registered_plugins_[plugin]

        def make_params(params):
            return [(_type_mapping[x[0]], x[1], _string.replace(x[2], "_", "")) for x in params]

        params = make_params(params)
        # add the run mode argument ...
        params.insert(0, (PDB_INT32, "run-mode", "The run mode { RUN-INTERACTIVE (0), RUN-NONINTERACTIVE (1) }"))

        results = make_params(results)

        if domain:
            try:
                (domain, locale_dir) = domain
                gimp.domain_register(domain, locale_dir)
            except ValueError:
                gimp.domain_register(domain)

        gimp.install_procedure(plugin, blurb, help, author, copyright, date, label, imagetypes, plugin_type, params, results)
        if menu:
            gimp.menu_register(plugin, menu)
        if on_query:
            on_query()

def getXtra(xtra, valName, defValue):
    p = defValue
    if xtra.get(valName):
        p = xtra.get(valName)
    return p

def _get_defaults(proc_name):
    import gimpshelf

    (blurb, help, author, copyright, date, label, imagetypes, plugin_type, params, results, function, menu, domain, on_query, on_run) = _registered_plugins_[proc_name]

    key = "python-fu-save--" + proc_name

    if gimpshelf.shelf.has_key(key):
        return gimpshelf.shelf[key]
    else:
        # return the default values
        return [x[3] for x in params]

def _set_defaults(proc_name, defaults):
    import gimpshelf

    key = "python-fu-save--" + proc_name
    gimpshelf.shelf[key] = defaults

def _interact(proc_name, start_params):
    (blurb, help, author, copyright, date, label, imagetypes, plugin_type, params, results, function, menu, domain, on_query, on_run) = _registered_plugins_[proc_name]

    def run_script(run_params):
        params = start_params + tuple(run_params)
        _set_defaults(proc_name, params)
        return apply(function, params)

    params = params[len(start_params):]

    # short circuit for no parameters ...
    if len(params) == 0:
         return run_script([])

    import pygtk
    pygtk.require('2.0')

    import gimpui, gtk
#    import pango

    defaults = _get_defaults(proc_name)
    defaults = defaults[len(start_params):]

    class EntryValueError(Exception):
        pass

    def warning_dialog(parent, primary, secondary=None):
        dlg = gtk.MessageDialog(parent, gtk.DIALOG_DESTROY_WITH_PARENT, gtk.MESSAGE_WARNING, gtk.BUTTONS_CLOSE, primary)
        if secondary:
            dlg.format_secondary_text(secondary)
        dlg.run()
        dlg.destroy()

    def error_dialog(parent, proc_name):
        import sys, traceback

        exc_str = exc_only_str = _("Missing exception information")
        try:
            etype, value, tb = sys.exc_info()
            exc_str = "".join(traceback.format_exception(etype, value, tb))
            exc_only_str = "".join(traceback.format_exception_only(etype, value))
        finally:
            etype = value = tb = None

        title = _("An error occurred running %s") % proc_name
        dlg = gtk.MessageDialog(parent, gtk.DIALOG_DESTROY_WITH_PARENT, gtk.MESSAGE_ERROR, gtk.BUTTONS_CLOSE, title)
        dlg.format_secondary_text(exc_only_str)

        alignment = gtk.Alignment(0.0, 0.0, 1.0, 1.0)
        alignment.set_padding(0, 0, 12, 12)
        dlg.vbox.pack_start(alignment)
        alignment.show()

        expander = gtk.Expander(_("_More Information"));
        expander.set_use_underline(True)
        expander.set_spacing(6)
        alignment.add(expander)
        expander.show()

        scrolled = gtk.ScrolledWindow()
        scrolled.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        scrolled.set_size_request(-1, 200)
        expander.add(scrolled)
        scrolled.show()

        label = gtk.Label(exc_str)
        label.set_alignment(0.0, 0.0)
        label.set_padding(6, 6)
        label.set_selectable(True)
        scrolled.add_with_viewport(label)
        label.show()

        def response(widget, id):
            widget.destroy()

        dlg.connect("response", response)
        dlg.set_resizable(True)
        dlg.show()

    # define a mapping of param types to edit objects ...
    class StringEntry(gtk.Entry):
        def __init__(self, default=""):
            gtk.Entry.__init__(self)
            self.set_text(str(default))
            self.set_line_wrap(True)
            self.set_single_line_mode(False)

        def get_value(self):
            return self.get_text()

    class LabelTitle(gtk.Label):
        def __init__(self, default="", align = (0.0, 0.0)):
            gtk.Label.__init__(self)
            self.set_label(str(default))
            self.set_use_markup(True)
            if align != None:
                self.set_alignment(align[0],align[1])
        def get_value(self):
            return self.get_label()

    class TextEntry(gtk.ScrolledWindow):
        def __init__ (self, default=""):
            gtk.ScrolledWindow.__init__(self)
            self.set_shadow_type(gtk.SHADOW_IN)

            self.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
            self.set_size_request(100, -1)

            self.view = gtk.TextView()
            self.add(self.view)
            self.view.show()
            self.buffer = self.view.get_buffer()
            self.set_value(str(default))

        def set_value(self, text):
            self.buffer.set_text(text)

        def get_value(self):
            return self.buffer.get_text(self.buffer.get_start_iter(), self.buffer.get_end_iter())

    class IntEntry(StringEntry):
        def get_value(self):
            try:
                return int(self.get_text())
            except ValueError, e:
                raise EntryValueError, e.args

    class FloatEntry(StringEntry):
            def get_value(self):
                try:
                    return float(self.get_text())
                except ValueError, e:
                    raise EntryValueError, e.args

    def precision(step):
        # calculate a reasonable precision from a given step size
        if math.fabs(step) >= 1.0 or step == 0.0:
            digits = 0
        else:
            digits = abs(math.floor(math.log10(math.fabs(step))));
        if digits > 20:
            digits = 20
        return int(digits)

    class SliderEntry(gtk.HScale):
        # bounds is (upper, lower, step)
        def __init__(self, default=0, bounds=(0, 100, 5)):
            step = bounds[2]
            self.adj = gtk.Adjustment(default, bounds[0], bounds[1], step, 10 * step, 0)
            gtk.HScale.__init__(self, self.adj)
            self.set_digits(precision(step))

        def get_value(self):
            return self.adj.value

    class SpinnerEntry(gtk.SpinButton):
        # bounds is (upper, lower, step)
        def __init__(self, default=0, bounds=(0, 100, 5)):
            step = bounds[2]
            self.adj = gtk.Adjustment(default, bounds[0], bounds[1], step, 10 * step, 0)
            gtk.SpinButton.__init__(self, self.adj, step, precision(step))

    class ToggleEntry(gtk.ToggleButton):
        def __init__(self, default=0, labels=[_("No"),_("Yes")]):
            gtk.ToggleButton.__init__(self)
            self.labels = labels
            self.label = gtk.Label(labels[0])
            self.add(self.label)
            self.label.show()
            self.connect("toggled", self.changed)
            self.set_active(default)

        def changed(self, tog):
            if tog.get_active():
                self.label.set_text(self.labels[1])
            else:
                self.label.set_text(self.labels[0])

        def get_value(self):
            return self.get_active()

    class RadioEntry(gtk.VBox):
        def __init__(self, default=0, items=((_("Yes"), 1), (_("No"), 0))):
            gtk.VBox.__init__(self, homogeneous=False, spacing=2)

            button = None

            for (label, value) in items:
                button = gtk.RadioButton(button, label)
                self.pack_start(button)
                button.show()

                button.connect("toggled", self.changed, value)

                if value == default:
                    button.set_active(True)
                    self.active_value = value

        def changed(self, radio, value):
            if radio.get_active():
                self.active_value = value

        def get_value(self):
            return self.active_value

    class ComboEntry(gtk.ComboBox):
        def __init__(self, default=0, items=()):
            store = gtk.ListStore(str)
            for item in items:
                store.append([item])

            gtk.ComboBox.__init__(self, model=store)

            cell = gtk.CellRendererText()
            self.pack_start(cell)
            self.set_attributes(cell, text=0)

            self.set_active(default)

        def get_value(self):
            return self.get_active()

    def FileSelector(default=""):
        # FIXME: should this be os.path.separator?  If not, perhaps explain why?
        if default and default.endswith("/"):
            selector = DirnameSelector
            if default == "/": default = ""
        else:
            selector = FilenameSelector
        return selector(default)

    class FilenameSelector(gtk.FileChooserButton):
        def __init__(self, default="", save_mode=False):
            gtk.FileChooserButton.__init__(self,_("Python-Fu File Selection"))
            self.set_action(gtk.FILE_CHOOSER_ACTION_OPEN)
            if default:
                self.set_filename(default)

        def get_value(self):
            return self.get_filename()

    class DirnameSelector(gtk.FileChooserButton):
        def __init__(self, default=""):
            gtk.FileChooserButton.__init__(self,_("Python-Fu Folder Selection"))
            self.set_action(gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER)
            if default:
                self.set_filename(default)

        def get_value(self):
            return self.get_filename()

    class DrawCanvas(gtk.DrawingArea):
        def __init__(self,default, size=(100,100)):
            gtk.DrawingArea.__init__(self)
            self.set_size_request(size[0],size[1])
        def get_value(self):
            return 0

    _edit_mapping = {
            PF_INT8        : IntEntry,
            PF_INT16       : IntEntry,
            PF_INT32       : IntEntry,
            PF_FLOAT       : FloatEntry,
            PF_STRING      : StringEntry,
            PF_COLOR       : gimpui.ColorSelector,
            PF_ITEM        : IntEntry,  # should handle differently ...
            PF_IMAGE       : gimpui.ImageSelector,
            PF_LAYER       : gimpui.LayerSelector,
            PF_CHANNEL     : gimpui.ChannelSelector,
            PF_DRAWABLE    : gimpui.DrawableSelector,
            PF_VECTORS     : gimpui.VectorsSelector,

            PF_TOGGLE      : ToggleEntry,
            PF_SLIDER      : SliderEntry,
            PF_SPINNER     : SpinnerEntry,
            PF_RADIO       : RadioEntry,
            PF_OPTION      : ComboEntry,

            PF_FONT        : gimpui.FontSelector,
            PF_FILE        : FileSelector,
            PF_FILENAME    : FilenameSelector,
            PF_DIRNAME     : DirnameSelector,
            PF_BRUSH       : gimpui.BrushSelector,
            PF_PATTERN     : gimpui.PatternSelector,
            PF_GRADIENT    : gimpui.GradientSelector,
            PF_PALETTE     : gimpui.PaletteSelector,
            PF_TEXT        : TextEntry,
            PF_TAB         : TextEntry,
            PF_DRAW        : DrawCanvas,
            PF_TITLE       : LabelTitle,
    }
    # JFGARCIA - modified: to pass args to the on_run
    if on_run:
        on_run(start_params+tuple(params))

    dialog = gimpui.Dialog(proc_name, "python-fu", None, 0, None, proc_name, (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL, gtk.STOCK_OK, gtk.RESPONSE_OK))

    dialog.set_alternative_button_order((gtk.RESPONSE_OK, gtk.RESPONSE_CANCEL))

    dialog.set_transient()

    vbox = gtk.VBox(False, 12)
    vbox.set_border_width(12)
    dialog.vbox.pack_start(vbox)
    vbox.show()

    if blurb:
        if domain:
            try:
                (domain, locale_dir) = domain
                trans = gettext.translation(domain, locale_dir, fallback=True)
            except ValueError:
                trans = gettext.translation(domain, fallback=True)
            blurb = trans.ugettext(blurb)
        box = gimpui.HintBox(blurb)
        vbox.pack_start(box, expand=False)
        box.show()

    table = gtk.Table(1, 2, False)
    table.set_row_spacings(2)
    table.set_col_spacings(2)
    vbox.pack_start(table, expand=True)
    table.show()

    notebook = gtk.Notebook()
    tables=[]
    for i in range(len(params)):
        if params[i][0]==PF_TAB:
            vals = params[i][4]
            currPage = params[i][3]

            for n in vals:
                tbl = gtk.Table(2, 2, False)
                tbl.set_row_spacings(5)
                tbl.set_col_spacings(5)
                tbl.show()
                tables.append({'table':tbl,'rows':0, 'totCols':0})
                notebook.append_page(tbl, gtk.Label(n))
                notebook.set_current_page(currPage)


    if len(tables)>0:
        table.attach(notebook, 0,6,0,1)
        notebook.show()
    else:
        tables.append({'table':table,'rows':0})

    def response(dlg, id):
        if id == gtk.RESPONSE_OK:
            dlg.set_response_sensitive(gtk.RESPONSE_OK, False)
            dlg.set_response_sensitive(gtk.RESPONSE_CANCEL, False)
            params = []
            try:
                for wid in edit_wids:
                    params.append(wid.get_value())
            except EntryValueError:
                warning_dialog(dialog, _("Invalid input for '%s'") % wid.desc)
            else:
                try:
                    dialog.res = run_script(params)
                except Exception:
                    dlg.set_response_sensitive(gtk.RESPONSE_CANCEL, True)
                    error_dialog(dialog, proc_name)
                    raise
        gtk.main_quit()

    iniX=0
    dialog.connect("response", response)
    row=0
    edit_wids = []

    for i in range(len(params)):
        paramsi = params[i]
        pf_type = paramsi[0]
        name, desc = (paramsi[1], paramsi[2])
        def_val = defaults[i]
 
        tab, incr, rows, connect, cols, hlp = (0, 1, 1, None, 1, desc)
        if len(params[i])>5:
            xtra = params[i][5]
            incr = int(getXtra(xtra, 'newLine', 1))
            if incr==0 : cols = 1
            cols = getXtra(xtra, 'cols'   , cols)
            iniX = getXtra(xtra, 'iniCol' , iniX)
            rows = getXtra(xtra, 'rows'   , 1)
            tab  = getXtra(xtra, 'tab'    , 0)
            hlp  = getXtra(xtra, 'help'   , desc)
            connect = getXtra(xtra,'connect', None)
        else:
            iniX = 0
        tbl = tables[tab]['table']
        row = tables[tab]['rows']


        if pf_type==PF_TAB:
            wid = _edit_mapping[pf_type](None)
        else:
            if pf_type != PF_TITLE and desc != '':
                label = gtk.Label(desc)
                label.set_use_underline(True)
                label.set_alignment(0.0, 0.5)
                tbl.attach(label, iniX + 1, iniX + 2, row, row + rows, xoptions = gtk.FILL)
                label.show()

            if pf_type in (PF_SPINNER, PF_SLIDER, PF_RADIO, PF_OPTION, PF_DRAW):
                wid = _edit_mapping[pf_type](def_val, paramsi[4])
            elif pf_type==PF_TOGGLE:
                if isinstance(paramsi[4], list):
                    wid = _edit_mapping[pf_type](def_val, paramsi[4])
                else:
                    wid = _edit_mapping[pf_type](def_val)
            elif pf_type==PF_TITLE:
                wid = _edit_mapping[pf_type](desc)
            else:
                wid = _edit_mapping[pf_type](def_val)
            if pf_type!=PF_TITLE and desc != '':
                label.set_mnemonic_widget(wid)
                tbl.attach(wid, iniX + 2, iniX + 2 + cols, row, row + rows, yoptions=0)
                iniX = iniX + cols + 1
            else:
                tbl.attach(wid, iniX, iniX+cols, row, row + rows, xoptions=gtk.FILL)
            # jfGarcia: strip html tags from toolTip
            tTip = re.sub('<[^<]+?>', '', hlp)
            if pf_type != PF_TEXT:
                wid.set_tooltip_text(tTip)
            else:
                # Attach tip to TextView, not to ScrolledWindow
                wid.view.set_tooltip_text(tTip)

            wid.set_name(name)
            if connect:
                try:
                    wid.connect(connect['signal'],connect['function'])
                except Exception, e:
                    pass
            wid.show()
            wid.desc = desc

        edit_wids.append(wid)

        tables[tab]['rows'] += incr
        if incr==1:
           iniX=0

    progress_vbox = gtk.VBox(False, 6)
    vbox.pack_end(progress_vbox, expand=False)
    progress_vbox.show()

    progress = gimpui.ProgressBar()
    progress_vbox.pack_start(progress)
    progress.show()

    dialog.show()

    gtk.main()

    if hasattr(dialog, "res"):
        res = dialog.res
        dialog.destroy()
        return res
    else:
        dialog.destroy()
        raise CancelError

def _run(proc_name, params):
    run_mode = params[0]
    func = _registered_plugins_[proc_name][10]

    if run_mode == RUN_NONINTERACTIVE:
        return apply(func, params[1:])

    script_params = _registered_plugins_[proc_name][8]

    min_args = 0
    if len(params) > 1:
        for i in range(1, len(params)):
            param_type = _obj_mapping[script_params[i - 1][0]]
            if not isinstance(params[i], param_type):
                break
        min_args = i
    if len(script_params) > min_args:
        start_params = params[:min_args + 1]
        if run_mode == RUN_WITH_LAST_VALS:
            default_params = _get_defaults(proc_name)
            params = start_params + default_params[min_args:]
        else:
            params = start_params
    else:
       run_mode = RUN_NONINTERACTIVE
    if run_mode == RUN_INTERACTIVE:
        try:
            res = _interact(proc_name, params[1:])
        except CancelError:
            return
    else:
        res = apply(func, params[1:])
        gimp.displays_flush()
    return res

def main():
    """This should be called after registering the plug-in."""
    gimp.main(None, None, _query, _run)

def fail(msg):
    """Display an error message and quit"""
    gimp.message(msg)
    raise error, msg

def N_(message):
    return message


params = []
dbuf = None
p=os.path.dirname(sys.argv[0]) + os.sep +'locales'
gettext.install("follow-path", p, unicode=True) 

def alert(msg):
	if isinstance(msg, int):
		msg=str(msg)
	dialog = gtk.MessageDialog(None, 0, gtk.MESSAGE_INFO, gtk.BUTTONS_OK, msg)
	dialog.run()
	dialog.hide()

def get_name(obj):
    try:
        return obj.name
    except Exception, e:
        return ''

def find (node, id):
    if get_name(node) == id:
        return node
    try:
        if node.get_children():
            for child in node.get_children():
                ret = find(child, id)
                if ret: return ret
    except Exception,e:
        pass
    return None


def findValues(wgtl,arrNames):
	return ([find(wgtl,Name).get_value() for Name in arrNames])

def calcRot(x1 , x2 , y1 , y2):
	rot = math.atan2(y1-y2,x1-x2)
	return rot

def rndVal(v1,v2):
	if v1==0 and v2==0:
		return 0
	r1=v1 if v1<v2 else v2
	r2=v1 if v1>v2 else v2
	if r1==r2:
		return r1
	return random.randrange(r1,r2)

def drawLine(cnv, gc, x1, y1, x2, y2, rel):
    cnv.draw_line(gc, int(x1*rel),int(y1*rel) , int(x2*rel), int(y2*rel))

def drawCross(cnv, gc, x, y, w):
    cnv.draw_line(gc, int(x)-w,int(y) , int(x)+w, int(y))
    cnv.draw_line(gc, int(x),int(y)-w , int(x), int(y)+w)

def rotPnt(pnt,rot):
	cs = math.cos(rot)
	sn = math.sin(rot)
	nx = cs * pnt[0] - sn * pnt[1]
	ny = sn * pnt[0] + cs * pnt[1]
	return [nx, ny]

def exposeCanvas(widget,data):
	testFn()

def getCoords(path, copias, ease):
	posX=[]
	length = pdb.gimp_vectors_stroke_get_length(path, 1, 1)
	ldiv = length/(copias-1)
	for n in range(copias):
		nEase= n * ldiv if ease==False else n*n
		posX.append(nEase)
	rel=length/posX[copias-1]
	if ease==True:
		posX=[n * rel for n in posX]
	coords = []
	for n in range(copias):
		x_p, y_p, sl, valid = pdb.gimp_vectors_stroke_get_point_at_dist(path, 1, posX[n], 1)
		if not valid:
			x_p, y_p, sl, valid = pdb.gimp_vectors_stroke_get_point_at_dist(path, 1, posX[n]-0.0001, 1)
		coords.append([x_p,y_p])
	minY=min([nn[1] for nn in coords])
	maxY=max([nn[1] for nn in coords])
	mapY=maxY - minY
	return (coords,minY,maxY,mapY)

def testFn(widget):
	wgtl = widget.get_toplevel()
	bgColor = wgtl.style.bg[gtk.STATE_NORMAL]
	cnvs=find(wgtl,'preview')

	if isinstance(params[1], gimp.Vectors):
		path=params[1]
	else:
		path=find(wgtl,'refpath').get_active_vectors()

	length = pdb.gimp_vectors_stroke_get_length(path, 1, 1)
	if cnvs:
		style = cnvs.get_style()
		st=path.strokes[0]
		pts=st.points[0]
		gc = style.fg_gc[gtk.STATE_NORMAL]
		img=params[0]

		lyr=find(wgtl,'reflayer').get_active_layer()
		lcpy = pdb.gimp_layer_copy(lyr, True)
#		pdb.plug_in_autocrop_layer(img, lcpy)
		lo0, lo1, lw, lh=(lcpy.offsets[0], lcpy.offsets[1], lcpy.width, lcpy.height)
		pdb.gimp_item_delete(lcpy)

		W, H = cnvs.size_request()
		w, h = (img.width, img.height)
		gc0 = cnvs.window.new_gc()
		cmap = gc0.get_colormap()
# common colors
		blue  = cmap.alloc_color(red=0,green=0,blue=65535)
		red   = cmap.alloc_color(red=65535,green=0,blue=0)
		white = cmap.alloc_color("white")
		black = cmap.alloc_color("black")
		gc0.line_width = 1

		dbuf = gtk.gdk.Pixmap(cnvs.window,W,H,-1)
		gc1 = dbuf.new_gc()
		gc1.set_colormap(cmap)
		gc1.set_foreground(bgColor) # Erase previous image
		dbuf.draw_rectangle(gc1,True,0,0,W,H)
		Rel=(W*1.0)/(max(w,h)*1.0)
		n = 0

		gc1.set_foreground(white)
		dbuf.draw_rectangle(gc1,True,0,0,int(w*Rel),int(h*Rel))
		copias = int(find(wgtl,'copy').get_value())
		rotOrient = 0.0

		scIniX, scEndX, scEndY, scIniY = findValues(wgtl,['scIniX','scEndX','scEndY','scIniY'])
		ease,rI, rEnd = findValues(wgtl,['ease',"rotIni","rotEnd"])
		scXrnd1, scYrnd1, scXrnd2, scYrnd2, rotrndP, rotrndm = findValues(wgtl,['scXrndP','scYrndP','scXrndM','scYrndM','rotrndP','rotrndM'])
		orient,anX,anY =findValues(wgtl,['rot','anX','anY'])

		sciX, sceX, sciY, sceY = (scIniX / 100.0, scEndX / 100.0, scIniY / 100.0, scEndY / 100.0)
		scIncScX, scIncScY =((sceX-sciX)/copias,(sceY-sciY)/copias)
		incRot=(rEnd-rI)/copias
		length = pdb.gimp_vectors_stroke_get_length(path, 1, 1)
		ldiv = length/(copias-1)
		coords, minY, maxY, mapY = getCoords(path, copias, ease)

		for n in range(copias):
			x_p, y_p = (coords[n][0], coords[n][1])
			actY=(y_p-minY)/mapY
			drawCross(dbuf,gc,x_p*Rel,y_p*Rel,2)
			cuad = [[0.0,0.0],[0.0,lh],[lw,lh],[lw,0.0]] 
			scRX,scRY=(rndVal(scXrnd1 ,scXrnd2)/100.0 , rndVal(scYrnd1 ,scYrnd2)/100.0)
			slw=lw
			slh=lh
			if (sciX != 1 or sceX != 1 or sciY != 1 or sceY != 1 or scRX!=1 or scRY!=1):
				mapScY=int(find(wgtl,'mapScY').get_value())
				if mapScY:
					scX = int(lw*(sciX + ((sceX-sciX) * actY)))
					scY = int(lh*(sciY + ((sceY-sciY) * actY)))
				else:
					scX=int(lw*(sciX+scIncScX*n+scRX))
					scY=int(lh*(sciY+scIncScY*n+scRY))
				cuad = [[0.0,0.0],[0.0,scY],[scX,scY],[scX,0.0]]
				slw, slh=(scX, scY)
			rot=0.0
			if (incRot!=0.0):
				rot = math.radians(rI+n*incRot)
			rndRot = rndVal(rotrndP, rotrndm)
			if rndRot!=0:
				rot += math.radians(rndRot)
			if orient == 1:
				if n<copias:
					x_p2, y_p2, sl, valid = pdb.gimp_vectors_stroke_get_point_at_dist(path, 1, ldiv * (n+1), 1)
					if n<copias-1:
						rotOrient = calcRot(x_p, x_p2, y_p, y_p2)
			rotate = rot + rotOrient
			for nn in range(4):
				cuad[nn]=rotPnt(cuad[nn], rotate)

			ax = [l[0] for l in cuad]
			ay = [l[1] for l in cuad]
			ox, oy, oX, oY = (min(ax), min(ay) ,max(ax) , max(ay))
			aX = oX-ox
			aY = oY-oy
			for nn in range(4):
				cuad[nn]=[cuad[nn][0]-ox,cuad[nn][1]-oy]
			mX = [0.0,-slw/2.0,-slw][anX]
			mY = [-slh,-slh/2.0,0][anY]
			mX = [0.0,-aX/2.0,-aX][anX]
			mY = [-aY,-aY/2.0,0][anY]
			for nn in range(3):
				drawLine(dbuf,gc,(x_p+cuad[nn][0]+mX),(y_p+cuad[nn][1]+mY),(x_p+cuad[nn+1][0]+mX),(y_p+cuad[nn+1][1]+mY),Rel)
			drawLine(dbuf,gc,(x_p+cuad[3][0]+mX),(y_p+cuad[3][1]+mY),(x_p+cuad[0][0]+mX),(y_p+cuad[0][1]+mY),Rel)
		cnvs.window.draw_drawable(gc0,dbuf,0,0,0,0,W,H) # refresh

def between(v,vMin,vMax):
	if v<vMin: v=vMin
	if v>vMax: v=vMax
	return v

def moveLayerTo(layer, px, py, destX, destY):
	lw, lh=(layer.width, layer.height)
	ox, oy = layer.offsets
	incx = int([0, -lw/2.0,-lw][px])
	incy = int([-lh, -lh/2.0,0][py])
	layer.translate(destX + incx-ox,destY + incy-oy)
	ox,oy = layer.offsets
	return ([ox,ox + lw/2, destX][px],[destY, oy+lh/2, oy][py])

class cFPath(object):
	def __init__(self, img, path, lay, preview, copy, anchorTit, anX, anY, scTit, 
		sclIniX, sclEndX, scXrnd1, scXrnd2, sclIniY, sclEndY, scYrnd1, scYrnd2, mapScY,
		rotTit, rotIni,  rotEnd,  rotrndP, rotrndm, opaTit, opaIni,  opaEnd,  opaRndI, opaRndE, opaMapY, blurTit, blrType,
		blrIni,  blrEnd,  blrrndP, blrrndm, blaIni,  blaEnd,  blarndP, blarndm, blMapY,	hsvTit,  
		hueI,    hueE,    hueIRnd, hueErnd, hueMapY, satIni,  satEnd,  satIRnd, satERnd, satMapY,
		lgtIni,  lgtEnd,  lgtIRnd, lgtERnd,	lightMapY,	orient,  merge,   ease, tab2):
		copias = int(copy)
		rotOrient=0.0
		sciX, sceX, sciY, sceY = (sclIniX/100.0, sclEndX/100.0, sclIniY/100.0, sclEndY/100.0)
		rI,rEnd=(rotIni,rotEnd)
		scIncScX,scIncScY =((sceX-sciX)/copias,(sceY-sciY)/copias)
		oI,oE =(opaIni,  opaEnd)
		incOpa=(oE-oI)/copias
		hI, hE, sI, sE, lI, lE= (hueI, hueE, satIni, satEnd, lgtIni, lgtEnd)
		incRot=(rEnd-rI)/copias
		incHue, incSat, incLight=((hE-hI)/copias, (sE-sI)/copias, (lE-lI)/copias)
		length = pdb.gimp_vectors_stroke_get_length(path, 1, 1)
		ldiv = length/(copias-1)
		pdb.gimp_image_undo_group_start(img)
		lay_grp = pdb.gimp_layer_group_new(img)
		pdb.gimp_image_insert_layer(img, lay_grp, None, 0)
		coords, minY, maxY, mapY = getCoords(path, copias, ease)
		mapScY=int(mapScY)
		iblr=(blrEnd-blrIni)/(copias-1)
		ibla=(blaEnd-blaIni)/(copias-1)
		for n in range(copias):
			lcopy = pdb.gimp_layer_copy(lay, True)
			lcopy.name = lay.name + str(n)
			x_p,y_p = (coords[n][0],coords[n][1])
			actY=(y_p-minY)/mapY
			pdb.gimp_image_insert_layer(img, lcopy, lay_grp, 0)
#			pdb.plug_in_autocrop_layer(img, lcopy)
			scRX=rndVal(scXrnd1 ,scXrnd2)/100.0
			scRY=rndVal(scYrnd1 ,scYrnd2)/100.0
			if opaMapY==1:
				lcopy.opacity = abs(actY*oE-(1-actY)*oI)
			else:
				if incOpa<>0.0:
					lcopy.opacity=oI+incOpa*n
			if (sciX != 1 or sceX != 1 or sciY != 1 or sceY != 1 or scRX!=1 or scRY != 1):
				o=lcopy.offsets
				if mapScY:
					scX = (sciX + ((sceX-sciX) * actY))
					scY = (sciY + ((sceY-sciY) * actY))
					lcopy.name = lay.name + str(int(y_p)).zfill(5)
				else:
					scX=(sciX + scIncScX * n + scRX)
					scY=(sciY + scIncScY * n + scRY)
			hueRnd,satRnd,lgtRnd=(rndVal(hueIRnd,hueErnd),rndVal(satIRnd,satERnd),rndVal(lgtIRnd,lgtERnd))
			if (incHue!=0.0 or incSat!=0.0 or incLight!=0.0 or hueRnd!=0.0 or satRnd!=0.0 or lgtRnd!=0.0):
				actHue =  (hE-hI)*actY if hueMapY == 1 else incHue*n
				actLgt =  (lE-lI)*actY if lightMapY == 1 else incLight*n
				actSat =  (sE-sI)*actY if satMapY == 1 else incSat*n
				hueVal=between(int(hI+actHue+hueRnd),-180,180)
				lgtVal=between(int(lI+actLgt+lgtRnd),-100,100)
				satVal=between(int(sI+actSat+satRnd),-100,100)
				pdb.gimp_hue_saturation(lcopy, 0, hueVal, lgtVal, satVal)
			rot = 0.0
			if (incRot != 0):
				rot = math.radians(rI+n*incRot)
			rndRot = rndVal(rotrndP, rotrndm)
			if rndRot != 0.0:
				rot += math.radians(rndRot)
			if orient == 1:
				if n<copias:
					x_p2, y_p2, sl, valid = pdb.gimp_vectors_stroke_get_point_at_dist(path, 1, ldiv * (n+1), 1)
					if n<copias-1:
						rotOrient = calcRot(x_p, x_p2, y_p, y_p2)
			rotate = rot + rotOrient
			# el ultimo proceso BLUR:
			#"linear","radial","zoom"
			if ((blrType in [0,2]) and (blrIni!=0 or blrEnd!=0 or blrrndP>0 or blrrndm>0)):
				blrrnd = rndVal(blrrndP,blrrndm)
				actBlr = abs(actY*blrEnd-(1-actY)*blrIni) if blMapY else blrIni+iblr*n
				gimp.message(str(actBlr))
				pdb.plug_in_mblur(img, lcopy, blrType, int(actBlr)+blrrnd, 0, 0, 0)
			elif ((blrType==1) and (blaIni>0 or blaEnd>0 or blarndP>0 or blarndm>0)):
				blarnd=rndVal(blarndP,blarndm)
				actBAng = abs(actY*blaEnd-(1-actY)*blaIni) if blMapY == 1 else blaIni+ibla*n
				pdb.plug_in_mblur(img, lcopy, blrType, 0, int(actBAng)+blarnd, 0, 0)
			#pdb.plug_in_mblur(img, lcopy, blrType, length, angle, center_x, center_y)
			tx, ty = moveLayerTo(lcopy, anX, anY, int(x_p), int(y_p))
			item = pdb.gimp_item_transform_2d(lcopy, tx, ty, scX, scY, rotate, tx, ty)
		if mapScY:
			for n,item in enumerate(sorted(lay_grp.children,key=lambda x: x.name,reverse=True)):
				pdb.gimp_image_reorder_item(img, item, lay_grp, n)
		if merge==1:
			while len(lay_grp.children)>1:
				pdb.gimp_image_merge_down(img, lay_grp.children[0], 0)
		pdb.gimp_image_undo_group_end(img)
		pdb.gimp_displays_flush()

def fPath(img, srcPath, lay, preview, copy, anchorTit, anX, anY, scTit,
    sclIniX, sclEndX, scXrndP, scXrndm, sclIniY, sclEndY, scYrndP, scYrndm,
	mapScY,
    rotTit, rotIni,  rotEnd,  rotrndP, rotrndm, opaTit, opaIni,  opaEnd,  opaRndI, opaRndE, opaMapY, blurTit,
    blrType, blrIni,  blrEnd,  blrrndP, blrrndm, blaIni,  blaEnd,  blarndP, blarndm, blMapY,
	hsvTit, hueI,    hueE,    hueIrnd, hueErnd, hueMapY, satIni,  satEnd,  satIrnd, satErnd, satMapY,
	lgtIni,  lgtEnd,  lgtIRnd, lgtERnd, lightMapY, orient,  merge,   ease, ptab):
	try:
		cFPath(img, srcPath,lay,preview, copy, anchorTit,anX, anY, scTit,sclIniX, sclEndX, scXrndP, scXrndm,  sclIniY, sclEndY, scYrndP, scYrndm,
		mapScY,
		rotTit, rotIni,  rotEnd,  rotrndP, rotrndm, opaTit, opaIni,  opaEnd,  opaRndI, opaRndE, opaMapY, blurTit,  blrType,
		blrIni,  blrEnd,  blrrndP, blrrndm, blaIni,  blaEnd,  blarndP, blarndm, blMapY,
		hsvTit, hueI,    hueE,    hueIrnd, hueErnd,  hueMapY, satIni,  satEnd,  satIrnd, satErnd, satMapY,
		lgtIni,  lgtEnd,  lgtIRnd, lgtERnd, lightMapY, orient,merge,ease,ptab)
	except Exception,e:
		print e.args[0]
		pdb.gimp_message(e.args[0])
	pdb.gimp_displays_flush()
	return

def myrun(arg=None):
	global params
	params=arg

### Registration
whoiam='\n'+os.path.abspath(sys.argv[0])

connect = {'signal':'changed','function':testFn}
sRndFrom = _("Random from")
arrMapY = [_("Don't map on Y"),_("Map on Y")]
hlpMapY = "Maps the first value on the lowest Y coordinate and the 2nd on the highest"
fnArgs=[
	(PF_IMAGE  , "image"   , "Input image"    , None ,None, {'cols':9,'connect':connect}),
	(PF_VECTORS, "refpath" , "Input path"     , None ,None, {'cols':9,'connect':connect}),
	(PF_LAYER  , "reflayer", "Layer"          , None ,None, {'cols':9,'connect':connect}),
	(PF_DRAW   , "preview" , _("P\nR\nE\nV\nI\nE\nW")     , 0    ,(300,300) ,{'cols':1,'newLine':'0','iniCol':0,'rows':10,'connect':{'signal':'expose-event','function':exposeCanvas}}),
	(PF_SPINNER, "copy"    , _("Copies:")     , 10   ,(1,+500,1),{'cols':1,'newLine':1,'connect':connect,'iniCol':2}),
	(PF_TITLE , "anchorTit", _("<b>Anchor</b> (Center for layer transforms)"), (0.0,0.0) ,None,{'cols':8,'iniCol':3,'newLine':1}),
	(PF_OPTION , "anX"     , "X:"     , 1 , (_("left"),_("center"),_("right")),{'cols':3,'newLine':'0','iniCol':2,'connect':connect}),
	(PF_OPTION , "anY"     , "Y:"     , 1 , (_("bottom"),_("center"),_("top")),{'cols':3,'newLine':1,'iniCol':6,'connect':connect}),
	(PF_TITLE , "scTit"    , _("<b>Scale</b>"), (0.0,0.0) ,None,{'cols':8,'iniCol':3,'newLine':1,}),
	(PF_SPINNER, "scIniX"  , _("X from")      , 100, (1, +200,    1),{'newLine':'0','iniCol':2,'connect':connect}),
	(PF_SPINNER, "scEndX"  , _("To")          , 100, (1, +200,    1),{'newLine':'0','connect':connect}),
	(PF_SPINNER, "scXrndP" , sRndFrom          , 0,   (1, +200,    1),{'newLine':'0','connect':connect}),
	(PF_SPINNER, "scXrndM" , _("To")          , 0,   (1, +200,    1),{'newLine':1,'cols':1,'connect':connect}),

	(PF_SPINNER, "scIniY"  , _("Y from")      , 100, (1, +200,    1),{'iniCol':2,'newLine':'0','connect':connect}),
	(PF_SPINNER, "scEndY"  , _("To")          , 100, (1, +200,    1),{'newLine':'0','connect':connect}),
	(PF_SPINNER, "scYrndP" , sRndFrom          , 0,   (1, +200,    1),{'newLine':'0','connect':connect}),
	(PF_SPINNER, "scYrndM" , _("To")          , 0,   (1, +200,    1),{'newLine':1,'connect':connect}),

	(PF_TOGGLE , "mapScY"  , ""                , 0,[_("Don't map scale on Y value"),_("Map scale on Y value")],{'iniCol':3,'cols':7,'newLine':1,'connect':{'signal':'toggled','function':testFn}}),

	(PF_TITLE , "rotTit"   , _("<b>Rotate</b>"), (0.0,0.0) ,None,{'cols':8,'iniCol':3,'newLine':1}),
	(PF_SPINNER, "rotIni"  , _("From") , 0    , (-360, +360, 1),{'iniCol':2,'newLine':'0','connect':connect}),
	(PF_SPINNER, "rotEnd"  , _("To")          , 0  , (-360, +360, 1),{'newLine':'0','connect':connect}),
	(PF_SPINNER, "rotrndP" , sRndFrom          , 0  , (-360, +360, 1),{'newLine':'0','connect':connect}),
	(PF_SPINNER, "rotrndM" , _("To")          , 0  , (-360, +360, 1),{'cols':1,'newLine':1,'connect':connect}),

	(PF_TITLE  , "opaTit"  , _("<b>Opacity</b>"),(0.0,0.0), None  ,{'tab':1,'cols':5}),
	(PF_SPINNER, "opaIni"  , _("From")        , 100 ,(0,    +100, 1),{'newLine':'0','tab':1}),
	(PF_SPINNER, "opaEnd"  , _("To")          , 100 ,(0,    +100, 1),{'newLine':'0','tab':1}),
	(PF_SPINNER, "oparndP" , sRndFrom          , 0  , (0,    +100, 1),{'newLine':'0','tab':1}),
	(PF_SPINNER, "oparndM" , _("To")          , 0  , (0,    +100, 1),{'newLine':'0','tab':1}),
	(PF_TOGGLE , "opaMapY" , ""               , 0  ,arrMapY,{'newLine':1,'tab':1,'iniCol':9,'help':hlpMapY}),

	(PF_TITLE  , "blurTit"  , _("<b>Blur</b>"),None, None  ,{'tab':1}),

	(PF_OPTION,  "blrType"  , _("Type")         , 0  , ("linear","radial","zoom"),{'cols':5,'newLine':1,'tab':1}),
	(PF_SPINNER, "blrIni"  , _("Length from")   , 0  , (0,    +100, 1),{'newLine':'0','tab':1}),
	(PF_SPINNER, "blrEnd"  , _("To")            , 0  , (0,    +100, 1),{'newLine':'0','tab':1}),
	(PF_SPINNER, "blrrndP" , sRndFrom            , 0  , (0,    +100, 1),{'newLine':'0','tab':1}),
	(PF_SPINNER, "blrrndM" , _("To")            , 0  , (0,    +100, 1),{'newLine':1,'tab':1}),
	(PF_SPINNER, "blaIni"  , _("Angle")         , 0  , (0,    +360, 1),{'newLine':'0','tab':1}),
	(PF_SPINNER, "blaEnd"  , _("To")            , 0  , (0,    +360, 1),{'newLine':'0','tab':1}),
	(PF_SPINNER, "blarndP" , sRndFrom            , 0  , (0,    +360, 1),{'newLine':'0','tab':1}),
	(PF_SPINNER, "blarndM" , _("To")            , 0  , (0,    +360, 1),{'newLine':'0','tab':1}),
	(PF_TOGGLE , "blMapY" , ""                   , 0  ,arrMapY,{'newLine':1,'tab':1,'iniCol':9,'help':hlpMapY}),

	(PF_TITLE  , "HSVtit"  , _("<b>HSL</b>")    ,(0,0), None           ,{'tab':1}),
	(PF_SPINNER, "hueIni"  , _("Hue")           , 0  , (-180, +180, 1),{'newLine':'0','tab':1}),
	(PF_SPINNER, "hueEnd"  , _("To")            , 0  , (-180, +180, 1),{'newLine':'0','tab':1}),
	(PF_SPINNER, "hueIRnd" , sRndFrom            , 0  , (-180, +180, 1),{'newLine':'0','tab':1}),
	(PF_SPINNER, "hueERnd" , _("To")            , 0  , (-180, +180, 1),{'newLine':'0','tab':1}),
	(PF_TOGGLE , "hueMapY" , ""                  , 0  ,arrMapY,{'newLine':1,'tab':1,'iniCol':9,'help':hlpMapY}),
	
	(PF_SPINNER, "satIni"  , _("Saturation from"), 0 , (-100, +100, 1),{'newLine':'0','tab':1}),
	(PF_SPINNER, "satEnd"  , _("To")            , 0  , (-100, +100, 1),{'newLine':'0','tab':1}),
	(PF_SPINNER, "satIRnd" , sRndFrom            , 0 ,  (-100, +100, 1),{'newLine':'0','tab':1}),
	(PF_SPINNER, "satERnd" , _("To")            , 0  , (-100, +100, 1),{'newLine':'0','tab':1}),
	(PF_TOGGLE , "satMapY" , ""                  , 0  ,arrMapY,{'newLine':1,'tab':1,'iniCol':9,'help':hlpMapY}),

	(PF_SPINNER, "lightIni" , _("Lightness:")    , 0  , (-100, +100, 1),{'newLine':'0','tab':1}),
	(PF_SPINNER, "lightEnd" , _("To:")           , 0  , (-100, +100, 1),{'newLine':'0','tab':1}),
	(PF_SPINNER, "lightIRnd", sRndFrom           , 0  , (-100, +100, 1),{'newLine':'0','tab':1}),
	(PF_SPINNER, "lightERnd", _("To:")           , 0  , (-100, +100, 1),{'newLine':'0','tab':1}),
	(PF_TOGGLE , "lightMapY" , ""                , 0  ,arrMapY,{'newLine':1,'tab':1,'iniCol':9,'help':hlpMapY}),

	(PF_TOGGLE , "rot"     , ""                  , 0,[_("Don't orient on path"),_("Orient on path")],{'help':_('Makes the layer rotate with the path'),'iniCol':3,'cols':3,'newLine':'0','connect':{'signal':'toggled','function':testFn}}),
	(PF_TOGGLE , "merge"   , ""                  , 0,[_("Don't merge layers"),_("Merge layers")],{'cols':3,'newLine':'0','iniCol':6}),
	(PF_TOGGLE , "ease"    , ""                  , 0,["Don't ease","Ease"],{'iniCol':9,'cols':2,'newLine':'1','connect':{'signal':'toggled','function':testFn}}),
	(PF_TAB    , "ptab"    , "tabs"              , 0, (_('Position, rotation & scale'),_('Opacity, blur & color')) )
]

register("arakne-follow-path5", N_("Repeat a layer on a path..."+whoiam), "Follow path", "J.F.Garcia", "J.F.Garcia", "2014", _("Follow path"), "RGB*,GRAY*",
	fnArgs, [], fPath,#function
	menu="<Vectors>/Tools", domain=("gimp20-python", gimp.locale_directory), on_run = myrun)

main()
