#!/usr/bin/env python

# Label_Arrows_From_Path Rel 2
# Created by Tin Tran
# Comments directed to http://gimpchat.com or http://gimpscripts.com
#
# License: GPLv3
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY# without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# To view a copy of the GNU General Public License
# visit: http://www.gnu.org/licenses/gpl.html
#
#
# ------------
#| Change Log |
# ------------
# Rel 1: Initial release. has cut and pasted code from ofnuts for arrow heads and svg vector_to_line_stroke which uses svg to stroke path with line
# Rel 2: Changed code suggested/given by ofnuts so that script would work on Linux and OSX as well as Windows.

import math
import string
#import Image
from gimpfu import *
from array import array
#ofnuts' arrow head code starts =========================================================================================
AXIS_TANGENT=0
AXIS_BEST=1
AXIS_NAMES=["Tangent","Best fit"]

SIDE_START=0
SIDE_END=1
SIDE_BOTH=2
SIDE_NAMES=["Start","End","Both"]

class ArrowHeadAdder(object):
	arrowWingSize = 0
	arrowWingAngle = 0
	closeArrowHead = False
	angle = 0
	arrowHeadAxis = 0
	sides = 0
	arrowHeadAtStart=False
	arrowHeadAtEnd=False
	image=None
	sourcePath=None
	
	def __init__(self, image, path, size, angle, sides, close, axis):
		self.image = image
		self.sourcePath = path
		self.arrowWingSize = size
		self.angle = angle # kept for display
		self.arrowWingAngle = (angle / 180) * math.pi
		self.sides = sides # kept for display
		if sides == SIDE_START or sides == SIDE_BOTH:
			self.arrowHeadAtStart = True
		if sides == SIDE_END or sides == SIDE_BOTH:
			self.arrowHeadAtEnd = True
		self.closeArrowHead = close
		self.arrowHeadAxis = axis
		
	def __str__(self):
		return '%d, %d, %s, "%s, %s"' % (self.arrowWingSize, self.angle, SIDE_NAMES[self.sides], AXIS_NAMES[self.arrowHeadAxis], self.closeArrowHead)
		
	def generatePathWithArrowsHeads(self):
		#arrowHeadsPath=pdb.gimp_vectors_new(self.image, 'Arrow heads(%s) for %s' % (self,self.sourcePath.name))
		arrowHeadsPath=pdb.gimp_vectors_new(self.image, 'Arrow Head Vectors')
		pdb.gimp_image_add_vectors(self.image, arrowHeadsPath, 0)

		for stroke in self.sourcePath.strokes:
			points,closed=stroke.points
			if not closed:
				self.addArrowHeads(arrowHeadsPath,points)
		arrowHeadsPath.visible = True
	
	def addArrowHeads(self,arrowHeadsPath,points):
		if self.arrowHeadAtStart:
			endPoints=self.getHeadDataAtStart(points)
			headStrokePoints=self.generateHeadStroke(endPoints)
			self.addArrowHead(arrowHeadsPath,headStrokePoints)
			
		if self.arrowHeadAtEnd:
			endPoints=self.getHeadDataAtEnd(points)
			headStrokePoints=self.generateHeadStroke(endPoints)
			self.addArrowHead(arrowHeadsPath,headStrokePoints)
	
	def addArrowHead(self,arrowHeadsPath,headPoints):
			sid = pdb.gimp_vectors_stroke_new_from_points(arrowHeadsPath,0, len(headPoints), headPoints, self.closeArrowHead)

	def generateHeadStroke(self,endPoints):
		arrowTip = endPoints[0:2]
		tangentPoint = self.tangentReferencePoint(endPoints)
		# Arrow wings start from the tip, towards the tangent anchor
		tangentAngle = self.segmentAngle(arrowTip,tangentPoint)
		
		wingTip1 = self.pointAtAngleDistance(arrowTip, tangentAngle + self.arrowWingAngle, self.arrowWingSize)
		wingTip2 = self.pointAtAngleDistance(arrowTip, tangentAngle - self.arrowWingAngle, self.arrowWingSize)
		# Since the arrow head is all straight lines we merely triplicate the points
		strokePoints=wingTip1*3 + arrowTip*3 + wingTip2*3
		return strokePoints
		
	def tangentReferencePoint(self,endPoints):
		if self.arrowHeadAxis == AXIS_TANGENT:
			return self.trueTangentPoint(endPoints)
		else: # self.arrowHeadAxis == AXIS_BEST
			return self.bestFitPoint(endPoints)
		
	# The points of interest:
	# - the ending point
	# - its tangent handle towards next/previous anchor point
	# - the tangent handle from next/previous point to ending point
	# - the next previous anchor point
	def getHeadDataAtStart(self,points):
		return points[2:10]
		
	def getHeadDataAtEnd(self,points):
		return self.reversePoints(points[-10:-2])
		
	def reversePoints(self,points):
		reversed=[]
		for i in range(len(points)-2,-1,-2):
			reversed.append(points[i])
			reversed.append(points[i+1])
		return reversed

	# Tangent determination. The pure tangent is given by:
	# - the tangent handle for this anchor point, if it exists
	# - the tangent handle at the next/revious anchor point, if it exists
	# - the next/previous anchor point
	# However since missing handles use the coordinate of the anchor, the 
	# third case is the same as the 2nd.
	def trueTangentPoint(self,points):
		if points[2:4] == points[0:2]:
			return points[4:6]
		else:
			return points[2:4]	
	
	# Rough calculation to find where the curve enters the visual triangle 
	# created by the wings. The length isn't accurate, and nothing says that
	# pointAt(t) is actually at length*t. But it seems good enough.
	def bestFitPoint(self,points):
		curveLength = self.approximateBezierLength(points)
		t = self.arrowWingSize * math.cos(self.arrowWingAngle) / curveLength
		bestFitPoint = self.bezierPointAt(points, t)
		return bestFitPoint

	# Calculation of the corrdinates of a point on the curve. The computation
	# here is a literal equivalent to the geometric construction. The canonical
	# formula could be faster but less fun to write :)
	def bezierPointAt(self, points, t):
		# First order points
		m11 = self.pointAt(points[0:2], points[2:4], t)
		m12 = self.pointAt(points[2:4], points[4:6], t)
		m13 = self.pointAt(points[4:6], points[6:8], t)
		# Second order points
		m21 = self.pointAt(m11, m12, t)
		m22 = self.pointAt(m12, m13, t)
		#Final point
		mp = self.pointAt(m21, m22, t)
		return mp
	
	def pointAt(self,p1,p2,t):
		x=p1[0]*(1-t) + p2[0]*t
		y=p1[1]*(1-t) + p2[1]*t 
		return [x,y]
		
	# Very approximate length of a Bezier curve, taken as the average of the direct
	# distance between anchors and the envelope (anchor->handle->handle->anchor) length.
	# A few tests show it is accurate to about 5% for "reasonable" curves.
	def approximateBezierLength(self,points):
		direct = self.distance(points[0:2], points[6:8])
		envelope1 = self.distance(points[0:2], points[2:4])
		envelope2 = self.distance(points[2:4], points[4:6])
		envelope3 = self.distance(points[4:6], points[6:8])
		envelope = envelope1 + envelope2 + envelope3
		approximateLength = (direct + envelope) / 2
		return approximateLength
	
	def distance(self,p1,p2):
		return math.sqrt((p1[0]-p2[0]) ** 2 + (p1[1]-p2[1]) ** 2)


	def printPoint(self, name, xy):
		print '%s: %3.2f,%3.2f' % (name, xy[0], xy[1])

	def segmentAngle(self,fromPoint,toPoint):
		dX = toPoint[0] - fromPoint[0]
		dY = toPoint[1] - fromPoint[1]
		
		if math.fabs(dX) < .001:
			theta = math.copysign(math.pi/2, dY)
		else:
			theta = math.atan(dY / dX)
			if dX < 0:
				theta = theta + math.pi
		return theta % (math.pi*2)

	# Computes point at signed distance and given oriented angle
	def pointAtAngleDistance(self,origin,alpha,distance):
		deltaX = distance*math.cos(alpha)
		deltaY = distance*math.sin(alpha)
		x = origin[0] + deltaX
		y = origin[1] + deltaY
		return [x,y]

def arrowHeads(image, sourcePath, size, angle, sides, close, axis):
	try:
		arrowHeadAdder = ArrowHeadAdder(image, sourcePath, size, angle, sides, close, axis)
		arrowHeadAdder.generatePathWithArrowsHeads()
        except Exception as e:
		print e.args[0]
		pdb.gimp_message(e.args[0])

	pdb.gimp_displays_flush()
	return;
#ofnuts' arrow head code ends =========================================================================================
def color_to_hex(color):
	whole_color = (color[0],color[1],color[2]);
	return '#%02x%02x%02x' % whole_color
def vector_to_line_stroke(image, vector, layer, color="#000000", width=1, capstyle="butt", joinstyle="miter", miterlimit=10, shaperendering="auto"):
    import re, tempfile
    import os, string, sys
    newelements = {
            'stroke': color,
            'stroke-width': width,
            'stroke-linecap': capstyle,
            'stroke-linejoin': joinstyle,
            'stroke-miterlimit': miterlimit,
			'shape-rendering': shaperendering,
            }
    svg = pdb.gimp_vectors_export_to_string(image, vector)
    #fix width and height to be resolution (px/inch)-independent
    svg = re.sub(r'(<svg\s[^>]*\swidth\s*=\s*)\S*"', r'\1"%dpx"' % image.width, svg, flags=re.DOTALL)
    svg = re.sub(r'(<svg\s[^>]*\sheight\s*=\s*)\S*"', r'\1"%dpx"' % image.height, svg, flags=re.DOTALL)
    svg = re.sub(r'(<path\s[^>]*)\sstroke\s*=\s*"black"', r'\1', svg, flags=re.DOTALL)
    svg = re.sub(r'(<path\s[^>]*)\sstroke-width\s*=\s*"1"', r'\1', svg, flags=re.DOTALL)
    svg = re.sub(r'(<path\s)', r'\1' + ''.join([r'%s="%s" ' % i for i in newelements.items()]), svg, flags=re.DOTALL)
    tf=tempfile.NamedTemporaryFile(mode='w', suffix='.svg', dir=gimp.directory, delete=False)
    tf.write(svg)
    tf.close()
    newlayer = pdb.gimp_file_load_layer(image, tf.name)
    os.remove(tf.name)
    image.add_layer(newlayer) #needs to be added to the image to be able to copy from
    copyname = pdb.gimp_edit_named_copy(newlayer, "stroke")
    image.remove_layer(newlayer)
    floating_sel = pdb.gimp_edit_named_paste(layer, copyname, True)
    pdb.gimp_floating_sel_anchor(floating_sel)
#=====================================================================================================================
def draw_numbers_and_arrows(image,fill_layer,number_layer,x1,y1,x2,y2,x,y,radius,fill_circle,write_number,number,font):
	#x1,y1,x2,y2 are selection bounds
	#x,y is the point where we want to find surrounding rectangle
	ret_value = 0
	if (x >= x1) and (x <= x2) and (y >= y1) and (y <= y2): #only do this if point is within selection bounds
		if fill_circle == 1 or write_number == 1:
			pdb.gimp_selection_none(image);
			
		pdb.gimp_image_select_ellipse(image,CHANNEL_OP_ADD,x-radius,y-radius,radius*2,radius*2)
		if fill_circle == 1:
			pdb.gimp_edit_fill(fill_layer,BACKGROUND_FILL)
	    
		if write_number == 1:
			pdb.gimp_selection_none(image)
			text_layer = pdb.gimp_text_layer_new(image,str(number),font,200,0)
			pdb.gimp_image_insert_layer(image,text_layer,None,-1)
			
			#when text is created it's not exactly centered in layer sometimes spaces around it is uneven
			#So here we'll crop layer so that we're left with only the text, this way it'll be dead center in our circle
			#pdb.gimp_image_select_item(image,CHANNEL_OP_REPLACE,text_layer)
			pdb.plug_in_autocrop_layer(image,text_layer)
			
			#calculate ratio to transform text into circle
			text_width = pdb.gimp_drawable_width(text_layer)
			text_height = pdb.gimp_drawable_height(text_layer)
			width_over_height_ratio = (text_width * 1.0)/text_height
			
			if width_over_height_ratio <= 1:
				halfheight = radius/2
				halfwidth = halfheight * width_over_height_ratio
			else:
				halfwidth = radius/2
				halfheight = halfwidth / width_over_height_ratio
				
			pdb.gimp_item_transform_perspective(text_layer,x-halfwidth,y-halfheight,x+halfwidth,y-halfheight,x-halfwidth,y+halfheight,x+halfwidth,y+halfheight)
			pdb.gimp_image_merge_down(image,text_layer, EXPAND_AS_NECESSARY)
			ret_value = 1 #we drew a number we return 1 to use as increment
	return ret_value
			
def python_label_arrows_from_path(image, layer, radius, fill_circle, write_number,font,draw_arrow,arrow_size,arrow_angle,arrow_axis,stroke_line_width) :
	pdb.gimp_image_undo_group_start(image)
	
	#grab selection bounding box values
	selection = pdb.gimp_selection_bounds(image);
	
	x1 = selection[1];
	y1 = selection[2];
	x2 = selection[3];
	y2 = selection[4];
	
	pdb.gimp_selection_none(image)
	
	
	#get active vectors and do some work
	active_vectors = pdb.gimp_image_get_active_vectors(image)
	num_strokes,stroke_ids = pdb.gimp_vectors_get_strokes(active_vectors)
	
	#creates new layer for arrows to be drawn on
	arrow_layer = 0;
	arrow_layer_name = "";
	if draw_arrow == 1:
		arrow_layer_name = "Draw Arrows";
		width = pdb.gimp_image_width(image)
		height = pdb.gimp_image_height(image)
		arrow_layer = pdb.gimp_layer_new(image,width,height,RGBA_IMAGE,arrow_layer_name,100,NORMAL_MODE)
		pdb.gimp_image_insert_layer(image,arrow_layer,None,-1)
		arrow_layer_name = pdb.gimp_item_get_name(arrow_layer)
		#use offnuts to path arrow heads
		bgcolor = pdb.gimp_context_get_background()
		arrowHeads(image,active_vectors,arrow_size,arrow_angle,SIDE_END,0,arrow_axis)
		arrowhead_vectors = pdb.gimp_image_get_vectors_by_name(image,'Arrow Head Vectors')
		#use svg to stroke it so it looks like stroke with line
		vector_to_line_stroke(image,arrowhead_vectors,arrow_layer,color_to_hex(bgcolor),stroke_line_width,"square","miter",10,"auto")
		vector_to_line_stroke(image,active_vectors,arrow_layer,color_to_hex(bgcolor),stroke_line_width,"square","bevel",10,"auto")
		#we're done so let's remove arrow heads vectors
		pdb.gimp_image_remove_vectors(image,arrowhead_vectors)
		
	#creates new layer for fill to be drawn on
	fill_layer = 0;
	fill_layer_name = "";
	if fill_circle == 1:
		fill_layer_name = "Draw Fill";
		width = pdb.gimp_image_width(image)
		height = pdb.gimp_image_height(image)
		fill_layer = pdb.gimp_layer_new(image,width,height,RGBA_IMAGE,fill_layer_name,100,NORMAL_MODE)
		pdb.gimp_image_insert_layer(image,fill_layer,None,-1)
		fill_layer_name = pdb.gimp_item_get_name(fill_layer)
	#creates new layer for number to be drawn on
	number_layer = 0;
	number_layer_name = "";
	if write_number == 1:
		number_layer_name = "Draw Numbers";
		width = pdb.gimp_image_width(image)
		height = pdb.gimp_image_height(image)
		number_layer = pdb.gimp_layer_new(image,width,height,RGBA_IMAGE,number_layer_name,100,NORMAL_MODE)
		pdb.gimp_image_insert_layer(image,number_layer,None,-1)
		number_layer_name = pdb.gimp_item_get_name(number_layer)
	
	
	
	
	number = 1
	for stroke_id in stroke_ids:
		type,num_points,controlpoints,closed = pdb.gimp_vectors_stroke_get_points(active_vectors,stroke_id)
		#pdb.gimp_message(str(num_points))
		#pdb.gimp_message(str(controlpoints))
		
		#for i in range(0,num_points/6): #each point has 3 control points x 2 coordinates (so 6 values)
		i = 0
		x = controlpoints[(i*6) + 2]  #third index is middle control point's x
		y = controlpoints[(i*6) + 3]  #fourth index is middle control point's y
		inc = draw_numbers_and_arrows(image,fill_layer,number_layer,x1,y1,x2,y2,x,y,radius,fill_circle,write_number,number,font)
		if write_number == 1: #if we write number we merged down so we gotta grab that layer pointer again
			number_layer = pdb.gimp_image_get_layer_by_name(image,number_layer_name) #after it's merged we gotta grab it again
		number = number+inc
	
	#if we draw anything we just select none else leave it as is as user just wants to make selection
	if fill_circle == 1 or write_number == 1:
		pdb.gimp_selection_none(image)
	
	pdb.gimp_image_undo_group_end(image)
	pdb.gimp_displays_flush()
    #return

register(
	"python_fu_label_arrows_from_path",
	"Label Arrows From Path",
	"Label Arrows From Path...",
	"Tin Tran",
	"Tin Tran",
	"Sept 2017",
	"<Image>/Python-Fu/Label Arrows From Path",
	"*",      # Create a new image, don't work on an existing one
	[
	(PF_SPINNER, "radius", "Label Radius:", 25, (1, 2000, 10)),
	(PF_TOGGLE, "fill_circle",   "Fill Circle With Background Color:", 1),
	(PF_TOGGLE, "write_number", "Write Numbers With Foreground Color:", 1),
	(PF_FONT, "number_font", "Number's Font:", "Sans Bold"),
	(PF_TOGGLE, "draw_arrow", "Draw Arrows With Background Color:", 1),
	(PF_SPINNER, "arrow_size", "Arrows Size:", 20, (1, +200, 1)),	
	(PF_SPINNER, "arrow_angle", "Arrows Angle:", 20, (1, +60, 1)),
	#(PF_OPTION, "arrow_side", "Arrows Ends:", SIDE_END, SIDE_NAMES),
	#(PF_TOGGLE, "arrow_close", "Arrows Close:", 0),
	(PF_OPTION, "arrow_axis", "Arrows Axis:", AXIS_BEST, AXIS_NAMES),
	(PF_SPINNER, "stroke_line_width", "Stroke Line Width:", 6.0, (1, 100, 0.5)),
	#(PF_COLOR, "black",  "Black point color",  (0,0,0) ),
	#(PF_COLOR, "white",  "White point color",  (255,255,255) ),
	#(PF_COLOR, "gray",  "Gray point color",  (128,128,128) )
	#(PF_FILE, "infilename", "Temp Filepath", "/Default/Path")
	#(PF_DIRNAME, "source_directory", "Source Directory", "") for some reason, on my computer when i(Tin) use PF_DIRNAME the pythonw.exe would crash
	],
	[],
	python_label_arrows_from_path)

main()
