#!/usr/bin/env python
# -*- coding: utf-8 -*-

# GIMP plugin to space/spread layers

# (c) Ofnuts 2012
#
#   History:
#
#   v0.0: 2012-06-29: Initial version
#   v0.1: 2012-11-29: Add spacing, allow overlap, and rename
#   v0.2: 2012-11-30: Fix type problem
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.


import math, sys, os
from gimpfu import *

class Arranger(object):
	def __init__(self,targetCoordinate):
		self.targetCoordinate=targetCoordinate
		
	def getLayers(self,image):
		# extract visible layers
		layers=[l for l in image.layers if l.visible]
		if len(layers) < 2:
			raise Exception('Need as least 2 visible layers (%d found)' % len(layers))
		# order by coordinates
		layers.sort(key=lambda l: l.offsets[self.targetCoordinate])
		totalLayersSize=0
		maxLayerSize=0
		for l in layers:
			layerSize=[l.width,l.height][self.targetCoordinate]
			totalLayersSize=totalLayersSize+layerSize
			if maxLayerSize < layerSize:
				maxLayerSize = layerSize
		return layers,totalLayersSize,maxLayerSize
		
	def doArrangeLayers(self,image):
		layers,totalLayersSize,maxLayerSize=self.getLayers(image);
		startOffset,minSpacing,extraPixels=self.computeSpacing(image,len(layers),totalLayersSize,maxLayerSize)
		
		nextOffset=startOffset
		for l in layers:
			offsets=[xy for xy in l.offsets]
			sizes=[xy for xy in (l.width,l.height)]
			offsets[self.targetCoordinate]=nextOffset
			l.set_offsets(offsets[0],offsets[1])
			nextOffset=nextOffset+minSpacing+sizes[self.targetCoordinate]
			if extraPixels:
				nextOffset=nextOffset+1
				extraPixels=extraPixels-1
		return;

	def arrangeLayers(self,image):
		pdb.gimp_image_undo_group_start(image)
		try:
			self.doArrangeLayers(image)
		except Exception as e:
			print e.args[0]
			pdb.gimp_message(e.args[0])

		pdb.gimp_image_undo_group_end(image)
		pdb.gimp_displays_flush()	
		
class Spreader(Arranger): 		
	def __init__(self,targetCoordinate,outer):
		super(Spreader, self).__init__(targetCoordinate)
		self.outer=outer;
	
	def computeSpacing(self,image,numLayers,totalLayersSize,maxLayerSize):
       		selected,x1,y1,x2,y2=pdb.gimp_selection_bounds(image)
		selection=[x1,y1,x2,y2]
		offsetMin=selection[self.targetCoordinate]
		offsetMax=selection[self.targetCoordinate+2]
		totalSpan=offsetMax-offsetMin
		if totalSpan < maxLayerSize:
                        raise Exception('Designated span (%dpx) smaller than larger layer size (%dpx)' % (totalSpan,maxLayerSize))
		totalSpace=totalSpan-totalLayersSize
		
		if self.outer:
			voids=numLayers+1
		else:
			voids=numLayers-1
		minSpacing=totalSpace/voids
		extraPixels=totalSpace-(minSpacing*voids)
		
		if self.outer:
			startOffset=offsetMin+minSpacing
		else:
			startOffset=offsetMin
		return startOffset,minSpacing,extraPixels	

class Spacer(Arranger): 		
	def __init__(self,targetCoordinate,spacing):
		super(Spacer, self).__init__(targetCoordinate)
		self.spacing=int(spacing);

	def computeSpacing(self,image,numLayers,totalLayersSize,maxLayerSize):
       		selected,x1,y1,x2,y2=pdb.gimp_selection_bounds(image)
		selection=[x1,y1,x2,y2]
		return selection[self.targetCoordinate],self.spacing,0	

# Different types of spacers. One could also have initialized the base spacer
# directly (I.e. Spreader(1,True,True).spread(image)) but that would be a bit 
# less readable.
class InnerHorizontalSpreader(Spreader):
	def __init__(self):
		super(InnerHorizontalSpreader, self).__init__(0,False)

class InnerVerticalSpreader(Spreader):
	def __init__(self):
		super(InnerVerticalSpreader, self).__init__(1,False)

class OuterHorizontalSpreader(Spreader):
	def __init__(self):
		super(OuterHorizontalSpreader, self).__init__(0,True)

class OuterVerticalSpreader(Spreader):
	def __init__(self):
		super(OuterVerticalSpreader, self).__init__(1,True)

class HorizontalSpacer(Spacer):
	def __init__(self,spacing):
		super(HorizontalSpacer, self).__init__(0,spacing)

class VerticalSpacer(Spacer):
	def __init__(self,spacing):
		super(VerticalSpacer, self).__init__(1,spacing)

# Main functions
def spreadHorizontallyInner(image):
	InnerHorizontalSpreader().arrangeLayers(image)
	return;
	
def spreadVerticallyInner(image):
	InnerVerticalSpreader().arrangeLayers(image)
	return;
	
def spreadHorizontallyOuter(image):
	OuterHorizontalSpreader().arrangeLayers(image)
	return;
	
def spreadVerticallyOuter(image):
	OuterVerticalSpreader().arrangeLayers(image)
	return;
	
def spaceHorizontally(image,spacing):
	HorizontalSpacer(spacing).arrangeLayers(image)
	return;
	
def spaceVertically(image,spacing):
	VerticalSpacer(spacing).arrangeLayers(image)
	return;
	
### Registration
whoiam='\n'+os.path.abspath(sys.argv[0])
author='Ofnuts'
spreadMenu='<Image>/Image/Arrange layers/Spread/'
spaceMenu='<Image>/Image/Arrange layers/Space'
copyrightYear='2012'

spreadHIDesc='Spread layers horizontally'
spreadVIDesc='Spread layers vertically'
spreadHODesc='Spread layers horizontally with outer spaces'
spreadVODesc='Spread layers vertically with outer spaces'
spaceHDesc='Space layers horizontally'
spaceVDesc='Space layers vertically'

spreadParms=[(PF_IMAGE, 'image', 'Input image', None)]
spaceParms=[(PF_IMAGE, 'image', 'Input image', None),(PF_SPINNER, "spacing", "Spacing:", 0, (-1000, 1000, 1)),]

register(
	'spread-layers-horizontal-inner',
	spreadHIDesc+whoiam,spreadHIDesc,
	author,author,
	copyrightYear,
	'Horizontally',
	'*',spreadParms,[],
	spreadHorizontallyInner,
	menu=spreadMenu
)

register(
	'spread-layers-vertical-inner',
	spreadVIDesc+whoiam,spreadVIDesc,
	author,author,
	copyrightYear,
	'Vertically',
	'*',spreadParms,[],
	spreadVerticallyInner,
	menu=spreadMenu
)

register(
	'spread-layers-horizontal-outer',
	spreadHODesc+whoiam,spreadHODesc,
	author,author,
	copyrightYear,
	'Horizontally with outer spaces',
	'*',spreadParms,[],
	spreadHorizontallyOuter,
	menu=spreadMenu
)

register(
	'spread-layers-vertical-outer',
	spreadVODesc+whoiam,spreadVODesc,
	author,author,
	copyrightYear,
	'Vertically with outer spaces',
	'*',spreadParms,[],
	spreadVerticallyOuter,
	menu=spreadMenu
)

register(
	'space-layers-horizontal',
	spaceHDesc+whoiam,spaceHDesc,
	author,author,
	copyrightYear,
	'Horizontally',
	'*',spaceParms,[],
	spaceHorizontally,
	menu=spaceMenu
)

register(
	'space-layers-vertical',
	spaceVDesc+whoiam,spaceVDesc,
	author,author,
	copyrightYear,
	'Vertically',
	'*',spaceParms,[],
	spaceVertically,
	menu=spaceMenu
)

main()
