# Auto Hatch 
# creates new layers of automatic hatching by reducing image down to a certain number of colors
#           and then calling "auto_hatch_sub_function" for each area 
# author: original Tin Tran in script-fu
#           translated/modified by Diego in python
# date: 2015/2017/2018 (version for gimp 2.9.9)
#-----------------------------------------------------------------------------------------------  

'''
# ---------------------------------------------------------------------------------

 Filter to create a new image quantized and hatched with different types of hatches 
 
 .................................................................................. 
 
 Release 1.0   initial
 
 ------------------------------------------------------------------
 COPYRIGHT NOTICE
 ------------------------------------------------------------------
 
 This program is free software you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation either version 3 of the License, or
 (at your option) any later version.  
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program if not, you can view the GNU General Public
 License version 3 at the web site http://www.gnu.org/licenses/gpl-3.0.html
 Alternatively you can write to the Free Software Foundation, Inc., 675 Mass
 Ave, Cambridge, MA 02139, USA.
 
 --------------------------------------------------------------------------------------------------------------------------------------------------
 other info
 --------------------------------------------------------------------------------------------------------------------------------------------------

 --------------------------------------------------------------------------------------------------------------------------------------------------
 the following line terminates the long initial comments
'''

#!/usr/bin/env python
from gimpfu import *
import decimal
import math
import random
import time
import datetime

############################################################################
# include, then use: erro_msg or info_msg
#---------------------------------------------------------------------------
def erro_msg (msg):
    ###
    currenth = pdb.gimp_message_get_handler()
    pdb.gimp_message_set_handler(0) # box
    gimp.message ("ERROR: "+msg)
    pdb.gimp_message_set_handler(currenth) # current
    ###
def info_msg (msg):
    ###
    currenth = pdb.gimp_message_get_handler()
    pdb.gimp_message_set_handler(2) # error console
    gimp.message ("INFO: "+msg)
    pdb.gimp_message_set_handler(currenth) # current
    ###

# ========================================================
# Ancillary Routine
# --------------------------------------------------------
def auto_hatch_sub_function (image, layer, areacolor, color, bg_color, option, rotate, percent, gridsize):
        
    #info_msg ("entered subfunction: color = "+str(color))
    #info_msg ("entered subfunction: bg_color = "+str(bg_color))
    
    # redoing the selection (maybe lost?)-----------------------------------------------------
    red = areacolor[0]
    green = areacolor[1]
    blue = areacolor[2]
    #info_msg ("area colour = "+str(red)+", "+str(green)+", "+str(blue))
    pdb.gimp_context_set_sample_threshold (0.05)
    pdb.gimp_image_select_color (image, CHANNEL_OP_REPLACE, layer, ((red, green, blue, 255)))
    # end redoing the selection --------------------------------------------------------------
     
    #START =======================================================
 
    #if percent > 50 we make percent 100-percent and swap foreground and background color
    #this is to avoid a circle having a diameter greater than the width/height of grid
    #but it should work for lines and squares as well 
    if (percent > 50):
        percent =  100 - percent 
        temp = color 
        color = bg_color 
        bg_color = temp 
    if (percent == 0): #there's a bug on zero percent for some reason
        percent = 0.01
    #create new layer
    new_layer = pdb.gimp_layer_new (image, image.width, image.height,
                                    RGBA_IMAGE, "Hatch Selection", 100.0, LAYER_MODE_NORMAL) #creates layer 2.9.9
    #insert above current layer
    image.add_layer (new_layer, 0)	 	  
    #set that layer to be active layer
    pdb.gimp_image_set_active_layer (image, new_layer)
    #*********************************
    selection = pdb.gimp_selection_save (image)
    
    #creates new image and layer to draw hatch on and display it
    gridsize = gridsize*10  #make image 10 times bigger   
    hatch_image = pdb.gimp_image_new (gridsize, gridsize, RGB)   
    hatch_layer = pdb.gimp_layer_new (hatch_image, gridsize, gridsize,
                                      RGBA_IMAGE, "Hatch", 100.0, LAYER_MODE_NORMAL) #creates layer 2.9.9
    hatch_image.add_layer (hatch_layer, 0)
    pdb.gimp_image_set_active_layer (hatch_image, hatch_layer)	  
    #pdb.gimp_display_new (hatch_image)
    #return()
 
    pdb.gimp_selection_all (hatch_image)
    pdb.gimp_context_set_foreground (bg_color)
    pdb.gimp_edit_fill (hatch_layer, FILL_FOREGROUND)     # 2.9.9
    
    #info_msg ("option = "+str(option))
    if (option == 0): #lines
        linewidth = gridsize*percent/100.0 
        if (linewidth<1):
            linewidth=1
        pdb.gimp_image_select_rectangle (hatch_image, CHANNEL_OP_REPLACE, 0, 0, gridsize, linewidth)
        pdb.gimp_context_set_foreground (color)
        pdb.gimp_edit_fill (hatch_layer, FILL_FOREGROUND)     # 2.9.9
        
    if (option == 1): #option 1 squares
        linewidth = math.sqrt((gridsize*percent/100.0)*gridsize)
#						(set! linewidth (sqrt (* (* (/ percent 100.0) height) height)))
        #info_msg ("entered subfunction: linewidth = "+str(linewidth))
        pdb.gimp_image_select_rectangle (hatch_image, CHANNEL_OP_REPLACE, 0, 0, linewidth, linewidth)
        pdb.gimp_context_set_foreground (color)
        pdb.gimp_edit_fill (hatch_layer, FILL_FOREGROUND)     # 2.9.9
              
    if (option == 2): #option 2 circle/dots
        #calculate linewidth as diameter or dot
        linewidth = (math.sqrt((gridsize*percent/100.0)*gridsize))/3.1415*2
#						 (set! linewidth (* (sqrt (/ (* (* (/ percent 100.0) height) height) 3.1415)) 2))
        linewidth = math.sqrt((gridsize*percent/100.0)*gridsize)    # try simpler
        pdb.gimp_image_select_ellipse (hatch_image, CHANNEL_OP_REPLACE, 0, 0, linewidth, linewidth)
        pdb.gimp_context_set_foreground (color)
        pdb.gimp_edit_fill (hatch_layer, FILL_FOREGROUND)     # 2.9.9
        
    if (option == 3): #option 3 triangles
        linewidth = (math.sqrt((gridsize*percent/100.0)*gridsize*2))
#					     (set! linewidth (sqrt (*(* (* (/ percent 100.0) height) height) 2)))
        pdb.gimp_image_select_polygon (hatch_image, CHANNEL_OP_REPLACE,	6, (0, 0, linewidth, 0, 0, linewidth))
        pdb.gimp_context_set_foreground (color)
        pdb.gimp_edit_fill (hatch_layer, FILL_FOREGROUND)     # 2.9.9
        
    if (option == 4): #option 4 zigzags
        linewidth = gridsize*percent/100.0
#                        (set! linewidth (* (* (/ percent 100.0) height)))        
        pdb.gimp_image_select_polygon (hatch_image, CHANNEL_OP_REPLACE,	12,
                                       (0, 0, linewidth, 0, gridsize, gridsize/2,
                                        linewidth, gridsize, 0, gridsize, gridsize-linewidth, gridsize/2))
        pdb.gimp_context_set_foreground (color)
        pdb.gimp_edit_fill (hatch_layer, FILL_FOREGROUND)     # 2.9.9
        
    if (option == 5): #option 5 fans
        linewidth = gridsize*percent/100.0/2
        if (linewidth<1):
            linewidth=1
        pdb.gimp_image_select_rectangle (hatch_image, CHANNEL_OP_REPLACE, 0, 0, gridsize/2, linewidth)
        pdb.gimp_image_select_rectangle (hatch_image, CHANNEL_OP_ADD, gridsize-linewidth, 0, linewidth, gridsize/2)
        pdb.gimp_image_select_rectangle (hatch_image, CHANNEL_OP_ADD, gridsize/2, gridsize-linewidth, gridsize/2, linewidth)
        pdb.gimp_image_select_rectangle (hatch_image, CHANNEL_OP_ADD, 0, gridsize/2, linewidth, gridsize/2)
        pdb.gimp_context_set_foreground (color)
        pdb.gimp_edit_fill (hatch_layer, FILL_FOREGROUND)     # 2.9.9
        
    if (option == 6): # option 6 rings
        linewidth = math.sqrt(gridsize*percent*1.5/100.0*gridsize/3.1415)*2
#    					 (set! linewidth (* (sqrt (/ (* (* (/ (* percent 1.5) 100.0) height) height) 3.1415)) 2))
        pdb.gimp_image_select_ellipse (hatch_image, CHANNEL_OP_REPLACE, 0, 0, linewidth, linewidth)
        pdb.gimp_context_set_foreground (color)
        pdb.gimp_edit_fill (hatch_layer, FILL_FOREGROUND)     # 2.9.9
        small_linewidth = math.sqrt(gridsize*percent*0.5/100.0*gridsize/3.1415)*2
#        		         (set! small_linewidth (* (sqrt (/ (* (* (/ (* percent 0.5) 100.0) height) height) 3.1415)) 2))
        pdb.gimp_image_select_ellipse (hatch_image, CHANNEL_OP_REPLACE, linewidth/2-small_linewidth/2, linewidth/2-small_linewidth/2,
                                                                        small_linewidth, small_linewidth)
        pdb.gimp_context_set_foreground (bg_color)
        pdb.gimp_edit_fill (hatch_layer, FILL_FOREGROUND)     # 2.9.9
        
    if (option == 7): # option 7 hollow
        linewidth = math.sqrt(gridsize*percent*1.5/100.0*gridsize)
#							(set! linewidth (sqrt (* (* (/ (* percent 1.5) 100.0) height) height)))
        pdb.gimp_image_select_rectangle (hatch_image, CHANNEL_OP_REPLACE, 0, 0, linewidth, linewidth)
        pdb.gimp_context_set_foreground (color)
        pdb.gimp_edit_fill (hatch_layer, FILL_FOREGROUND)     # 2.9.9
        small_linewidth = math.sqrt(gridsize*percent*0.5/100.0*gridsize) 
#							(set! small-linewidth (sqrt (* (* (/ (* percent 0.5) 100.0) height) height)))
        pdb.gimp_image_select_rectangle (hatch_image, CHANNEL_OP_REPLACE, 
                                         linewidth/2-small_linewidth/2, linewidth/2-small_linewidth/2,
                                         small_linewidth, small_linewidth)
        pdb.gimp_context_set_foreground (bg_color)
        pdb.gimp_edit_fill (hatch_layer, FILL_FOREGROUND)     # 2.9.9
        
    #info_msg ("linewidth = "+str(linewidth))
    
    gridsize = gridsize/10  # resize image to original gridsize
    pdb.gimp_image_scale (hatch_image, gridsize, gridsize)	  
    pdb.gimp_selection_all (hatch_image) 
    #copy this hatch layer
    pdb.gimp_edit_copy (hatch_layer)
    pdb.gimp_image_delete (hatch_image)
 
    #pattern fill and rotate 
    pdb.gimp_selection_all (image)
    (nr,patterns) = pdb.gimp_patterns_get_list ("")
    pdb.gimp_context_set_pattern (patterns[0])
    # from: (gimp-context-set-pattern (list-ref (cadr (gimp-patterns-get-list "")) 0))  ;*JT*
    pdb.gimp_edit_fill (new_layer, FILL_PATTERN)
    rotated = pdb.gimp_item_transform_rotate (new_layer, rotate*3.1415/180, True, 0, 0) 
    pdb.gimp_floating_sel_anchor (rotated)
 
    #selects the original selection, cut its inversion
    pdb.gimp_image_select_item (image, CHANNEL_OP_REPLACE, selection)
    pdb.gimp_selection_invert (image)
    pdb.gimp_edit_cut (new_layer)
    #pdb.gimp_selection_invert (image)
 
    #removes the channel after we're done
    pdb.gimp_image_remove_channel (image, selection)
 
    #set visible to true
    pdb.gimp_item_set_visible (new_layer, True)
 
    #DONE ========================================================
    
#------------------------- END OF SUB FUNCTION -----------------------------

############################################################################
#   M A I N
############################################################################

def AutoHatch299 (inImage, inLayer, widthheight,
                hatches,
                #hatch-option
                lines, squares, circles, triangles, zigzags, fans, rings, hollowsquares,
                is_components,
                start_angle,
                rotate_increment,
                gridsize, doFlatten) :
 
    pdb.gimp_context_push
    currFG = pdb.gimp_context_get_foreground()
    currBG = pdb.gimp_context_get_background()
    
    owidth = inImage.width
    oheight = inImage.height
    type = inImage.base_type
    name = inLayer.name
    if '.' in name:
        split = name.split(".", 1)
        name = split[0]
    suffix = ", auto-hatched"
    
    ts = time.time()
    st = datetime.datetime.fromtimestamp(ts).strftime('%Y-%m-%d %H-%M-%S')
    
    # calculate scale factor for the new output image
    realWidthHeight = owidth+oheight      # real size of the input image
    if widthheight == 0 :   # means the user wants to have same size as input
        widthheight = realWidthHeight                                    
    optiWidthHeight = widthheight       # size to work on to get best results
    scale = decimal.Decimal(optiWidthHeight)/decimal.Decimal(realWidthHeight)      # ratio for scaling

    rwidth=int(owidth*scale)
    rheight=int(oheight*scale)
    # note : original will be scaled back at the end  
    pdb.gimp_image_scale (inImage, rwidth, rheight)
    
    if hatches==0:   #auto
        hatches=int(optiWidthHeight/250)
    if gridsize==0:   #auto
        gridsize=int(optiWidthHeight/200)
    
# --------------------------------------------------------------------------------------

    newImage = pdb.gimp_image_new (rwidth, rheight, type)
    savedOriginal = pdb.gimp_layer_new_from_drawable(inLayer, newImage)     # background layer
    savedOriginal.mode = LAYER_MODE_NORMAL    # 2.9.9
    savedOriginal.name = "Original Image Layer"
    savedOriginal.opacity = 100.0 
    newImage.add_layer (savedOriginal, 0)
    pdb.gimp_layer_scale(savedOriginal, rwidth, rheight, False)
    pdb.gimp_item_set_visible (savedOriginal,False)
    
    newHatched = pdb.gimp_layer_new_from_drawable(inLayer, newImage)     # background layer
    newHatched.mode = LAYER_MODE_NORMAL    # 2.9.9
    newHatched.name = "Background Image Layer"
    newHatched.opacity = 100.0 
    newImage.add_layer (newHatched, 0)
    pdb.gimp_layer_scale(newHatched, rwidth, rheight, False)
    alpha=pdb.gimp_drawable_has_alpha(newHatched)
    if (alpha==False):
        pdb.gimp_layer_add_alpha (newHatched)                            #  RGBA
    pdb.gimp_image_undo_disable (newImage)                          # save time and space
    
# logic here    
    # -----------------------------------------------------------------------------------------------
    # allow space for drawing patterns
    if (newImage.width>newImage.height):
        new_width = newImage.width*2 
        new_height = newImage.width*2 
    else:
        new_width = newImage.height*2 
        new_height = newImage.height*2
    #resize canvas
    deltawidth = new_width-newImage.width
    deltaheight = new_height-newImage.height
    pdb.gimp_image_resize (newImage, new_width, new_height, deltawidth/2,  deltaheight/2)
        
    #pdb.gimp_display_new (newImage)
    #return()   
    
    #
    
    option = []
    option_size = 0
    
    #convert to indexed
 
    #puts all the options into the option array
    
    if (lines == True):
        option.append (0)
        option_size+=1
    if (squares == True):
        option.append (1)
        option_size+=1
    if (circles == True):
        option.append (2)
        option_size+=1
    if (triangles == True):
        option.append (3)
        option_size+=1
    if (zigzags == True):
        option.append (4)
        option_size+=1
    if (fans == True):
        option.append (5)
        option_size+=1
    if (rings == True):
        option.append (6)
        option_size+=1
    if (hollowsquares == True):
        option.append (7)
        option_size+=1
    # if none is set we'll default to lines
    if (option_size == 0):
        option.append (option_size)
        option_size+=1
                    
    #info_msg ("options = "+str(option))
    #info_msg ("options number = "+str(option_size))
    
    pdb.gimp_image_convert_indexed (newImage, NO_DITHER, MAKE_PALETTE, hatches, False, False, "unused palette name")    # <<<<<<<<<<<<
    #grabs color map
    pdb.gimp_context_set_sample_threshold (0)
    (nrcol,colors) = pdb.gimp_image_get_colormap (newImage)
    #info_msg ("colours = "+str(colors))

    pdb.gimp_image_convert_rgb (newImage)   # back to RGB
    
    rotate_angle = start_angle 
 
    y = hatches  # loop hatches number of times, starts from last one
    #y = 1 # test
    #while (y>0):
    while (y>0):    
    #do work here
        red = colors[int((y-1)*3)+0]
        green = colors[int((y-1)*3)+1]
        blue = colors[int((y-1)*3)+2]
        #info_msg ("colour = "+str(red)+", "+str(green)+", "+str(blue))
        pdb.gimp_context_set_sample_threshold (0.05)
        pdb.gimp_image_select_color (newImage, CHANNEL_OP_REPLACE, newHatched, ((red, green, blue, 255)))
        
        #pdb.gimp_display_new (newImage)
        
        value = max(red,green,blue)/255.0*100
        #info_msg ("value = "+str(value))
        if (value>0): #if value is non-zero we can calculate new colors
            new_red = int(red*100/value);  new_green = int(green*100/value); new_blue = int(blue*100/value)
        else:
            new_red=0; new_green=0; new_blue=0
        #info_msg ("new colour = "+str(new_red)+", "+str(new_green)+", "+str(new_blue))
        #default bg to black and enter calculation phase   
        bg_red=0; bg_green=0; bg_blue=0    
            
        if (is_components==True):
            #info_msg ("component")
            #recalculate colors and background color so that we have components
            max_channel = max (red, green, blue)   # decide which channel to break out based on highest channel
            if (max_channel == red):
                #info_msg ("max=RED")
            #do red
            #trying to break out red -------------------------------
                value = max(green,blue)/255.0*100
                if (value>0): #if value is non-zero we can calculate new colors
                    new_green=green*100/value; new_blue=blue*100/value
                else:
                    new_green=0; new_blue=0
                bg_value=100-value #bg_value is percentage to be represented by background
                if (bg_value==0):
                    bg_value=0.01 
                bg_red=red*100/bg_value 
                if (bg_red>255):		
                    bg_red=255
                    bg_represented=255*bg_value/100 
                    bg_leftover=red-bg_represented 
                    new_red=bg_leftover*100/value 
                else:								
                #bg-red can contain all reds needed so just set new_red to zero
                    new_red=0
                #end of breaking out red channel -------------------------------------
            elif (max_channel==green):
                #info_msg ("max=GREEN")
            #break out green
            #trying to break out green -------------------------------
                value = max(red,blue)/255.0*100
                if (value>0): #if value is non-zero we can calculate new colors
                    new_red=red*100/value; new_blue=blue*100/value
                else:
                    new_red=0; new_blue=0
                bg_value=100-value #bg_value is percentage to be represented by background
                if (bg_value==0):
                    bg_value=0.01 
                bg_green=green*100/bg_value
                
                if (bg_green>255):		
                    bg_green=255
                    bg_represented=255*bg_value/100 
                    bg_leftover=green-bg_represented 
                    new_green=bg_leftover*100/value 
                else:								
                #bg-green can contain all greens needed so just set new_green to zero
                    new_green=0
                #end of breaking out green channel -------------------------------------
            else:
                #info_msg ("max=BLUE")
            #break out blue
            #trying to break out blue -------------------------------
                value = max(red,green)/255.0*100
                if (value>0): #if value is non-zero we can calculate new colors
                    new_red=red*100/value; new_green=green*100/value
                else:
                    new_red=0; new_green=0
                bg_value=100-value #bg_value is percentage to be represented by background
                if (bg_value==0):
                    bg_value=0.01 
                bg_blue=green*100/bg_value 
                if (bg_blue>255):		
                    bg_blue=255
                    bg_represented=255*bg_value/100 
                    bg_leftover=blue-bg_represented 
                    new_bluen=bg_leftover*100/value 
                else:								
                #bg-blue can contain all blues needed so just set new_blue to zero
                    new_blue=0
                #end of breaking out blue channel -------------------------------------
         
        # if value is zero,we get no hatch pattern from breaking components,it's boring
        # so we'll just try to hatch over black at least we'll get pattern
        if (value==0):
            #info_msg ("value 0")
            #calculate values to use for hatch
            value =  max (red, green, blue) * 100 / 255.0  
            if (value > 0): # if value is non-zero we can calculate new colors
                new_red = red * 100 / value ;  new_green = green * 100 / value ;  new_blue = blue * 100 / value
            else:
                new_red = 0 ;  new_green =  0 ;  new_blue =  0 
        # default bg to black and enter calculation phase
            bg_red = 0 ; bg_green = 0 ; bg_blue = 0
            
        # set do-option based on selected options
        #do_option = option [int(y-1)]
        do_option = option [int(y%option_size)] 
        #info_msg ("do_option = "+str(do_option))
        
        auto_hatch_sub_function (newImage, newHatched,
                                ((int(red), int(green), int(blue))),    # try add to inform sub-function
                                ((int(new_red), int(new_green), int(new_blue))),
                                ((int(bg_red), int(bg_green), int(bg_blue))),
                                do_option,
                                rotate_angle,
                                value,
                                gridsize)
 
        rotate_angle = rotate_angle+rotate_increment 
            
        # loop control
        y -= 1 

    #pdb.gimp_display_new (newImage)
    #return
     
    #FinalL = pdb.gimp_image_merge_visible_layers (newImage, CLIP_TO_BOTTOM_LAYER)
    FinalL = pdb.gimp_layer_new_from_visible(newImage, newImage, st+" "+name+suffix)
    newImage.add_layer (FinalL, 0)            
    #pdb.gimp_item_set_visible (savedOriginal,True)
    FinalL.mode = LAYER_MODE_NORMAL    # 2.9.9
    pdb.gimp_selection_none (newImage)
    
    # addition of a PLASMA
#    newPLASMA = FinalL.copy()
#    newImage.add_layer (newPLASMA, 0)
#    newPLASMA.mode = LAYER_MODE_OVERLAY    # 2.9.9
#    newPLASMA.opacity = 90.0
#    if (is_components==True):
#        newPLASMA.opacity = 30.0
#    newPLASMA.name = "newPLASMA: Layer to hold Plasma Colours"
#    seed = random.randint (1,10000)
#    pdb.plug_in_plasma (newImage, newPLASMA,    # 2.9.9
#                         seed,  # seed
#                         5.0)   # turbolence 
    
    #pdb.gimp_display_new (newImage)
    #return
    
    if doFlatten == True:
        Outcome = pdb.gimp_image_merge_visible_layers (newImage, CLIP_TO_IMAGE) # all layers except savedOriginal
        Outcome.name = st+" "+name+suffix
        pdb.gimp_item_set_visible (savedOriginal,True)
    else:    
        Outcome = pdb.gimp_layer_new_from_visible(newImage, newImage, st+" "+name+suffix)
        newImage.add_layer (Outcome, 0)
    
    pdb.gimp_image_resize (newImage, rwidth, rheight, -deltawidth/2,  -deltaheight/2)
    pdb.gimp_layer_resize_to_image_size (Outcome)
    
    pdb.gimp_image_undo_enable (newImage)                           

    pdb.gimp_display_new (newImage)
    
    pdb.gimp_displays_flush

    pdb.gimp_context_pop
    pdb.gimp_context_set_foreground(currFG)
    pdb.gimp_context_set_background(currBG)
    
    pdb.gimp_image_scale (inImage, owidth, oheight)                  # source back to original size       
    
    #return() 
    return (Outcome)

# =====================================================================================================

register(
    "AutoHatch299",    
    "to create an auto-hatched image (from Tin's script)",   
    "This script transforms the input drawable into a Quantized Hatched outcome (from a Tin's script).",
    "Diego from a Tin's script-fu", 
    "Tin & Diego", 
    "2015 / September 2017 / January 2018 (2.9.9)",
    "AutoHatch299 (for Gimp 2.9.9)", 
    "RGB*,GRAY*", 
    [
      (PF_IMAGE,      "image",      "Input image", None),
      (PF_DRAWABLE,   "idrawable",  "Input drawable", None),
      (PF_ADJUSTMENT, "size",       "Width+Height of New Image"+"\n (0=same as input)"+
                                    "\n (2000-4000 recommended)", 3600, (0, 9600, 200)),
      (PF_ADJUSTMENT, "howmany",    "Number of hatches"+"\n (0=auto)", 0, (0, 255, 1)),
      (PF_TOGGLE,     "lines",      "Hatch with Lines", True),
      (PF_TOGGLE,     "squares",    "Hatch with Squares", True),
      (PF_TOGGLE,     "circles",    "Hatch with Circles", True),
      (PF_TOGGLE,     "triangles",  "Hatch with Triangles", True),
      (PF_TOGGLE,     "zigzags",    "Hatch with Zigzags", True),
      (PF_TOGGLE,     "fans",       "Hatch with Fans", True),
      (PF_TOGGLE,     "rings",      "Hatch with Rings", True),
      (PF_TOGGLE,     "hollows",    "Hatch with Hollow Squares", True),
      (PF_TOGGLE,     "colours",    "Break into color components instead of hatching over black", False),
      (PF_ADJUSTMENT, "angle",      "Hatch Start Angle", 45, (-360, 360, 1)),
      (PF_ADJUSTMENT, "increment",  "Hatch Increment Angle For Each Hatch", 15, (-360, 360, 1)),
      (PF_ADJUSTMENT, "grid",       "Hatch Grid Width and Height"+"\n (0=auto)", 0, (0, 100, 1)),
      (PF_TOGGLE,     "flatten",    "Flatten Image?", True),
    ], 
    [
      (PF_DRAWABLE,   "odrawable",  "Output drawable", None),  
    ],
    AutoHatch299,
    menu="<Image>/Filters/Artistic/Hatch",
    )

main()
