#!/usr/bin/env python
#-*- coding: utf-8 -*- 

# GIMP plugin to manage collections of add-ons

# (c) Sean Bogie, MareroQ, Ofnuts 2013
#
#   History:
#
#   v3.0: 2013-05-26: Rewrite and cleanup by Ofnuts:
#			- one single Python file to handle all managers
#			- user changes are handled with a configuration file
#			- can be used without a configuration file
#			- uses symlinks when possible (Linux, OSX)
#			- keeps activated addons in subfolders (allows filtering in the Gimp lists)
#			- UI simplification

#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

authors='Sean Bogie, MareroQ, Ofnuts' 

import pygtk,gtk,re,sys,os,glob,shutil,ConfigParser,codecs,copy,zipfile,time
pygtk.require('2.0')
from gimpfu import *

baseName=os.path.basename(sys.argv[0])
versioned=re.match(r'(.+)-\d+\.\d+\.py',baseName)
if versioned:
	execName=versioned.group(1)
else:
	execName=os.path.splitext(baseName)[0]
logFileName=gimp.directory+'/'+execName+'.log'

platformsWithTraceToTerminal=['linux2']

if sys.platform not in platformsWithTraceToTerminal:
	sys.stdout=open(logFileName,'w')

def trace(s):
	print s
	sys.stdout.flush()

def brushes_callback():
	call_back('brushes')
	
def dynamics_callback():
	call_back('dynamics')
	
def fonts_callback():	
	call_back('fonts')
	
def gradients_callback():	
	call_back('gradients')
	
def palettes_callback():
	call_back('palettes')
	
def patterns_callback():
	call_back('patterns')
	
def scripts_callback():
	call_back('scripts')
	
def call_back(addonType):
	trace('Starting %s manager' % addonType)
	configuration=Configuration(builtinSections)
	configuration.loadConfiguration()
	manager=AddonCollectionManager(configuration,commonConfigVars,addonType)
	manager.showMainDialog()
	manager.main()
	trace('Exiting %s manager' % addonType)

def boolean(string):
	return string.upper().strip() in ['TRUE','YES']

builtinSections={
		'all': {
			#'enable':'brushes fonts patterns'
			'enable':'brushes dynamics fonts gradients palettes patterns scripts',
			'addons_active':'{GimpUser}/{type}',
			'addons_stored':'{GimpUser}/{type}_storage',
			'menu_location':'<{Type}>',
			'menu_entry':'{Type} sets...',
			'menu_description':'Manage {type} sets...',
			'dialog_title':'{Type} sets manager',
			'use_subdirectory':'yes',
			'use_link':str('symlink' in dir(os))
			},
		'brushes':{
			'extensions':'.gbr .vbr .gih .abr .GBR .VBR .GIH'
			},
		'dynamics':{
			'extensions':'.gdyn .GDYN'
			},
		'fonts':{
			'extensions':'.ttf .otf .TTF .OTF'
			},
		'gradients':{
			'extensions':'.ggr .GGR'
			},
		'palettes':{
			'extensions':'.pal .PAL .gpl .GPL'
			},
		'patterns':{
			'extensions':'.png .pat .PNG .PAT .jpg .JPG'
			},
		'scripts':{
			'menu_location':'<Image>/Help',
			'extensions':'.scm .SCM',
			'use_subdirectory':'no',
			},
	}
	
# gradients, dynamics, scripts, palettes	
	
# Technical data specific to each type
initData={
		'brushes':{
			'gimpRefresh':'gimp_brushes_refresh',
			'gimpProcedure':'addon-manager-brushes',
			'gimpCallback':brushes_callback
			},
		'dynamics':{
			'gimpRefresh':'gimp_dynamics_refresh',
			'gimpProcedure':'addon-manager-dynamics',
			'gimpCallback':dynamics_callback
			},
		'fonts':{
			'gimpRefresh':'gimp_fonts_refresh',
			'gimpProcedure':'addon-manager-fonts',
			'gimpCallback':fonts_callback
			},
		'gradients':{
			'gimpRefresh':'gimp_gradients_refresh',
			'gimpProcedure':'addon-manager-gradients',
			'gimpCallback':gradients_callback
			},
		'palettes':{
			'gimpRefresh':'gimp_palettes_refresh',
			'gimpProcedure':'addon-manager-palettes',
			'gimpCallback':palettes_callback
			},
		'patterns':{
			'gimpRefresh':'gimp_patterns_refresh',
			'gimpProcedure':'addon-manager-patterns',
			'gimpCallback':patterns_callback
			},
		'scripts':{
			'gimpRefresh':'script-fu-refresh',
			'gimpProcedure':'addon-manager-scripts',
			'gimpCallback':scripts_callback
			}
	}
	

commonConfigVars={
		'{UserHome}':os.path.expanduser('~/')[:-1],
		'{GimpUser}':gimp.directory,
		'{GimpData}':gimp.data_directory,
		'{GimpPlugin}':gimp.plug_in_directory
	}

class Configuration(object):
	'''Class handling all the configuration data'''
	def __init__(self,defaults):
		self.configData=ConfigParser.ConfigParser()
		for section in defaults:
			self.configData.add_section(section)
			options=defaults[section]
			for option in options:
				self.configData.set(section,option,options[option])
		trace('Configuration initialized OK')
	
	def loadConfiguration(self):
		pluginDir=os.path.join(gimp.directory,'plug-ins')
		configFilePath=os.path.join(pluginDir,execName+'.ini')
		trace('Reading configuration file %s' % configFilePath)
		try:
			with codecs.open(configFilePath, 'r', encoding='utf-8') as configFile:
				self.configData.readfp(configFile)
			trace('Configuration file %s read successfully' % configFilePath)
		except Exception as e:
			trace('Configuration file %s not found or not readable: %s' % (configFilePath,e))

	def getValue(self,section,option,configVars):
		value=self.configData.get(section,option)
		if configVars:
			for v in configVars:
				value=value.replace(v,configVars[v])
		return value
	
	def getAddonTypeSection(self,addonType,configVars):
		addonTypeData={}
		for option in self.configData.options('all'):
			addonTypeData[option]=self.getValue('all',option,configVars)
		for option in self.configData.options(addonType):
			addonTypeData[option]=self.getValue(addonType,option,configVars)
		return addonTypeData
	
# Classes to handle the places where we keep the addOns
class AddonContainer(object):
	'''Parent of classes that handle the places where we keep the addons
		If link:
			if subdir
				- beginActivate(): link the storage directory
				- endDeactivate(): unlink the storage directory
			else:
				- activate(): link add-on to top storage directory
				- deactivate(): unlink add-on from top storage directory
		Otherwise (includes ZIP):
			if subdir:
				- beginActivate(): create the active subdirectory
				- activate(addon): copy to active subdirectory
				- deactivate(addon): remove copy to active subdirectory
				- endDeactivate(): remove the active subdirectory
			else:
				- activate(addon): copy to active top directory
				- deactivate(addon): remove copy to active top directory
	
	From the methods:
		beginActivate(): 	link & subdir: link storage dir; 
					!link & subdir: create subdir
					else: nop
		activate(addon): 	link & subdir: nop
					link & !subdir: link addon
					!link & !subdir: copy addon
					!link & subdir: copy addon
		endActivate:		nop
		
		beginDeactivate:	nop
		deactivate(addon): 	link & subdir: nop
					link & !subdir: unlink addon
					!link & !subdir: unlink addon
					!link & subdir: unlink addon
		
		endDeactivate():	link & subdir: unlink storage dir
					!link & subdir: remove subdir
					else: nop
	'''
	
	def __init__(self,containerPath,activeDir,extensions,subdir,link):
		self.containerPath=containerPath
		self.containerName=os.path.basename(containerPath)
		self.activeDir=activeDir
		self.extensions=extensions
		self.subdir=subdir
		self.link=link
		
	def storedAddons(self):
		'''returns the list of names of stored addons.
		This method is implemented by the derived classes'''
		pass
		
	def addonName(self,fullname):
		'''returns the base name of the addon from its stored full name
		can be redefined in derived classes'''
		return os.path.basename(fullname)

	def activeAddonsDir(self):
		'''computes the matching activated addons directory name'''
		name,ext=os.path.splitext(self.containerName)
		return self.activeDir+'/'+name
		
	def activeAddonPath(self,storedAddonName):
		'''computes the activated addon name'''
		if self.subdir:
			return self.activeAddonsDir()+'/'+self.addonName(storedAddonName)
		else:
			return self.activeAddonsDir()+'-'+self.addonName(storedAddonName)
		
	def beginActivate(self):
		# Create the directory for the active addons
		if self.subdir:
			activeAddonsDir=self.activeAddonsDir()
			if self.link:
				trace('Linking %s as %s' % (self.containerPath,activeAddonsDir))
				try:			
					os.symlink(self.containerPath,self.activeAddonsDir())
				except Exception as e:
					trace('Cannot create link %s: %s' % (activeAddonsDir,e)) 			
			else:
				trace('Creating %s' % activeAddonsDir)
				try:			
					os.mkdir(activeAddonsDir)
				except Exception as e:
					trace('Cannot create link %s: %s' % (activeAddonsDir,e)) 			
	
	def activate(self,storedAddon):
		'''Implemented by derived classes'''
		pass

	def endActivate(self):
		'''Implemented by derived classes'''
		pass
	
	def beginDeactivate(self):
		'''Implemented by derived classes'''
		pass
	
	def deactivate(self,storedAddon):
		'''Removes the addon from active storage'''
		if not (self.link and self.subdir): 
			filePath=self.activeAddonPath(storedAddon)
			try:
				os.remove(filePath) # removes link or copy
			except Exception as e:
				trace('Cannot de-activate %s: %s' % (filePath,e)) 

	def endDeactivate(self):
		if self.subdir:
			activeAddonsDir=self.activeAddonsDir()
			trace('Removing %s' % activeAddonsDir)
			try:			
				if self.link:			
					os.unlink(activeAddonsDir)
				else:
					os.rmdir(activeAddonsDir)
			except Exception as e:
				trace('Cannot remove %s' % (activeAddonsDir,e)) 
		
	
class AddonDirectory(AddonContainer):
	'''Class to handle directory with addons'''
	def __init__(self,containerPath,activeDir,extensions,subdir,link):
		super(AddonDirectory, self).__init__(containerPath,activeDir,extensions,subdir,link)
		
	def storedAddons(self):
		storedAddons=[f for f in glob.glob(self.containerPath+u'/*') if f.endswith(self.extensions)]
		trace('%d addons stored in %s' % (len(storedAddons),self.containerPath))
		return storedAddons
		
	def activate(self,storedAddon):
		activeAddonPath=self.activeAddonPath(storedAddon)
		if self.link:
			if not self.subdir:
				trace('Linking %s as %s' % (storedAddon,activeAddonPath))
				try:			
					os.symlink(storedAddon,activeAddonPath)
				except Exception as e:
					trace('Cannot create link %s: %s' % (activeAddonPath,e)) 			
		else:
			trace('Copying %s to %s' % (storedAddon,activeAddonPath))
			try:
				shutil.copyfile(storedAddon,activeAddonPath)
			except Exception as e:
				trace('Cannot copy addon to %s: %e' % (activeAddonPath,e)) 			

class AddonZip(AddonContainer):
	'''Class to handle zip with addons'''
	def __init__(self,containerPath,activeDir,extensions,subdir):
		super(AddonZip, self).__init__(containerPath,activeDir,extensions,subdir,False)
		self.zipFile=zipfile.ZipFile(containerPath)
		
	def storedAddons(self):
		storedAddons=[f for f in self.zipFile.namelist() if f.endswith(self.extensions)]
		trace('%d addons stored in %s' % (len(storedAddons),self.containerPath))
		return storedAddons

	def activate(self,storedAddon):
		activeAddonPath=self.activeAddonPath(storedAddon)
		trace('Activating %s as %s' % (storedAddon,activeAddonPath))
		activeAddon=None
		try:
			activeAddon=open(activeAddonPath,"wb")
			activeAddon.write(self.zipFile.read(storedAddon))
		except Exception as e:
			trace('Error activating %s from %s/%s: %s' % (activeAddonPath,self.containerName,storedAddon,e))
		finally:
			if activeAddon:
				activeAddon.close()

class AddonCollectionManager(object):
	
	def __init__(self,config,commonConfigVars,addonType):
		self.addonType=addonType
		self.configVars=self.extendConfigVars(commonConfigVars)
		self.config=config.getAddonTypeSection(addonType,self.configVars)
		self.data=initData[addonType]
		self.addonsActive=self.config['addons_active']
		self.addonsStored=self.config['addons_stored']
		self.extensions=tuple(self.config['extensions'].split())
		self.activelist=os.path.join(self.addonsStored,'.active')
		self.subdir=boolean(self.config['use_subdirectory'])
		self.link=boolean(self.config['use_link'])
		self.containers = self.getContainers()
		self.activeContainersNames=self.getActiveContainersNames()
		trace('Manager of %s created OK: active: %s, stored: %s, subdir: %s, link: %s, extensions: %s' % (addonType,self.addonsActive,self.addonsStored,self.subdir,self.link,self.extensions))

	def extendConfigVars(self,commonConfigVars):
		configVars=copy.copy(commonConfigVars)
		configVars['{type}']=self.addonType
		configVars['{Type}']=self.addonType.capitalize()
		return configVars
		
	def getContainers(self):
		containers={}
		trace('Searching directories as %s for %s' % (self.addonsStored+u'/*',self.addonType))
		dirs=[d for d in glob.glob(self.addonsStored+u'/*') if os.path.isdir(d)]
		trace('%d directories as %s for %s' % (len(dirs),self.addonsStored+u'/*',self.addonType))
		for d in dirs:
			containers[os.path.basename(d)]=AddonDirectory(d,self.addonsActive,self.extensions,self.subdir,self.link)
		zips=[z for z in glob.glob(self.addonsStored+u'/*.zip') if os.path.isfile(z) and zipfile.is_zipfile(z)]
		for z in zips:
			containers[os.path.basename(z)]=AddonZip(z,self.addonsActive,self.extensions,self.subdir)
		return containers
		
	def getActiveContainersNames(self):
		activeContainersNames=[]
		try:
			with codecs.open(self.activelist,'r',encoding='utf-8') as f:
				activeContainersNames=f.read().splitlines()
		except IOError:
			pass;
		return activeContainersNames
	
	def saveActiveContainersNames(self,activeContainersNames):
		try:
			with codecs.open(self.activelist,'w',encoding='utf-8') as f:
				f.write('\n'.join(activeContainersNames))
		except Exception as e:
			trace('Cannot write back active containers list to %s: %s' % (activelist,e))

	def doUpdateActiveAddons(self,deactivatedContainersNames,activatedContainersNames):
		totalUpdates=0

		deactivatedContainersLists={}
		for containerName in deactivatedContainersNames:
			addons=self.containers[containerName].storedAddons()
			totalUpdates=totalUpdates+len(addons)
			deactivatedContainersLists[containerName]=addons
			
		activatedContainersLists={}
		for containerName in activatedContainersNames:
			addons=self.containers[containerName].storedAddons()
			totalUpdates=totalUpdates+len(addons)
			activatedContainersLists[containerName]=addons
			
		currentUpdates = 0
		self.setProgress(0.)

		self.setStatus("Deactivating removed addons...")
		for containerName in deactivatedContainersLists:
			container=self.containers[containerName]
			addons=deactivatedContainersLists[containerName]
			container.beginDeactivate()
			for addon in addons:
				container.deactivate(addon)
				currentUpdates=currentUpdates+1
				self.setProgress(float(currentUpdates)/totalUpdates)
				gtk.main_iteration()
			container.endDeactivate()
				
		self.setStatus("Activating added addons...")
		for containerName in activatedContainersLists:
			container=self.containers[containerName]
			addons=activatedContainersLists[containerName]
			container.beginActivate()
			for addon in addons:
				container.activate(addon)
				currentUpdates=currentUpdates+1
				self.setProgress(float(currentUpdates)/totalUpdates)
				gtk.main_iteration()
			container.endActivate()
				
		self.setStatus("Refreshing list...")
		gtk.main_iteration()
		try:
			trace('Refresh: %s' % self.data['gimpRefresh'])
			pdb[self.data['gimpRefresh']]()
			trace('Refresh OK')
		except:
			pass
		self.setStatus('Done.')

	def clickedOK(self, widget, data=None):
		trace('OK sensed')
		oldActiveContainersNames = set(self.getActiveContainersNames())
		newActiveContainersNames = set([unicode(button.get_label()) for button in self.checkButtons if button.get_active()])
		
		activatedContainersNames=newActiveContainersNames-oldActiveContainersNames
		deactivatedContainersNames=oldActiveContainersNames-newActiveContainersNames
		
		trace('Old containers: %s' % oldActiveContainersNames)
		trace('New containers: %s' % newActiveContainersNames)
		trace('Activated containers: %s' % activatedContainersNames)
		trace('Deactivated containers: %s' % deactivatedContainersNames)
		
		self.doUpdateActiveAddons(deactivatedContainersNames,activatedContainersNames)

		self.saveActiveContainersNames(newActiveContainersNames)

	def setStatus(self,s):
		self.status.set_text(s)

	def setProgress(self,p):
		self.progressbar.set_fraction(p)
		
	def destroyMainDialog(self, widget, data=None):
		trace('Destroying main')
		gtk.main_quit()

	def showMainDialog(self):
		trace('Creating main dialog')
		self.mainDialog=gtk.Dialog()
		self.mainDialog.connect("destroy", self.destroyMainDialog)
		self.mainDialog.set_title(self.config['dialog_title'])
		self.mainDialog.set_border_width(10)
		self.mainDialog.set_size_request(300, 500)
		trace('Main dialog created')
	
		scrollBox = gtk.ScrolledWindow()
		scrollBox.set_border_width(10)
		scrollBox.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
		self.mainDialog.vbox.pack_start(scrollBox, True, True, 0)
		scrollBox.show()
		trace('Scrollbox created')
		self.status = gtk.Label("(Make your selections above)")
		self.mainDialog.vbox.pack_start(self.status, False, False, 0)
		self.progressbar = gtk.ProgressBar()
		self.mainDialog.vbox.pack_start(self.progressbar, False, False, 0)
		self.status.show()
		self.progressbar.show()

		vbox = gtk.VBox()
		self.checkButtons=[]
		for containerName in sorted(self.containers.keys()):
			button=gtk.CheckButton(containerName)
			button.set_active(containerName in self.activeContainersNames)
			self.checkButtons.append(button)
			vbox.pack_start(button, False, False, 1)
			button.show()
		scrollBox.add_with_viewport(vbox)
		vbox.show()
		trace('VBox created')

		cancelButton = gtk.Button(stock=gtk.STOCK_CLOSE)
		cancelButton.connect_object("clicked", gtk.Widget.destroy, self.mainDialog)
		self.mainDialog.action_area.pack_start(cancelButton, True, True, 0)
		cancelButton.show()
		trace('Cancel button created')

		okButton = gtk.Button(stock=gtk.STOCK_OK)
		okButton.connect("clicked", self.clickedOK)
		self.mainDialog.action_area.pack_start(okButton, True, True, 0)
		okButton.show()
		trace('OK button created')

		self.mainDialog.show()
		gtk.main_iteration()

	def main(self):
		gtk.main()
        
	def register(self):
	        register(
			self.data['gimpProcedure'],
			self.config['menu_description'],
			'Add-on manager. This code takes no arguments, but isn\'t meant to be called programmatically.',
			authors,authors,'2013',
			self.config['menu_entry'],
			'',[],[],
			self.data['gimpCallback'],
			menu=self.config['menu_location']
		)

### Initialization
configuration=Configuration(builtinSections)
configuration.loadConfiguration()

# Initialize the managers listed in the ini file
enabled=configuration.getValue('all','enable',None).split()
for addonType in initData:
	if addonType in enabled:
		trace('Creating manager for %s' % addonType)
		manager=AddonCollectionManager(configuration,commonConfigVars,addonType)
		manager.register()
		
main()
