#!/usr/bin/env python

'''
Gimp plugin "Uncrop side"
Increase image/canvas size and synthesize band from edge of original.
Author:
david marsden, dm
Version:
1.0 dm 20/10/2018
License:
This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
The GNU Public License is available at
http://www.gnu.org/copyleft/gpl.html
The effect for users:  
widens the field of view, maintaining perspective of original
Should be undoable, except for loss of selection.
Should work on any image type, any count of layers and channels (although only active layer is affected.)
Programming notes:
Scheme uses - in names, python uses _
IN: Nothing special.  The selection is immaterial but is not preserved.
OUT larger layer and image.  All other layers not enlarged.
'''

from gimpfu import *

gettext.install("resynthesizer", gimp.locale_directory, unicode=True)
    
def resizeImage(image, percentEnlargeParam, side):
    # resize image by percent (converted to pixel units)
    deltaFraction = (percentEnlargeParam / 100) + 1.0
    priorWidth = pdb.gimp_image_width(image)
    priorHeight = pdb.gimp_image_height(image)
    newWidth = priorWidth
    newHeight = priorHeight
    OffX = 0
    OffY = 0    
    if   side == 0:
         newWidth = priorWidth * deltaFraction
         newHeight = priorHeight
         OffX = newWidth - priorWidth
         OffY = 0
         direction = 6
    elif side == 1:
         newWidth = priorWidth *  deltaFraction
         newHeight = priorHeight
         OffX = 0
         OffY = 0
         direction = 6
    elif side == 2:
         newWidth = priorWidth
         newHeight = priorHeight * deltaFraction
         OffX = 0
         OffY = newHeight - priorHeight
         direction = 7
    elif side == 3:
         newWidth = priorWidth
         newHeight = priorHeight * deltaFraction
         OffX = 0
         OffY = 0
         direction = 7       
    pdb.gimp_image_resize(image, newWidth, newHeight, OffX, OffY)
    
        
def uncrop_side(orgImage, drawable, percentEnlargeParam, sampleWidth, side, direction=6):
    '''
    Create frisket stencil selection in a temp image to pass as source (corpus) to plugin resynthesizer,
    which does the substantive work.
    '''
    
    if not pdb.gimp_item_is_layer(drawable):
      pdb.gimp_message(_("A layer must be active, not a channel."))
      return
      
    pdb.gimp_image_undo_group_start(orgImage)
    
    # copy original into temp for later use
    tempImage = pdb.gimp_image_duplicate(orgImage)
    if not tempImage:
        raise RuntimeError, "Failed duplicate image"
    
    '''
    Prepare target: enlarge canvas and select the new, blank area
    '''
    
    # Save original bounds to later select outer area
    pdb.gimp_selection_all(orgImage)
    selectAllPrior = pdb.gimp_selection_save(orgImage)
    # Resize image alone doesn't resize layer, so resize layer also
    resizeImage(orgImage, percentEnlargeParam, side)
    pdb.gimp_layer_resize_to_image_size(drawable)
    pdb.gimp_image_select_item(orgImage, CHANNEL_OP_REPLACE, selectAllPrior)
    # select outer area, the new blank canvas.
    pdb.gimp_selection_invert(orgImage)
    # Assert target image is ready.                  
    
    #Prepare source (corpus) layer, an area at edge of original, in a dupe.
    
    # Working with the original size.
    # Could be alpha channel transparency
    workLayer = pdb.gimp_image_get_active_layer(tempImage)
    if not workLayer:
        raise RuntimeError, "Failed get active layer"
    
    pdb.gimp_selection_all(tempImage)
    priorWidth = pdb.gimp_image_width(tempImage)
    priorHeight = pdb.gimp_image_height(tempImage)
    deltaWidth =  priorWidth
    deltaHeight = priorHeight
    OffX = 0
    OffY = 0    
    if   side == 0:    
         deltaWidth = sampleWidth
         deltaHeight = priorHeight  
         OffX = 0
         OffY = 0
         direction = 6
    elif side == 1:
         deltaWidth = sampleWidth
         deltaHeight = priorHeight
         OffX = priorWidth - sampleWidth
         OffY = 0
         direction = 6
    elif side == 2:
         deltaWidth =  priorWidth
         deltaHeight = sampleWidth
         OffX = 0
         OffY = 0
         direction = 7
    elif side == 3:
         deltaWidth =  priorWidth
         deltaHeight = sampleWidth
         OffX = 0
         OffY = priorHeight - sampleWidth
         direction = 7       
    pdb.gimp_image_crop(tempImage, deltaWidth, deltaHeight, OffX, OffY)
    pdb.gimp_selection_all(tempImage)    
    pdb.gimp_selection_invert(tempImage)    # invert interior selection into a frisket
    # Note that v1 resynthesizer required an inverted selection !!
    # No need to crop corpus to save memory.
      
    # Note that the API hasn't changed but use_border param now has more values.
    # border param=5 means inside out direction. 5+0=5 outward concentric, 5+1=6 outward from sides, 5+2=7 outward from above and below.
    pdb.plug_in_resynthesizer(orgImage, drawable, 0,0,direction, workLayer.ID, -1, -1, 0.0, 0.117, 16, 500)
    
    # Clean up. 
    # Any errors now are moot.
    pdb.gimp_selection_none(orgImage)
    pdb.gimp_image_remove_channel(orgImage, selectAllPrior)
    pdb.gimp_image_undo_group_end(orgImage)
    pdb.gimp_displays_flush()
    gimp.delete(tempImage)  # Comment out to debug corpus creation.


register(
    "python_fu_uncrop_side",
    N_("Enlarge image by synthesizing a border that matches the edge, maintaining perspective."),
    "Requires separate resynthesizer plugin.",
    "David Marsden",
    "Copyright 2018 David Marsden",
    "2018",
    N_("Uncrop side..."),
    "RGB*, GRAY*",
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_DRAWABLE, "drawable", "Input drawable", None),
      (PF_SLIDER, "percentEnlargeParam", _("Percent enlargement"), 5, (1, 20, 1)),
      (PF_SLIDER, "sampleWidth", _("Sample Width"), 50, (10, 100, 1)),
      (PF_OPTION, "side", "Side", 0, ("L", "R", "T", "B"))
    ],
    [],
    uncrop_side,
    menu="<Image>/Filters/Enhance",
    domain=("resynthesizer", gimp.locale_directory)
    )

main()
