#!/usr/bin/env python

# gimp10_constantize Rel 3
# Created by Tin Tran 
# Comments directed to http://gimplearn.net
# allows user to choose an .scm or a .py file and it'll replace all occurences of old constant names with GIMP 10 constant names.
# it'll make a backup copy with _backup appended to file name and overwrite the file with GIMP 10 constants. 
#
# License: GPLv3
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY# without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# To view a copy of the GNU General Public License
# visit: http://www.gnu.org/licenses/gpl.html
#
#
# ------------
#| Change Log |
# ------------
# Rel 1: Initial release.
# Rel 2: Added MASK_TYPES
# Rel 3: Added BUCKET_FILLS
from gimpfu import *
import os
from shutil import copyfile #just used later to quickly make a backup copy

import stat
#just friendly to make it executable if it's py file, if it fails well it's not a big deal we tried.
def make_executable(filename):
	try:
		st = os.stat(filename)
		os.chmod(filename, st.st_mode | stat.S_IXUSR | stat.S_IXGRP | stat.S_IXOTH)
	except:
		pass #just ignore and move on.

def minus_to_underscore(name):
	return name.replace("-","_")

gimp10_layer_modes = ['LAYER-MODE-NORMAL-LEGACY','LAYER-MODE-DISSOLVE','LAYER-MODE-BEHIND-LEGACY','LAYER-MODE-MULTIPLY-LEGACY','LAYER-MODE-SCREEN-LEGACY','LAYER-MODE-OVERLAY-LEGACY','LAYER-MODE-DIFFERENCE-LEGACY','LAYER-MODE-ADDITION-LEGACY','LAYER-MODE-SUBTRACT-LEGACY','LAYER-MODE-DARKEN-ONLY-LEGACY','LAYER-MODE-LIGHTEN-ONLY-LEGACY','LAYER-MODE-HSV-HUE-LEGACY','LAYER-MODE-HSV-SATURATION-LEGACY','LAYER-MODE-HSL-COLOR-LEGACY','LAYER-MODE-HSV-VALUE-LEGACY','LAYER-MODE-DIVIDE-LEGACY','LAYER-MODE-DODGE-LEGACY','LAYER-MODE-BURN-LEGACY','LAYER-MODE-HARDLIGHT-LEGACY','LAYER-MODE-SOFTLIGHT-LEGACY','LAYER-MODE-GRAIN-EXTRACT-LEGACY','LAYER-MODE-GRAIN-MERGE-LEGACY','LAYER-MODE-COLOR-ERASE-LEGACY','LAYER-MODE-OVERLAY','LAYER-MODE-LCH-HUE','LAYER-MODE-LCH-CHROMA','LAYER-MODE-LCH-COLOR','LAYER-MODE-LCH-LIGHTNESS','LAYER-MODE-NORMAL','LAYER-MODE-BEHIND','LAYER-MODE-MULTIPLY','LAYER-MODE-SCREEN','LAYER-MODE-DIFFERENCE','LAYER-MODE-ADDITION','LAYER-MODE-SUBTRACT','LAYER-MODE-DARKEN-ONLY','LAYER-MODE-LIGHTEN-ONLY','LAYER-MODE-HSV-HUE','LAYER-MODE-HSV-SATURATION','LAYER-MODE-HSL-COLOR','LAYER-MODE-HSV-VALUE','LAYER-MODE-DIVIDE','LAYER-MODE-DODGE','LAYER-MODE-BURN','LAYER-MODE-HARDLIGHT','LAYER-MODE-SOFTLIGHT','LAYER-MODE-GRAIN-EXTRACT','LAYER-MODE-GRAIN-MERGE','LAYER-MODE-VIVID-LIGHT','LAYER-MODE-PIN-LIGHT','LAYER-MODE-LINEAR-LIGHT','LAYER-MODE-HARD-MIX','LAYER-MODE-EXCLUSION','LAYER-MODE-LINEAR-BURN','LAYER-MODE-LUMA-DARKEN-ONLY','LAYER-MODE-LUMA-LIGHTEN-ONLY','LAYER-MODE-LUMINANCE','LAYER-MODE-COLOR-ERASE','LAYER-MODE-ERASE','LAYER-MODE-MERGE','LAYER-MODE-SPLIT','LAYER-MODE-PASS-THROUGH','LAYER-MODE-REPLACE','LAYER-MODE-ANTI-ERASE']
def to_gimp10_scmlayermode(scmlayermode): #
	legacy_mode = 'LAYER-MODE-' + scmlayermode.replace('-MODE','') + '-LEGACY'
	new_mode = 'LAYER-MODE-' + scmlayermode.replace('-MODE','')
	if legacy_mode in gimp10_layer_modes:
		return legacy_mode
	else:
		return new_mode

#gimp10 blendmodes: BLEND-FG-BG-RGB (0), BLEND-FG-BG-HSV (1), BLEND-FG-TRANSPARENT (2), BLEND-CUSTOM (3)
def to_gimp10_scmblendmode(scmblendmode):
	return 'BLEND-' + scmblendmode.replace('-MODE','')

gimp10_fill_modes = ['FILL-FOREGROUND','FILL-BACKGROUND', 'FILL-WHITE', 'FILL-TRANSPARENT', 'FILL-PATTERN']
def to_gimp10_scmfillmode(scmfillmode):
	newfill = 'FILL-' + scmfillmode.replace('-FILL','')
	if newfill in gimp10_fill_modes:
		return newfill
	else: #it's probably NO-FILL which doesn't have a value in gimp 10.... TODO: how do we handle this. just return NO-FILL for now
		return 'NO-FILL'
#gimp10 mask constants { ADD-MASK-WHITE (0), ADD-MASK-BLACK (1), ADD-MASK-ALPHA (2), ADD-MASK-ALPHA-TRANSFER (3), ADD-MASK-SELECTION (4), ADD-MASK-COPY (5), ADD-MASK-CHANNEL (6) }
#old gimp mask         { ADD-WHITE-MASK (0), ADD-BLACK-MASK (1), ADD-ALPHA-MASK (2), ADD-ALPHA-TRANSFER-MASK (3), ADD-SELECTION-MASK (4), ADD-COPY-MASK (5), ADD-CHANNEL-MASK (6) }
def to_gimp10_scmmasktype(scmmasktype):
	return scmmasktype.replace('-MASK','').replace('-','-MASK-')
#gimp10 desaturate modes { DESATURATE-LIGHTNESS (0), DESATURATE-LUMA (1),  DESATURATE-AVERAGE (2), DESATURATE-LUMINANCE (3), DESATURATE-VALUE (4) }
#old desaturate modes    { DESATURATE-LIGHTNESS (0), DESATURATE-LUMINOSITY (1) DESATURATE-AVERAGE (2)  DESATURATE-LUMINOSITY (1),  }
#TODO: possibly return 'DESATURATE-LUMA' when it's DESATURATE-LUMINOSITY? let's wait to see if any one reports errors.

#gimp10 bucket fill  { BUCKET-FILL-FG (0), BUCKET-FILL-BG (1), BUCKET-FILL-PATTERN (2) }
#oldgimp bucket fill { FG-BUCKET-FILL (0), BG-BUCKET-FILL (1), PATTERN-BUCKET-FILL (2) }
def to_gimp10_scmbucketfill(scmbucketfill):
	return "BUCKET-FILL-" + scmbucketfill.replace("-BUCKET-FILL","")

#gimp10 gradienttypes:{ GRADIENT-LINEAR (0), GRADIENT-BILINEAR (1), GRADIENT-RADIAL (2), GRADIENT-SQUARE (3), GRADIENT-CONICAL-SYMMETRIC (4), GRADIENT-CONICAL-ASYMMETRIC (5), GRADIENT-SHAPEBURST-ANGULAR (6), GRADIENT-SHAPEBURST-SPHERICAL (7), GRADIENT-SHAPEBURST-DIMPLED (8), GRADIENT-SPIRAL-CLOCKWISE (9), GRADIENT-SPIRAL-ANTICLOCKWISE (10) }
#gradient types above seem to remain unchanged from old gimp version

#oldgimp's imagetype for new layer                        { RGB-IMAGE (0), RGBA-IMAGE (1), GRAY-IMAGE (2), GRAYA-IMAGE (3), INDEXED-IMAGE (4), INDEXEDA-IMAGE (5) }
#gimp 10 imagetype The layer type(unchanged we're good :) { RGB-IMAGE (0), RGBA-IMAGE (1), GRAY-IMAGE (2), GRAYA-IMAGE (3), INDEXED-IMAGE (4), INDEXEDA-IMAGE (5) }

#.scm constant names
#scm_old_image_modes = ['RGB','GRAY','INDEXED'] #these are the same in GIMP 2.10 no need to change
scm_old_layer_modes = ['NORMAL-MODE','DISSOLVE-MODE','BEHIND-MODE','MULTIPLY-MODE','SCREEN-MODE','OVERLAY-MODE','DIFFERENCE-MODE','ADDITION-MODE','SUBTRACT-MODE','DARKEN-ONLY-MODE','LIGHTEN-ONLY-MODE','HUE-MODE','SATURATION-MODE','COLOR-MODE','VALUE-MODE','DIVIDE-MODE','DODGE-MODE','BURN-MODE','HARDLIGHT-MODE','SOFTLIGHT-MODE','GRAIN-EXTRACT-MODE','GRAIN-MERGE-MODE','COLOR-ERASE-MODE','ERASE-MODE','REPLACE-MODE','ANTI-ERASE-MODE']
scm_old_blend_modes = ['FG-BG-RGB-MODE','FG-BG-HSV-MODE','FG-TRANSPARENT-MODE','CUSTOM-MODE']
scm_old_fill_modes = ['FOREGROUND-FILL','BACKGROUND-FILL','WHITE-FILL','TRANSPARENT-FILL','PATTERN-FILL','NO-FILL']
scm_old_mask_types = ['ADD-WHITE-MASK','ADD-BLACK-MASK','ADD-ALPHA-MASK','ADD-ALPHA-TRANSFER-MASK','ADD-SELECTION-MASK','ADD-COPY-MASK','ADD-CHANNEL-MASK']
scm_old_bucket_fills = ['FG-BUCKET-FILL','BG-BUCKET-FILL','PATTERN-BUCKET-FILL']
#convert to new modes below based on rules defined in mapped functions
scm_new_layer_modes = map(to_gimp10_scmlayermode,scm_old_layer_modes)
scm_new_blend_modes = map(to_gimp10_scmblendmode,scm_old_blend_modes)
scm_new_fill_modes = map(to_gimp10_scmfillmode,scm_old_fill_modes)
scm_new_mask_types = map(to_gimp10_scmmasktype,scm_old_mask_types)
scm_new_bucket_fills = map(to_gimp10_scmbucketfill,scm_old_bucket_fills)

#.py constant names dynamically converted from .scm values declared above
py_old_layer_modes = map(minus_to_underscore,scm_old_layer_modes)
py_old_blend_modes = map(minus_to_underscore,scm_old_blend_modes)
py_old_fill_modes = map(minus_to_underscore,scm_old_fill_modes)
py_old_mask_types = map(minus_to_underscore,scm_old_mask_types)
py_old_bucket_fills = map(minus_to_underscore,scm_old_bucket_fills)

py_new_layer_modes = map(minus_to_underscore,scm_new_layer_modes)
py_new_blend_modes = map(minus_to_underscore,scm_new_blend_modes)
py_new_fill_modes = map(minus_to_underscore,scm_new_fill_modes)
py_new_mask_types = map(minus_to_underscore,scm_new_mask_types)
py_new_bucket_fills = map(minus_to_underscore,scm_new_bucket_fills)

#we can pass the below function a string, old values and new values and it'll replace occurences of old values with new values
def replacewith(s,oldvalues,newvalues):
	new_s = s
	for i in range(0,len(oldvalues)):
		new_s = new_s.replace(oldvalues[i],newvalues[i])
	return new_s



def python_gimp10_constantize_tt(image, layer, filefullpath):
	#pdb.gimp_image_undo_group_start(image)
	#pdb.gimp_context_push()
	
	#make sure it's a .scm file or .py file
	if filefullpath.endswith(".scm") or filefullpath.endswith(".py"):
		pass
	else:
		raise ValueError('input file must be .scm or .py file')

	#back_up file as filefullpath with '_backup' appended at the end
	backup_filefullpath = filefullpath + '_backup'
	copyfile(filefullpath,backup_filefullpath) #back it up.

	#open file and read it in to our lines to process
	f = open(filefullpath,"r")
	lines = f.readlines()
	f.close()
	#do work of replacing constants in files
	if filefullpath.endswith(".scm"): #.scm file
		f = open(filefullpath,"w")
		for line in lines:
			line = replacewith(line,scm_old_layer_modes,scm_new_layer_modes)
			line = replacewith(line,scm_old_blend_modes,scm_new_blend_modes)
			line = replacewith(line,scm_old_fill_modes,scm_new_fill_modes)
			line = replacewith(line,scm_old_mask_types,scm_new_mask_types)
			line = replacewith(line,scm_old_bucket_fills,scm_new_bucket_fills)
			f.write(line)
		f.close()
	elif filefullpath.endswith(".py"): #.py file
		f = open(filefullpath,"w")
		for line in lines:
			line = replacewith(line,py_old_layer_modes,py_new_layer_modes)
			line = replacewith(line,py_old_blend_modes,py_new_blend_modes)
			line = replacewith(line,py_old_fill_modes,py_new_fill_modes)
			line = replacewith(line,py_old_mask_types,py_new_mask_types)
			line = replacewith(line,py_old_bucket_fills,py_new_bucket_fills)
			f.write(line)
		f.close()
		make_executable(filefullpath)
	
	#pdb.gimp_context_pop()
	#pdb.gimp_image_undo_group_end(image)
	#pdb.gimp_displays_flush()
    #return

register(
	"python_fu_gimp10_constantize_tt",                           
	"GIMP 10 constantizes a file",
	"GIMP 10 constantizes a file",
	"Tin Tran",
	"Tin Tran",
	"September 2018",
	"<Image>/Python-Fu/GIMP10 constantize...",             #Menu path
	"RGB*, GRAY*", 
	[
	(PF_FILE, "filefullpath", ".scm or .py file to GIMP10 constantize", 0),
	],
	[],
	python_gimp10_constantize_tt)

main()
