#!/usr/bin/env python
# -*- coding: utf-8 -*-

# License: GPLv3
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY# without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# To view a copy of the GNU General Public License
# visit: http://www.gnu.org/licenses/gpl.html

#Author:
#lloyd konneker, lkk, bootch at nc.rr.com

# ------------
#| Change Log |
# ------------

# Ver 0.1 12/2009 Initial version.
# Ver 0.2 2019 input-format:  TGA, PBM, PNM, PGM, PPM or BMP. rename png to bmp, add location autotrace, add   + " " to command
# Ver 0.3 add output directory to save temp file, replace "Smoothing" (PF_OPTION) to "Corner-threshold" (PF_SPINNER)  
# Ver 0.4 - resignation from the selection of directories in order to eliminate the problem of spaces in names (which autotrace does not tolerate),
#		  - add options "Path visible? ",
#         - the rendered image(layer and path) is named as the source layer (accepts spaces - because the name is given after the autotrace has finished)

from gimpfu import *
import subprocess
import sys, os, gettext

def sys_file(f_name):
  sys_encoded = sys.getfilesystemencoding()
  if os.name == 'nt': coded = f_name.encode(sys_encoded)
  else: coded = f_name
  return(coded)
  
whoiam = os.path.abspath(sys.argv[0])
# initialize internationalisation in a user-locale (by rob_brz1)
locale_directory = os.path.join(os.path.dirname(whoiam), 'locale')
gettext.install( "plugin-trace", locale_directory, unicode=True )

def colorcountArgs(colorcountParam):
  '''
  Color count must be reduced from 1-256.
  Value 0 means 256 to autotrace.  It does NOT mean: don't reduce colors!!!.
  IE autotrace uses 8-bit color and must reduce color to that range.
  I decided not to futz with recoding 256 to zero.
  The user only sees 1-255.
  1 doesn't always make sense (outlines with 1 color?), but works.
  '''
  return " --color-count " + str(colorcountParam) + " "

def tracetypeArgs(tracetypeParam):
  '''
  User choice of type of tracing (outline and centerline).
  !!! Note that preserve-width is NOT a parameter of centerline tracing, as some docs suggest:
  --centerline --preserve-width --width-weight-factor 8.0 " # trace varying width centerline
  '''
  if tracetypeParam == 1:
    return "--centerline " # trace thin centerline
  else:
    return " "  # autotrace defaults to trace outline

def despeckleArgs(despeckleParam):
  '''
  Note autotrace enforces range 1-20 on despeckle-level, 0.0-8.0 on despeckle-tightness.
  
  Note despeckle-level is NOT a radius, but the power of 2 for the speckle size:
  1 gives some size block (say 2) and 2 gives twice as many pixels (say 4)
  and 3 gives twice again (say 8).
  See despeckle.c: despeckle() in autotrace package.
  
  !!! Most importantly, despeckle-level and despeckle-tightness are coupled.
  Level stipulates the size of feature.
  Level and tightness stipulate the color difference of features.
  Color difference to despeckle = 256/(1+ tightness*level.)
  Note below that level increases but tightness increases then decreases
  because it multiplies level.
  
  Tightness 0 means that every color difference can be despeckled,
  ie a black speckle can go to white (difference of 256).
  '''
  if despeckleParam == 0:   # Less: 2**3 = 8 pixels = 3 pixel diameter
    return "--despeckle-level 3 --despeckle-tightness 0.0 "
  elif despeckleParam == 1: # Moderate: 2**5 = 32 pixels = 6 pixel diameter
    return "--despeckle-level 5 --despeckle-tightness 2.0 "  
  else:                     # Most: 2**8 = 256 pixels = 16 pixel diameter
    return "--despeckle-level 8 --despeckle-tightness 0.0 "

def smoothingArgs(smoothingParam):
  params = " --remove-adjacent-corners " + "--corner-threshold " + str(smoothingParam)
  return params
  
'''
These are unadulterated autotrace parameters.
Experiments show that these are mostly useless or hard to use.
'''
def cornersArgs(cornerthresholdParam):
  '''
  Always remove adjacent corners.
  I assume that the alternative is only useful for rare cases.
  TBD understand the other parameters and use them if defaults are not appropriate.
  '''
  return "--corner-threshold " + str(cornerthresholdParam) + " --remove-adjacent-corners "

def fittingArgs(fittingParam):
  '''
  The args that affect fitting of splines.
  
  This is the param that determines when a fitted spline is close enough.
  A float, in pixel units, of the max orthogonal distance between fitted spline and original curve.
  TBD other parameters for this.
  '''
  return "--error-threshold " + str(fittingParam) + " "

def iterationArgs(iterationParam):
  '''
  Smoothing iterations before spline fitting.
  Smoothing is done in the pixel domain??
  '''
  return "--filter-iterations " + str(iterationParam) + " "


def plugin_main(image, layer, saveas, colorcountParam,
     tracetypeParam, despeckleParam, smoothingParam, pathParam, vis, rem_bmp):  # , fittingParam, iterationParam):

  # if link.find('') == -1 : return
	
  pdb.gimp_image_set_active_layer(image, layer)
  oldname = layer.name
  if saveas==0:
    tempfilename = pdb.gimp_temp_name("bmp")
  if saveas==1:
    tempfilename = pdb.gimp_temp_name("ppm")  

  tempsvgfilename = pdb.gimp_temp_name("svg")

  # Save in temporary.  Note: empty user entered file name
 
  tempimage = pdb.gimp_image_duplicate(image)
  if not tempimage:
    raise RuntimeError

  tempdrawable = pdb.gimp_image_get_active_drawable(image)
  
  pdb.gimp_progress_set_text ("Saving a copy")
  layer.name = layer.name.decode("utf-8")
 
  tempsvgfilename = pdb.gimp_temp_name("svg")    
  if saveas==0:
 
    xchange_utf = tempfilename
    gimp.pdb.file_bmp_save(tempimage, tempdrawable, xchange_utf, xchange_utf, run_mode=RUN_INTERACTIVE)

  if saveas==1:

    pdb.gimp_layer_flatten(tempdrawable)
    xchange_utf = tempfilename  
    pdb.file_ppm_save(tempimage, tempdrawable, xchange_utf, xchange_utf, 0)
	
  '''
  Open the temporary svg file.  
  1) to catch possible os errors early
  2) to redirect stdout of the command to this file descriptor
  '''
  try:
    outfile = open(tempsvgfilename, 'w')
  except:
    raise RuntimeError, "Could not open temporary svg file: " + tempsvgfilename
    
  '''
  Cat command string for autotrace.
  Autotrace out defaults to stdout, we later connect stdout to outfile
  (instead of using an arg to autotrace -output-file foo).
  !!! Note spaces are important, between args and before file name
  '''
  xchange_sys = sys_file(xchange_utf)
  
  if os.name == 'nt':
    binDir="autotrace.exe"
  else:
    binDir="/usr/bin/autotrace"
	
  command = binDir \
  + colorcountArgs(colorcountParam) \
  + tracetypeArgs(tracetypeParam) \
  + despeckleArgs(despeckleParam) \
  + smoothingArgs(smoothingParam) \
  + " " + "-output-format svg -output-file" \
  + " " + tempsvgfilename \
  + " " + xchange_sys

  print command, tempsvgfilename
  
  '''
  Invoke autotrace.
  Child process proceeds independently. Then wait for its completion, so we can kill it if it hangs.
  autotrace can hang if user enters parameters too aggressive (eg many colors, fine grained tracing)
  shell=False : we don't need shell (filename globbing or redirection) but args are a sequence, not a string
  shell=True: use a string
  '''
  pdb.gimp_progress_set_text ("Tracing")
  pdb.gimp_progress_pulse()

  child = subprocess.Popen(command, shell=True, stdout=outfile)
  child.communicate() # wait for child to terminate

  '''
  Limitation of gimp: seems to hang on importing thousands of paths from large SVG.  See Bugzilla 604175.
  So rather than hang Gimp, preclude user from continuing if SVG file too large
  AND user wants to import paths.
  '''
  if os.path.getsize(tempsvgfilename) > 600100 and pathParam == 1:  
    # Magic number, wag for file size as of 2009, my computer, etc.
    pdb.gimp_message("Trace plugin error: SVG file has too many individual paths to import.")
    return  # Leaves files in gimp/tmp, to get cleaned later
    
  resolutionx, resolutiony = pdb.gimp_image_get_resolution(tempimage) # Same resolution as original

  pdb.gimp_progress_set_text ("Rendering")

  try:  # Missing file is usually autotrace failure of unknown reason

    if pathParam==3:
	
      #pdb.gimp_path_import(tempimage, tempsvgfilename, TRUE, TRUE)
      svgimage = pdb.file_svg_load(tempsvgfilename, "", layer.width, layer.height, 0, 1, run_mode=RUN_INTERACTIVE)

    else:  
      svgimage = pdb.file_svg_load(tempsvgfilename, "", resolutionx, layer.width, layer.height, pathParam)

  except RuntimeError:
    pdb.gimp_message("The underlying Autotrace program failed for unknown reasons, possibly not enough resources.\
      Try reducing colors, lower quality, or more despeckling.")
	
  if vis==1:
    for vector in svgimage.vectors:
      pdb.gimp_vectors_set_visible(vector, True)
	  
  for vector in svgimage.vectors:
    vector.name = oldname 

  active_layer = pdb.gimp_image_get_active_layer(svgimage)
  active_layer.name = active_layer.name.replace("Rendered SVG", oldname)
  active_layer.name = active_layer.name.replace("Wyrenderowany plik SVG", oldname)

  # cleanup
  if rem_bmp==1:
     os.remove(xchange_utf)
  #svgimage.close()
  #os.remove(tempsvgfilename)  
  # Don't delete the svg file, user might want it???
  # os.remove(tempsvgfilename)   delete the temporary svg file
  # Note the new image is dirty in Gimp and the user will be asked to save before closing,
  # but the name of the image is still the name of the tempsvgfilename and the extension is .svg
  # which Gimp does not currently support saving: the user will be asked for another extension.
  
  gimp.Display(svgimage)
  gimp.displays_flush()
 
register(
		"python_fu_trace",
		"Create new image (from active visible layer) with fewer colors and outline or centerline paths.\nRequires separate autotrace program which must be installed in the bin Gimp directory.\nName:"+whoiam ,
		"Calls separate autotrace package to trace edges or centerlines, reducing colors.  Uses temporary files.  Often leaves gaps between colors.  A sequence of filters: despeckle, quantize, trace vectors, render vectors. Version:0.4",
		"Lloyd Konneker/MrQ",
		"GPL",
		"2009/2019",
		"<Image>/Filters/Edge-Detect/_Trace...", # menuitem with accelerator key
		"*",                                     # image types
		[
		(PF_OPTION, "saveas", "Save layer to a temp-file as:",0,[" BMP - interactive visible \n (Use: Do not write color space)", " PPM - active layer\n (flattens the layer)"]),
		(PF_SPINNER, "colorcountParam",    "Reduce color count to:", 10, (1, 255, 1)),
		(PF_OPTION, "tracetypeParam",    "Trace:",0,["Outlines", "Centerlines"]),
		(PF_OPTION, "despeckleParam",    "Despeckling:",1,["Speckly","Some Despeckling","More Despeckling"]),
		(PF_SPINNER, "smoothingParam", "Corner-threshold (angle-in-deg.):\n(Smooth 2 <-100=default-> 180 Jaggy)", 100, (2, 180, 1)),
		(PF_OPTION, "pathParam",         "Create paths:",0,["None","Many, Individual","One, Merged", "Interactive\n(for resize layer)"]),
		(PF_BOOL, "vis", ("Path visible? "), False),		
		(PF_BOOL, "rem_bmp", ("Delete temp-file?\n(file tmp\*.svg delete manually)"), False),	
		],
		[],
		plugin_main,
		)

main()
 
