#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# From data of anchors, we construct a C2-continuous Gimp path.
# This is an implementation of the ideas in Stuart Kent's text:
# https://www.stkent.com/2015/07/03/building-smooth-paths-using-bezier-curves.html
# with some modifications: equations extended to produce G2-continuous curves;
# closed strokes are included.

# History:
# v0.1: 2022-02-24: First, experimental plugin.
# v0.2: 2022-03-01: Restrict the effect according to selection.
# v0.3: 2022-03-07: Better restriction. Option to straighten unprocessed segments.
# v0.4: 2022-03-09: Open strokes: option: keep the first or the last segment.
# v0.5: 2022-03-13: Open strokes: option: keep the first and last direction.
#                   Abandon "keep the first or the last segment".
# v0.6: 2022-03-17: Tunings. First published version.
# v0.7: 2022-03-28: Added option to preserve non-zero handles.
#                   Removed option to preserve tangent directions.
#                   Removed option to straighten unprocessed segments.
#                   Unpublished
# v0.8: 2022-04-02: G1-continuity at anchors where precisely one preserved handle.
#       2022-04-28: Port to Gimp2.99 / python3 by Thomas Manni

# (c) Markku Koppinen 2021
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published
#   by the Free Software Foundation; either version 3 of the License, or
#   (at your option) any later version.
#
#   This very file is the complete source code to the program.
#
#   If you make and redistribute changes to this code, please mark it
#   in reasonable ways as different from the original version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   The GPL v3 licence is available at: https://www.gnu.org/licenses/gpl-3..en.html'

import sys
from cmath import exp as cexp
import gi
gi.require_version('Gimp', '3.0')
from gi.repository import Gimp
gi.require_version('GimpUi', '3.0')
from gi.repository import GimpUi
from gi.repository import GObject
from gi.repository import GLib


#==============================================================
#             class BezierUtils       
#==============================================================

# Note: The plane is treated as the complex plane.

class BezierUtils(object):
    """Data structures to handle Gimp vectors objects by means of individual
    Bezier arcs.
    By a "Bezier arc" we mean here a Bezier curve determined by 4 control points
    p0,p1,p2,p3.
    This class contains data structures needed to do computations on
    a Bezier curve which are done arc by arc.
    To this end, Gimp data structures
    - Gimp.Vectors
    - Gimp.VectorsBezierStroke
    are here represented as data structures (classes)
    - Vectors
    - Stroke
    - BezierCurve
    - BezierArc
    - ControlPoint = complex
    with the aim that
    - each control point [x,y] is represented as an instance of ControlPoint;
    - each Bezier arc (four control points, see above) is represented as an
      instance of BezierArc.
    """
    
    ControlPoint = complex
    
    class Vectors(object):
        """Essentially same data as in a Gimp.Vectors object except that
        strokes are instances of Stroke rather than Gimp.VectorsBezierStroke.
        Attributes:
        - stroke_list: [Stroke]
        - name:        string
        """

        def __init__(self, stroke_list, name='Vectors'):
            self.strokes = stroke_list
            self.name = name

        def __str__(self):
            s = "{} {}".format(type(self).__name__, self.name)
            for i, stroke in enumerate(self.strokes):
                s += '\n  '+str(stroke)

            return s

        @staticmethod
        def from_gimp_vectors(vectors):
            """ Conversion Gimp.Vectors -> Vectors """
            strokes = []
            count = 0
            vectors_name = vectors.get_name()
            for stroke_id in vectors.get_strokes():
                stroketype, points, closed = vectors.stroke_get_points(stroke_id)
                cpoints = list(map(lambda x, y: BezierUtils.ControlPoint(x, y), points[::2], points[1::2]))
                name = vectors_name + ' stroke ' + str(count)
                strokes.append(BezierUtils.Stroke(cpoints, closed, name))
                count += 1
            return BezierUtils.Vectors(strokes, vectors_name)

        def to_gimp_vectors(self, image, name=None):
            """ Conversion Vectors instance -> Gimp.Vectors """
            
            name = self.name if name is None else name

            vectors = Gimp.Vectors.new(image, name)

            for stroke in self.strokes:
                points = [i for cp in stroke.cpoints for i in [cp.real, cp.imag]]
                vectors.stroke_new_from_points(
                    Gimp.VectorsStrokeType.BEZIER,
                    points,
                    stroke.closed)
            return vectors

    class Stroke(object):
        """Essentially same data as in a gimp.VectorsBezierStroke,
        except that control points list [x,y,x,y,...] is arranged as
        a list [ControlPoint] by joining successive x,y.
        Attributes:
        - cp_list:     [ControlPoint]
        - closed:      boolean
        - stroke_name: string
        """
        def __init__(self, cp_list, closed, stroke_name='Stroke'):
            self.name = stroke_name
            self.cpoints = cp_list
            self.closed = closed
        
        def __str__(self):
            s = '{}, closed: {}, {} Control points:'.format(self.name,
                                                            self.closed,
                                                            len(self.cpoints))
            for cp in self.cpoints:
                s += '\n    '+str(cp)
            return s
        
        def collapse_handles(self, threshold, collapse_all=False):
            """ collapse to exactly 0:
            1. either all handles, or
            2. all handles below the threshold.
            Args:
            - gv:        BezierUtils.GimpVectors
            - threshold: float
            """
            if collapse_all:
                for i in range(1, len(self.cpoints)-2, 3):
                    self.cpoints[i+1] = self.cpoints[i]
                    self.cpoints[i+2] = self.cpoints[i+3]
            else:
                for i in range(1, len(self.cpoints)-2, 3):
                    if abs(self.cpoints[i] - self.cpoints[i+1]) < threshold:
                        self.cpoints[i+1] = self.cpoints[i]
                    if abs(self.cpoints[i+2] - self.cpoints[i+3]) < threshold:
                        self.cpoints[i+2] = self.cpoints[i+3]

    class BezierCurve(object):
        """BezierCurve is a list of butting Bezier arcs with some extra data.
        Attributes:
        - stroke:     BezierUtils.Stroke
        - curve_name:  string
        Note: "Butting" means that for any pair of successive arcs, the last
              control point of the former equals the first control point of
              the latter.
              In initialization no checks are done about this condition.
        """

        def __init__(self, stroke, curve_name='BezierCurve'):
            self.name = curve_name
            self.arcs = []
            self.closed = stroke.closed
            self.head_handle = stroke.cpoints[0]
            self.tail_handle = stroke.cpoints[-1]
            inner = stroke.cpoints[1:-1]

            if len(inner) > 1: # Must be divisible by 4
                count = 0
                for i in range(0, len(inner)-1, 3):
                    self.arcs.append(BezierUtils.BezierArc(inner[i:i+4], 'arc '+str(count)))
                    count += 1
            elif len(inner) == 1: # Stroke has only one anchor
                self.arcs.append(BezierUtils.BezierArc([inner[0]], 'arc 0'))
            else:
                raise Exception("BezierUtils.BezierCurve.__init__: No anchors in stroke?")

            self._complete_arcs()

        def __str__(self):
            s = self.name + ': Bezier curve; Bezier arcs:'
            
            for i, arc in enumerate(self.arcs):
                s += '\n  arc '+ str(i)+ ': ' +str(arc)
            
            s += '\n  Head handle:    '+str(self.head_handle)
            s += '\n  Tail handle:    '+str(self.tail_handle)
            s += '\n  Closed:'    +str(self.closed)
            return s

        def _complete_arcs(self):
            """ In the case of closed curve, a gap-closing arc is added at the end,
            with necessary modification for the case of closed 1-anchor stroke.
            """
            if self.closed:
                gap_arc = BezierUtils.BezierArc(cp4=[self.arcs[-1].cp4[-1],
                                           self.tail_handle,
                                           self.head_handle,
                                           self.arcs[0].cp4[0]])
            
                if len(self.arcs[0].cp4) == 1: # 1-anchor stroke
                    self.arcs = [gap_arc]
                else:
                    self.arcs.append(gap_arc)

    class BezierArc(object):
        """Data structure for one Bezier arc: arc determined by 4 control points.
        Attributes:
        - cp4:      [p0,p1,p2,p3] where each pi:ControlPoint, or
                    [p0] if only one control point (may rise from
                    a stroke with only one anchor);
        - arc_name: string
        """
        def __init__(self, cp4, arc_name='BezierArc'):
            self.cp4 = cp4
            self.name = arc_name

        def __str__(self):
            s = 'BezierArc, control points:'
            for cp in self.cp4:
                s += '\n  '+ str(cp)
            if 'process' in self.__dict__.keys():
                s += "\n  process      = " + str(self.process)
                s += "\n  cut_at_start = " + str(self.cut_at_start)
                s += "\n  cut_at_end   = " + str(self.cut_at_end)
            return s

#==============================================================
#             class ImageSelection       
#==============================================================

class ImageSelection(object):
    """ Utility to get the selection state of a pixel """
    def __init__(self, image):
        self.sel = image.get_selection()
        self.width = image.get_width()
        self.height = image.get_height()

    def pixel_is_selected(self, x, y, threshold=127):
        if (0 <= x < self.width) and (0 <= y < self.height):
            bytes_value = self.sel.get_pixel(x, y)
            pixel_value = int.from_bytes(bytes_value, sys.byteorder)
            return (pixel_value > threshold)
        else:
            return False    

#==============================================================
#             Matrix inversion
#==============================================================
# Matrix invertion in Python, code taken from
# https://stackoverflow.com/questions/32114054/matrix-inversion-without-numpy

class SingularMatrixError(Exception):
    def __init__(self, message):
        self.message = message
    def __str__(self):
        return self.message

def eliminate(r1, r2, col, target=0):
    fac = (r2[col]-target) / r1[col]
    for i in range(len(r2)):
        r2[i] -= fac * r1[i]

def gauss(a):
    for i in range(len(a)):
        if a[i][i] == 0:
            for j in range(i+1, len(a)):
                if a[i][j] != 0:
                    a[i], a[j] = a[j], a[i]
                    break
            else:
                raise SingularMatrixError("Matrix is not invertible")

        for j in range(i+1, len(a)):
            eliminate(a[i], a[j], i)

    for i in range(len(a)-1, -1, -1):
        for j in range(i-1, -1, -1):
            eliminate(a[i], a[j], i)

    for i in range(len(a)):
        eliminate(a[i], a[i], i, target=1)

    return a

def inverse(a):
    tmp = [[] for _ in a]
    for i,row in enumerate(a):
        assert len(row) == len(a)
        tmp[i].extend(row + [0]*i + [1] + [0]*(len(a)-i-1))
    gauss(tmp)
    ret = []
    for i in range(len(tmp)):
        ret.append(tmp[i][len(tmp[i])//2:])
    return ret
    
#==============================================================
#             G2-continuity of a stroke
#==============================================================

def g2_continuity_enrich_arcs(bcurve,
                              selection,
                              selection_case,
                              preserve_nonzero_handles):
    """ Enrich the Bezier arcs with the info whether they should be
    processed by the G2 algorithm or not.
    Mark also cut points: points where the stroke must be cut prior the G2 algorithm.
    The cut points are determined by the following rule:
    The anchor between two arcs is a cut point if at least one of the
    following is true:
    - left_arc.cut_at_end = True
    - right_arc.cut_at_start = True
    - left_arc.process != right_arc.process
    Args:
    - bcurve                     BezierUtils.BezierCurve 
    - selection                  ImageSelection
    - selection_case:            string
    - preserve_nonzero_handles   boolean
    """
    ZERO = 1e-3

    for arc in bcurve.arcs:
        p0, p1, p2, p3 = arc.cp4
        
        if selection_case == 'ignore':
            arc.process = True
        else:
            p0_in_selection = selection.pixel_is_selected(p0.real, p0.imag)
            p3_in_selection = selection.pixel_is_selected(p3.real, p3.imag)
            
            if p0_in_selection != p3_in_selection:
                arc.process = False
            elif (p0_in_selection == (selection_case == 'inside')):
                arc.process = True
            else:
                arc.process = False
        
        if preserve_nonzero_handles:
            arc.cut_at_start = (abs(p0-p1) > ZERO)
            arc.cut_at_end = (abs(p2-p3) > ZERO)
        else:
            arc.cut_at_start = False
            arc.cut_at_end = False
        
    # Make cut points consistent:
    for i in range(len(bcurve.arcs)-1):
        first, second = bcurve.arcs[i], bcurve.arcs[i+1]
        cut_point = (first.cut_at_end or second.cut_at_start
                     or (first.process != second.process))
        first.cut_at_end = cut_point
        second.cut_at_start = cut_point
    
    if bcurve.closed:
        first, second = bcurve.arcs[-1], bcurve.arcs[0]
        cut_point = (first.cut_at_end or second.cut_at_start
                     or (first.process != second.process))
        first.cut_at_end = cut_point
        second.cut_at_start = cut_point


def g2_continuity_stroke(bcurve, K_tuning, L_param):
    """ Given a bezier curse with enriched arcs
    1. subdivide it to sections to be processed and sections not to be processed;
    2. call G2_continuity_anchorlist to process those that should be processed;
    3. stitch the sections back together;
    4. make a new stroke.
    Args:
    - bcurve        BezierUtils.BezierCurve
    - K_tuning:     callable (float->float)
    - L_param:      float
    Returns:
    - BezierUtils.Stroke
    """
    ZERO = 1e-3

    class Section(object):
        """Section is a maximal continuous part of a stroke such that
        all its segments should be processed by the G2 algorithm, or
        all its segments should be left unprocessed by the G2 algorithm.
        It consists of
        - a list of enriched BezierArc, each having equal 'process' field
        - a boolean attribute: process or not
        - closed_NN_HH_NH_HN: string (one of 'closed','NN','HH','NH','HN')
              Meanings:
              'closed': use G2-algorithm for closed curves
              'NN': open curve, "Natural boundary condition" at both ends
              'HH': open curve, boundary condition "Pre-determined handle" at both ends
              'NH': Different boundary condition at start and end
              'HN': Different boundary condition at start and end
              The pre-determined handles are the following two:
        - target_start_handle 
        - target_end_handle
        - input_2_vectors (the two input vectors for G2-algorithm)
        """
        def __init__(self,
                     arcs,
                     process=True,
                     closed_NN_HH_NH_HN=None,
                     target_start_handle=None,
                     target_end_handle=None,
                     input_2_vectors=None):
            self.arcs = arcs
            self.process = process
            self.closed_NN_HH_NH_HN = closed_NN_HH_NH_HN
            self.target_start_handle = target_start_handle
            self.target_end_handle = target_end_handle
            self.input_2_vectors = input_2_vectors

        def __str__(self):
            s = "Section: arcs:"
            for i,pa in enumerate(self.arcs):
                s += "\n  Arc "+str(i)+", control points:"
                for j,cp in enumerate(pa.cp4):
                        s += "\n      "+str(j)+":"+str(pa.process)+":"+str(cp)
            s += "\n    process = "+str(self.process)
            s += "\n    closed_NN_HH_NH_HN  = "+str(self.closed_NN_HH_NH_HN)
            s += "\n    target_start_handle = "+str(self.target_start_handle)
            s += "\n    target_end_handle   = "+str(self.target_end_handle)
            s += "\n    input_2_vectors   = "+str(self.input_2_vectors)
            return s

    # Step 1: Cut the bezier curve arcs in sections according the field
    # cut_at_end. (They should now be compatible with cut_at_start.)
    sections = []
    current_arcs = [bcurve.arcs[0]]
    current_process = bcurve.arcs[0].process
    
    for next_arc in bcurve.arcs[1:]:
        if not(current_arcs[-1].cut_at_end): 
            # Append next_arc to current_arcs
            current_arcs.append(next_arc)
            if next_arc.process != current_process:
                raise Exception("g2_continuity_stroke: Something wrong with bcurve.arcs")
        else:
            # End this section, start new
            sections.append(Section(arcs=current_arcs, process=current_process))
            current_arcs = [next_arc]
            current_process = next_arc.process
    
    sections.append(Section(arcs=current_arcs, process=current_process))
    no_cuts = (len(sections) == 1)
    
    # Step 2: If closed, and there were cut(s), and
    # - the first and last section have the same 'process' field, and
    # - the anchor where the joining would happen is itself not a cut point,
    # join the first and last section to one section:
    joined_first_last = False
    
    if bcurve.closed and not(no_cuts):
        first, last = sections[0], sections[-1]
        if (first.process == last.process) and not(last.arcs[-1].cut_at_end):
            joined = Section(arcs=last.arcs + first.arcs,
                             process = first.process)
            sections = [joined] + sections[1:-1]
            joined_first_last = True
        no_cuts = (len(sections) == 1) # Update this
    
    
    # Step 3: Update fields closed_NN_HH_NH_HN:
    for section in sections:
        if bcurve.closed and no_cuts:
            if not section.arcs[-1].cut_at_end:
                section.closed_NN_HH_NH_HN = 'closed'
                continue
            else:
                # The section must be processed as open
                pass
        # Assume: preserve_handle_threshold is already applied by collapsing short
        # handles, so handles to be preserved are those whose length exceeds ZERO.
        p0, p1, p2, p3 = section.arcs[0].cp4
        q0, q1, q2, q3 = section.arcs[-1].cp4
        if abs(p0-p1) < ZERO:
            case1 = 'N'
        else:
            case1 = 'H'
        if abs(q2-q3) < ZERO:
            case2 = 'N'
        else:
            case2 = 'H'
        section.closed_NN_HH_NH_HN = case1+case2
    
    # Step 4: Update fields target_start_handle and target_end_handle:
    for section in sections:
        cp4list = [pa.cp4 for pa in section.arcs]
        cp4_start = cp4list[0]
        cp4_end = cp4list[-1]
        if section.closed_NN_HH_NH_HN == 'closed':
            section.target_start_handle = None
            section.target_end_handle = None
        elif section.closed_NN_HH_NH_HN == 'NN':
            section.target_start_handle = None
            section.target_end_handle = None
        elif section.closed_NN_HH_NH_HN == 'HH':
            p0,p1,p2,p3 = cp4_start
            q0,q1,q2,q3 = cp4_end
            section.target_start_handle = -p0+p1
            section.target_end_handle = -q2+q3
        elif section.closed_NN_HH_NH_HN == 'NH':
            q0,q1,q2,q3 = cp4_end
            section.target_start_handle = None
            section.target_end_handle = -q2+q3
        elif section.closed_NN_HH_NH_HN == 'HN':
            p0,p1,p2,p3 = cp4_start
            section.target_start_handle = -p0+p1
            section.target_end_handle = None
        else:
            raise Exception("Something wrong")
    
    # Step 5: If two successive sections are xN-Hy or xH-Ny, make them xH-Hy and
    # update the relevant target handle using the neighbouring handle and K_tuning:
    # Note: This step causes that the path will be smoothed (G1-continuous)
    # at anchors where one handle is non-zero and the other is zero.
    # Collect left-right pairs of sections:
    if len(sections) > 1:
        left_right_pairs = [[sections[i], sections[i+1]]
                                            for i in range(len(sections)-1)]
        if bcurve.closed:
            left_right_pairs.append([sections[-1], sections[0]])
    elif bcurve.closed and no_cuts:
        left_right_pairs = [[sections[0],sections[0]]]
    else:
        left_right_pairs = []
    for left,right in left_right_pairs:
        if left.closed_NN_HH_NH_HN == 'closed':
            # closed stroke
            break
        left_case1, left_case2 = left.closed_NN_HH_NH_HN
        right_case1, right_case2 = right.closed_NN_HH_NH_HN
        # xH-Hy or xN-Ny
        if left_case2 == right_case1:                  
            continue
        left_cp4 = left.arcs[-1].cp4
        right_cp4 = right.arcs[0].cp4
        left_chord = abs(left_cp4[0] - left_cp4[-1])
        right_chord = abs(right_cp4[0] - right_cp4[-1])
        # xN-Hy
        if left_case2 == 'N':                          
            # Make target_end_handle for the left section
            K_factor = K_tuning(left_chord / right_chord)
            left.target_end_handle = K_factor * right.target_start_handle
            # change xN to xH
            left.closed_NN_HH_NH_HN = left_case1+'H'
        elif right_case1 == 'N':                       
            # xH-Ny
            # Make target_start_handle for the right section
            K_factor = K_tuning(right_chord / left_chord)
            right.target_start_handle = K_factor * left.target_end_handle
            # change Ny to Hy
            right.closed_NN_HH_NH_HN = 'H'+right_case2
        else:
            raise Exception("Something very wrong")
    
    # Step 6: Update field input_2_vectors:
    for section in sections:
            section.input_2_vectors = [section.target_start_handle,
                                       section.target_end_handle]
    
    # Step 7: Do the G2 processing:
    single_closed_section = (section.closed_NN_HH_NH_HN == 'closed')
    new_strokes = []
    
    for section in sections:
        cp4list = [pa.cp4 for pa in section.arcs]
        if section.process and (len(cp4list) > 1):
            # Process with the G2 algorithm
            anchors = [cp4[0] for cp4 in cp4list] + [cp4list[-1][-1]]
            new_stroke = g2_continuity_anchorlist(anchors,
                                                  single_closed_section,
                                                  K_tuning,
                                                  L_param,
                                                  section.closed_NN_HH_NH_HN,
                                                  section.input_2_vectors)
        else:
            # Do not process; copy the relevant parts from the original stroke:
            cplist = [cp4list[0][0]]
            for cp4 in cp4list:
                cplist += cp4[:-1]
            cplist += [cp4list[-1][-1]]*2
            new_stroke = BezierUtils.Stroke(cp_list=cplist,
                                            closed=single_closed_section)
        
        new_strokes.append(new_stroke)
    
    # Step 8: stitch the sections of each newly created stroke together:
    new_cpoints = []
    for i, stroke in enumerate(new_strokes):
        if i == 0:
            new_cpoints += stroke.cpoints[:-1]
        else:
            new_cpoints += stroke.cpoints[2:-1]
    
    new_cpoints.append(new_strokes[-1].cpoints[-1])

    return BezierUtils.Stroke(cp_list=new_cpoints, closed=bcurve.closed)


def g2_continuity_anchorlist(anchors,
                             closed,
                             K_tuning,
                             L_param,
                             closed_NN_HH_NH_HN,
                             input_2_vectors):
    """ G2_continuity_anchorlist
    From a list of anchors, construct a G2-continuous BezierUtils.Stroke
    (a composite Bezier curve).
    This routine contains the mathematics of the work.
    Args:
    - anchors:            [complex]
    - closed:             boolean
    - K_tuning:           callable (float->float)
    - L_param:            float
    - closed_NN_HH_NH_HN: string (one of 'closed', 'NN', 'HH', 'NH', 'HN')
    - input_2_vectors:    [complex,complex] (for boundary conditions in some cases)
    Returns:
    - BezierUtils.Stroke
    """
    A = anchors
    n = len(A)-1 # A[0] .. A[n]
    if closed:
        a = [abs(-A[i]+A[i+1]) for i in range(n-1)] + [abs(-A[n-1]+A[0])]
        k = [K_tuning(a[n-1]/a[0])] + [K_tuning(a[i-1]/a[i]) for i in range(1,n)]
        A_tilde = [-A[i]+A[i+1] for i in range(n-1)] + [-A[n-1]+A[0]]
    else:
        a = [abs(-A[i]+A[i+1]) for i in range(n)]
        k = [0] + [K_tuning(a[i-1]/a[i]) for i in range(1,n)] # 0 is placeholder, no effect
        A_tilde = [-A[i]+A[i+1] for i in range(n)]
    # Build the matrix equation:
    # The matrix M and the column vector Y of the closed case 
    # serve as the prototype for the other cases, so we build them first:
    Y = ([A_tilde[n-1] + (k[0]**2)*A_tilde[0]]
         + [A_tilde[i-1] + (k[i]**2)*A_tilde[i] for i in range(1,n)]
        )
    L1 = 2 + L_param/2
    N = [L1*ki + L1*(ki**2) for ki in k]
    matrix = [] # Build matrix M row by row
    matrix.append([N[0], (k[0]**2)*k[1]] + [0]*(n-3) + [1])     # row 0
    for i in range(1,n-1):                                      # rows 1,...,n-2
        matrix.append([0]*(i-1)
                    +[1, N[i], (k[i]**2)*k[i+1]]
                    +[0]*(n-i-2)
                    )
    matrix.append([(k[n-1]**2)*k[0]] + [0]*(n-3) + [1, N[n-1]]) # row n-1
    if closed: # Closed stroke: matrix equation already ok
        pass
    else: # Open stroke: matrix equation a little different
        if closed_NN_HH_NH_HN == 'NN':
            matrix[0] = [2, k[1]] + [0]*(n-2)
            matrix[n-1] = [0]*(n-2) + [1, N[n-1] - (1./2)*(k[n-1]**2)]
            Y[0] = A_tilde[0]
            Y[n-1] = A_tilde[n-2] + (1./2)*(k[n-1]**2)*A_tilde[n-1]
        elif closed_NN_HH_NH_HN == 'HH':
            X0, Ylast = input_2_vectors
            matrix[0] = [1] + [0]*(n-1)
            matrix[n-1] = [0]*(n-2) + [1, N[n-1]]
            Y[0] = X0
            Y[n-1] = A_tilde[n-2] + (k[n-1]**2)*A_tilde[n-1] - (k[n-1]**2)*Ylast
        elif closed_NN_HH_NH_HN == 'NH':
            _, Ylast = input_2_vectors
            matrix[0] = [2, k[1]] + [0]*(n-2)
            matrix[n-1] = [0]*(n-2) + [1, N[n-1]]
            Y[0] = A_tilde[0]
            Y[n-1] = A_tilde[n-2] + (k[n-1]**2)*A_tilde[n-1] - (k[n-1]**2)*Ylast
        elif closed_NN_HH_NH_HN == 'HN':
            X0,_ = input_2_vectors
            matrix[0] = [1] + [0]*(n-1)
            matrix[n-1] = [0]*(n-2) + [1, N[n-1] - (1./2)*(k[n-1]**2)]
            Y[0] = X0
            Y[n-1] = A_tilde[n-2] + (1./2)*(k[n-1]**2)*A_tilde[n-1]
        else:
            raise Exception("Unknown closed_NN_HH_NH_HN: "+closed_NN_HH_NH_HN)
    #print("Y:")
    #for Yi in Y:
    #    print(str(Yi))
    #print("Matrix M:")
    #for row in matrix:
    #    print(str(row))
    # Solve:
    try:
        inverse_matrix = inverse(matrix) # Inverse matrix
    except SingularMatrixError as e:
        #print(str(e))
        #print("Anchors: "+str(anchors))
        #print("Matrix:")
        #for row in matrix:
        #    print(str(row))
        raise SingularMatrixError(e.message)
    # pre_x is the solution of MX = Y:
    pre_x = [sum(inverse_matrix[i][j]*Y[j] for j in range(n)) for i in range(n)]
    if closed:
        x = pre_x # [x0,x1,x2,...,x(n-1)]
        y = [k[i+1]*x[i+1] for i in range(n-1)] + [k[0]*x[0]]
    else: # Open stroke: connection between x's and y's a little different
        if closed_NN_HH_NH_HN == 'NN':
            x = pre_x # [x0,x1,x2,...,x(n-1)]
            y = [k[i+1]*x[i+1] for i in range(n-1)] + [(1./2)*(A_tilde[n-1] - x[n-1])]
        elif closed_NN_HH_NH_HN == 'HH':
            x = pre_x # = [x0,x1,x2,...,x(n-1)]
            y = [k[i+1]*x[i+1] for i in range(n-1)] + [Ylast]
        elif closed_NN_HH_NH_HN == 'NH':
            x = pre_x # [x0,x1,x2,...,x(n-1)]
            y = [k[i+1]*x[i+1] for i in range(n-1)] + [Ylast]
        if closed_NN_HH_NH_HN == 'HN':
            x = pre_x # [x0,x1,x2,...,x(n-1)]
            y = [k[i+1]*x[i+1] for i in range(n-1)] + [(1./2)*(A_tilde[n-1] - x[n-1])]
    # Make the stroke:
    if closed: 
        cp_list = [A[0]-y[n-1]]
        for i in range(n-1):
            Ai = A[i]
            Bi = A[i+1]
            xi = x[i]
            yi = y[i]
            cp_list += [Ai, Ai + xi, Bi - yi]
        cp_list += [A[n-1], A[n-1]+x[n-1]]
    else:
        cp_list = [A[0]]
        for i in range(n):
            Ai = A[i]
            Bi = A[i+1]
            xi = x[i]
            yi = y[i]
            cp_list += [Ai, Ai + xi, Bi - yi]
        cp_list += [A[-1], A[-1]]

    return BezierUtils.Stroke(cp_list = cp_list, closed = closed)


def g2_continuity_main(image,
                       path,
                       K_tuning_parameter,
                       L_tuning_parameter,
                       force_C2,
                       preserve_nonzero_handles,
                       preserve_handle_threshold,
                       selection_option):
    if force_C2:
        # C2-continuity: K=1, L=0
        # Different edge lengths not taken into account in any way.
        K_tuning = lambda x : 1
        L_param = 0
    else:
        # More general G2-continuity
        # The function 'K_tuning' gives a one-parameter tuning option.
        # It will be called as 'tuning(a/b)' where a and b are lengths of
        # adjacent edges in the input path, hence the effect is to take into
        # account the different lengths of adjacent edges.
        # The following is a simple choice. Any function should give G2-continuity.
        # The parameter 'L_param' determines how strongly the path will bend around
        # each anchor.
        K_tuning = lambda x : pow(x, K_tuning_parameter / 4)
        L_param = (cexp(L_tuning_parameter / 4) - 1) * 3 # ad hoc
    
    selection = ImageSelection(image)
    vectors = BezierUtils.Vectors.from_gimp_vectors(path)

    new_strokes = []

    for stroke in vectors.strokes:
        stroke.collapse_handles(preserve_handle_threshold,
                                not preserve_nonzero_handles)
        if len(stroke.cpoints) < 6:
            new_strokes.append(stroke)
        else:
            bcurve = BezierUtils.BezierCurve(stroke)
            
            g2_continuity_enrich_arcs(bcurve,
                                      selection,
                                      selection_option,
                                      preserve_nonzero_handles)
            
            new_stroke = g2_continuity_stroke(bcurve, K_tuning, L_param)
            new_strokes.append(new_stroke)

    new_vectors = BezierUtils.Vectors(stroke_list=new_strokes)
    
    image.undo_group_start()

    new_path = new_vectors.to_gimp_vectors(image)
    name = path.get_name()

    if force_C2:
        name = name+"|C2"
    else:
        name = name+"|G2"+str((K_tuning_parameter,L_tuning_parameter))

    new_path.set_name(name)
    new_path.set_visible(True)
    image.insert_vectors(new_path, None, 0)
    image.undo_group_end()
    return new_path


#==============================================================
#             Gimp Procedure Registration       
#==============================================================

versionnumber = "0.8"
procedure_name = 'python-fu-g2-continuity'
procedure_author = "Markku Koppinen"
procedure_copyright = procedure_author
procedure_date = "2022"
image_types = "*"
menu_label = "G2 continuity"
menu_path = '<Vectors>/Tools'

procedure_name  = "G2_continuity"
procedure_blurb = ("A G2-continuous path is generated"
                   +"\nthrough the anchors of the input path."
                   +"\n(Version "+versionnumber+")")
procedure_help  = ("Given a path, a G2-continuous path is generated "
                   +"through its anchors. Optionally C2-continuity can be forced."
                   +"\n- Two tuning parameters are provided, both rather ad hoc:"
                   +"\nThe first adjusts how strongly differences in distances "
                   +"between successive anchors are taken into account; the second "
                   +"adjusts how tightly the path will bend at anchors."
                   +"\n- The plugin allows preserving handles whose length exceeds "
                   +"some given threshold. This enables the user to preserve shapes"
                   +"\nof sharp corners. The idea is that"
                   +"\n(1) the purpose is to smooth the path almost everywhere; but"
                   +"\n(2) at some anchors a sharp corner and its shape are to be preserved."
                   +"\nThis is achieved by shaping such corners with long handles (over the "
                   +"threshold) which the plugin then preserves, and having all other handles zero length "
                   +"or very short (below the threshold). "
                   +"The threshold has also the useful consequence that very short handles will be ignored "
                   +"which may inadvertently be present in the path."
                   +"\n- At anchors where both handles are preserved the path will be "
                   +"continuous; at anchors where one of the handles is preserved the path will be "
                   +"G1-continuous; everywhere else it will be G2-continuous."
                   +"\n- The plugin allows to restrict the effect by means of a selection; "
                   +"however, to have any effect, at least three consecutive anchors are needed.")


class G2Continuity(Gimp.PlugIn):
    __gproperties__ = {
        "run_mode": (Gimp.RunMode,
                     "Run mode",
                     "The run mode",
                     Gimp.RunMode.NONINTERACTIVE,
                     GObject.ParamFlags.READWRITE),
        "image": (Gimp.Image,
                  "Image",
                  "The image",
                  GObject.ParamFlags.READWRITE),
        "vectors": (Gimp.Vectors,
                    "Vectors",
                    "The vectors",
                    GObject.ParamFlags.READWRITE),
        "K_tuning_parameter": (float,
                       "_First tuning parameter",
                       "Adjusts how strongly segment lengths are taken into account.",
                       -2.0, 3.0, 1.0,
                       GObject.ParamFlags.READWRITE),
        "L_tuning_parameter": (float,
                       "S_econd tuning parameter",
                       "Adjusts how tightly the path will bend at anchors.",
                       -3.0, 4.0, 0.0,
                       GObject.ParamFlags.READWRITE),
        "force_C2": (bool,
                       "Force C_2-continuity",
                       "Force C2-continuity ?"
                       +"\nIf 'Yes', the two tuning parameters will have no effect.",
                       False,
                       GObject.ParamFlags.READWRITE),
        "preserve_nonzero_handles": (bool,
                       "_Preserve long handles",
                       "Preserve long handles (overriding G2-continuity locally)?"
                       +"\nIf 'Yes', set the threshold for \"long\" handles below.",
                       False,
                       GObject.ParamFlags.READWRITE),
        "preserve_handle_threshold": (float,
                       "The _threshold for \"long\" handles (pixels)",
                       "Use this to avoid very short handles that were perhaps made accidentally",
                       0.0, 1000.0, 5.0,
                       GObject.ParamFlags.READWRITE),
        "selection_option": (str,
                      "How to _use the selection",
                      "'ignore': Ignore any selection\n"
                     +"'inside': Restrict the effect to segments with both ends inside the selection\n"
                     +"'outside': Restrict the effect to segments with both ends outside of the selection",
                      "ignore",
                      GObject.ParamFlags.READWRITE)
    }

    def do_query_procedures(self):
        return [ 'python-fu-g2-continuity' ]

    def do_create_procedure(self, name):
        procedure = None
        if name == 'python-fu-g2-continuity':
            procedure = Gimp.Procedure.new(self, name, Gimp.PDBProcType.PLUGIN,
                                           self.run, None)
            procedure.set_documentation(procedure_blurb,
                                        procedure_help,
                                        name)
            procedure.set_attribution(procedure_author,
                                      procedure_copyright,
                                      procedure_date)
            procedure.set_menu_label(menu_label)
            procedure.add_menu_path(menu_path)
            
            procedure.add_argument_from_property(self, "run_mode")
            procedure.add_argument_from_property(self, "image")
            procedure.add_argument_from_property(self, "vectors")
            procedure.add_argument_from_property(self, "K_tuning_parameter")
            procedure.add_argument_from_property(self, "L_tuning_parameter")
            procedure.add_argument_from_property(self, "force_C2")
            procedure.add_argument_from_property(self, "preserve_nonzero_handles")
            procedure.add_argument_from_property(self, "preserve_handle_threshold")
            procedure.add_argument_from_property(self, "selection_option")

        return procedure
    
    def run(self, procedure, args, data):
        run_mode = args.index(0)
        image = args.index(1)
        vectors = args.index(2)

        config = procedure.create_config()
        config.begin_run(image, run_mode, args)

        if run_mode == Gimp.RunMode.INTERACTIVE:
            GimpUi.init(procedure_name)
            dialog = GimpUi.ProcedureDialog(procedure=procedure, config=config)
            dialog.fill(None)
            if not dialog.run():
                dialog.destroy()
                config.end_run(Gimp.PDBStatusType.CANCEL)
                return procedure.new_return_values(Gimp.PDBStatusType.CANCEL,
                                                   GLib.Error())
            else:
                dialog.destroy()
        
        g2_continuity_main(image,
                           vectors,
                           config.get_property('K_tuning_parameter'),
                           config.get_property('L_tuning_parameter'),
                           config.get_property('force_C2'),
                           config.get_property('preserve_nonzero_handles'),
                           config.get_property('preserve_handle_threshold'),
                           config.get_property('selection_option'))

        config.end_run(Gimp.PDBStatusType.SUCCESS)

        return procedure.new_return_values(Gimp.PDBStatusType.SUCCESS,
                                           GLib.Error())


Gimp.main(G2Continuity.__gtype__, sys.argv)

