#!/usr/bin/env python

# Gimp plugin "Uncrop side"
# Increase image/canvas size by synthesizing a band on the edge of the original image.
# Based on the GIMP plug-in "Uncrop" by lloyd konneker of 2009.
# Modified to synthesize image data on the selected edge of the image.
# Generally works better for small increments.
# Works well on some images, very badly on others!
# Comments from lloyd konneker's original about the necessary parameters which need to be
# passed to the resynthesizer module have been included as this information is difficult to
# find elsewhere.
#
# Comments from lloyd konneker's original:
# Note that the API hasn't changed but use_border param now has more values.
# border param=5 means inside out direction. 5+0=5 outward concentric, 5+1=6 outward from sides,
# 5+2=7 outward from above and below.
# End of comments from lloyd konneker's original.
#
# Dependencies: GIMP Resynthesizer plug-in.
#
# Version 1 20/10/2018
# Version 2 27/08/2020 Correction of enlargement calculation.
# Version 3 20/06/2022 Removes alpha channel if one exists.
# Version 4 21/06/2022 Outer area filled with white - no need to remove alpha channel.
#
# License: GPLv3
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY# without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# To view a copy of the GNU General Public License
# visit: http://www.gnu.org/licenses/gpl.html
#


from gimpfu import *

gettext.install("resynthesizer", gimp.locale_directory, unicode=True)
    
def resizeImage(orgImage, percentEnlargeParam, side):
            
    deltaFraction =(percentEnlargeParam / 100) + 1.0			# resize image by percent (converted to pixel units)
    priorWidth = pdb.gimp_image_width(orgImage)
    priorHeight = pdb.gimp_image_height(orgImage)
    newWidth = priorWidth
    newHeight = priorHeight
    OffX = 0
    OffY = 0
        
    if   side == 0:												# left
         newWidth = priorWidth * deltaFraction
         newHeight = priorHeight
         OffX = newWidth - priorWidth
         OffY = 0
         
    elif side == 1:												# right
         newWidth = priorWidth *  deltaFraction
         newHeight = priorHeight
         OffX = 0
         OffY = 0
         
    elif side == 2:												# top
         newWidth = priorWidth
         newHeight = priorHeight * deltaFraction
         OffX = 0
         OffY = newHeight - priorHeight
         
    elif side == 3:												# bottom
         newWidth = priorWidth
         newHeight = priorHeight * deltaFraction
         OffX = 0
         OffY = 0
                
    pdb.gimp_image_resize(orgImage, newWidth, newHeight, OffX, OffY)
    
################## starts here ################################    
        
def uncrop_side(orgImage, drawable, percentEnlargeParam, sampleWidth, side, direction=6, autisM=0.010, neighbourHood=16, tryS=500):
      
    pdb.gimp_image_undo_group_start(orgImage)
        
    tempImage = pdb.gimp_image_duplicate(orgImage)				# copy original into temp for later use        
        
    pdb.gimp_selection_all(orgImage)							# save original limits to subtract from outer area later
    selectAllPrior = pdb.gimp_selection_save(orgImage)
    
    resizeImage(orgImage, percentEnlargeParam, side)			# call to resize layer routine
    pdb.gimp_layer_resize_to_image_size(drawable)				# need to resize layer as well as image
    pdb.gimp_image_select_item(orgImage, CHANNEL_OP_REPLACE, selectAllPrior)    
    pdb.gimp_selection_invert(orgImage)							# select outer area, the new blank canvas
   
    workLayer = pdb.gimp_image_get_active_layer(tempImage)		# prepare source layer.
    
    pdb.gimp_selection_all(tempImage)
    priorWidth = pdb.gimp_image_width(tempImage)
    priorHeight = pdb.gimp_image_height(tempImage)
    deltaWidth =  priorWidth
    deltaHeight = priorHeight
    OffX = 0
    OffY = 0
        
    if   side == 0:												# left
         deltaWidth = sampleWidth
         deltaHeight = priorHeight  
         OffX = 0
         OffY = 0
         direction = 6
         
    elif side == 1:												# right
         deltaWidth = sampleWidth
         deltaHeight = priorHeight
         OffX = priorWidth - sampleWidth
         OffY = 0
         direction = 6
         
    elif side == 2:												# top
         deltaWidth =  priorWidth
         deltaHeight = sampleWidth
         OffX = 0
         OffY = 0
         direction = 7
         
    elif side == 3:												# bottom
         deltaWidth =  priorWidth
         deltaHeight = sampleWidth
         OffX = 0
         OffY = priorHeight - sampleWidth
         direction = 7
                
    pdb.gimp_image_crop(tempImage, deltaWidth, deltaHeight, OffX, OffY)
    pdb.gimp_selection_all(tempImage)
    pdb.gimp_drawable_edit_fill(drawable, FILL_WHITE)    

    pdb.plug_in_resynthesizer(orgImage, drawable, 0,0,direction, workLayer.ID, -1, -1, 0.0, autisM, 16, 500)
    
    pdb.gimp_selection_none(orgImage)
    pdb.gimp_image_remove_channel(orgImage, selectAllPrior)
    pdb.gimp_image_undo_group_end(orgImage)
    pdb.gimp_displays_flush()
    gimp.delete(tempImage)


register(
    "python_fu_uncrop_side",
    "Uncrop side",
    "Uncrop side",
    "David Marsden",
    "David Marsden",
    "October 2018",
    "<Image>/Filters/Enhance/Uncrop side...",
    "RGB*, GRAY*",
    [
      (PF_IMAGE, "image", "Input image", None),
      (PF_DRAWABLE, "drawable", "Input drawable", None),
      (PF_SLIDER, "percentEnlargeParam", _("Percent enlargement"), 5, (1, 20, 1)),
      (PF_SLIDER, "sampleWidth", _("Sample Width"), 50, (10, 100, 1)),
      (PF_OPTION, "side", "Side", 0, ("L", "R", "T", "B")),
      (PF_SLIDER, "autisM", _("Autism"), 0.010, (0.001, 1.000, 0.001)),
      (PF_SLIDER, "neighbourHood", _("Neighbourhood"), 16, (1, 100, 1)),
      (PF_SLIDER, "tryS", _("Search Thoroughness"), 500, (1, 500, 1))
    ],
    [],
    uncrop_side,
    menu="<Image>/Filters/Enhance",
    domain=("resynthesizer", gimp.locale_directory)
    )

main()
