#!/usr/bin/env python
# -*- coding: utf-8 -*-

# GIMP plug-in to backup open images at regular interval.

# Plug-in history:
# ° 2009, Starting from 'yahvuu' at http://
#     www.mail-archive.com/gimp-developer@lists.xcf.berkeley.edu/msg18118.html
# ° 2011, First version with GUI at http://registry.gimp.org/node/26115 ;
#     tested then with GIMP-2.6.11 and Python 2.7.1; by Robert Brizard.
# ° July 2012, Version 0.2: improves the config presets, adds the procedure 
#     'noUI_autosave_a', an 'Untitled' adaptation to GIMP-2.8.0
#     and instance verification for 'noUI_autosave_a'.
#     With 3 types of config presets:
#               Type                       Name        Nr
#     hard-coded,                        (Default),    1 ,
#     plug-in save with click on 'Stop', (LastStop),   1 ,
#     user save by click on 'Save...',   (integer<5),  5 .
# ° December 2012, Version 0.3: take care of some mishap. Window instance number 
#     for 'autosave_a', check folder and file existence. Added free space info. 
#     Tested on GIMP-2.8.2 and Python 2.7.3 . Benificial for external media.
# * November 2014, Version 0.4: add a reset button before the combobox choice if the 
#     actual config value list don't correspond (a dirty flag) and other tweaks of UI. 
#     Try for clearer instructions. Correct some minor bugs. 
#     Tested with GIMP-2.8.14 and Python 2.7.5

#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.


import os, time, sys
import gtk, shelve, pango
import gettext, pygtk
pygtk.require('2.0')
from gobject import timeout_add_seconds
from copy import deepcopy

def AS_mssgBox(mess):
    flag = gtk.DIALOG_MODAL|gtk.DIALOG_DESTROY_WITH_PARENT
    msgBox = gtk.MessageDialog(None, flag, gtk.MESSAGE_WARNING, gtk.BUTTONS_OK,\
    mess)
    msgBox.run()
    msgBox.destroy()

try:
    from gimpfu import *
    from gimpshelf import shelf
except ImportError:
    print("Note: GIMP is needed, '%s' is a plug-in.\n"%__file__)
    AS_mssgBox("Note: GIMP is needed, '%s' is a plug-in for it.\n"%__file__)
    sys.exit(1)

# for Windows file system
if os.name == 'nt':
    import ctypes
    file_encode = sys.getfilesystemencoding()
def sys_file(f_name):
    if os.name == 'nt': encoded = f_name.encode(file_encode)
    else: encoded = f_name
    return(encoded)

# Internationalization 'i18n'
locale_directory = os.path.join(os.path.dirname(os.path.abspath(__file__)), \
                    'locale') # for users; administrator: gimp.locale_directory
gettext.install( "autosave_a", locale_directory, unicode=True )

class Save_recall():
    """
    Store, load and query dictionary name with string variable keyNr
    N.B.: -this version does not accept unicode (eval?)
          -keyNr['dir_BU'] is utf-8 encoded
    """
    def __init__(self):
        folder = os.path.dirname(os.path.abspath(__file__))+os.sep+'autosave_a'
        # 'file_shelf' was 'autosave.cfg', change because of refactoring config
        self.file_shelf = folder+os.sep+'autosave1.cfg'
        # if folder not there, create it
        folder = sys_file(folder)
        if not os.path.exists(folder): os.mkdir(folder)
        if os.path.isfile(sys_file(self.file_shelf)): self.manage_folder_exist()
        
        
    def manage_folder_exist(self):
        """
        See if the folder in the preset still exist;
        for the numbered one if not remove the preset,
        for the other one, switch to an existing folder.
        """
        global warning
        
        f = shelve.open(self.file_shelf, writeback=True)
        actual_dir = gtk.FileChooserDialog()
        # list of all keys (but later only valid 'recall_' one will be kept)
        keys_lst = f.keys()
        recall_nvkey = []   # list of 'recall...' non-valid keys
        key_nw = []         # list of non-wanted keys (not 'recall...')
        #print('Values of keys_lst: '+str(keys_lst))
        for keyV in keys_lst:
            sys_folder = sys_file(f[keyV]['dir_BU'])
            if keyV[:6] == 'recall':
                if not os.path.exists(sys_folder):
                    recall_nvkey.append(keyV)
            else:
                key_nw.append(keyV)
                if not os.path.exists(sys_folder):
                    # here change to existing folder
                    f[keyV]['dir_BU'] = default_fold
                    if not warning:
                        warning += _("WARNING:\n did not find the folder in '%s'")\
                            %keyV+_(",\nchanging it to %s.\n")%f[keyV]['dir_BU']
                    else:
                        warning += _("Also '%s' was changed the same way.\n")%keyV

        # renumber and erase non-valid preset 
        nrb = len(recall_nvkey)
        if nrb > 0:
            #  and shelf.has_key('autosave') seems to affect the wire protocol
            for val in (key_nw + recall_nvkey): keys_lst.remove(val)

            # numbering the change number and compacting
            nre = len(keys_lst)
            if nre > 0:
                keys_lst.sort()
                for i in range(0, nre):
                    if 'recall_config%d'%i not in keys_lst:
                        f['recall_config%d'%i] = f[keys_lst[i]]
            
            # erasing the rest
            for i in range(nre, nrb+nre): del f['recall_config%d'%i]
            warning += _("At start, %d previous numbered config preset has been")%nrb\
                +_(" delete out of %d because their folder don't exist.")%(nrb+nre)
        f.close()
        return

    def save(self, keyNr):
        """
        keyNr is a string
        Store a keyNr dictionary in a shelf object
        """
        f = shelve.open(self.file_shelf)
        try: f[keyNr] = eval(keyNr)
        except: print("Problem: no dictionary with the name %s to save"%keyNr)
        f.close()
        return
        
    def has_key(self, keyNr):
        """
        Return true if a dictionary keyNr exist
        """
        f = shelve.open(self.file_shelf)
        if f.has_key(keyNr): rep = True
        else: rep = False
        f.close()
        return rep

    def list_dict(self):
        """
        Return list of keys in shelf_file
        """
        rep = []
        f = shelve.open(self.file_shelf)
        rep = f.keys()
        rep.sort()
        f.close()
        return rep

    def recall(self, keyNr):
        """
        Retreive the dictionary keyNr, if there
        """
        f = shelve.open(self.file_shelf)
        dictio = None
        if self.has_key(keyNr):
            dictio = f[keyNr]
        else:
            print("No '%s' key exist in shelve file!"%keyNr)
        f.close()
        return dictio

active = False
last_clock = 0
cntr = 0
backupFiles = {}

# Management of the saved config presets =======================================
warning = ""    # for non actual existence of folder
folder_prob = ""
actual_dir = gtk.FileChooserDialog()
default_fold = actual_dir.get_current_folder()
# check for non local file name
if default_fold == None: 
    # not a local folder so find an alternate folder?
    default_fold = os.path.dirname(os.path.abspath(__file__))+os.sep+'autosave_a'
    if not actual_dir.set_current_folder(default_fold):
        # if unable to use 'gtk.FileChooserDialog()', a message for the user
        folder_prob = _("ERROR:\nthis file system seems incompatible with 'autosave_a';")\
            +_("\nit is limited by choice to local folder only.")
        # bail out
        popup_mess(folder_prob)
        sys.exit(1)
    
    else:
        warning += _("WARNING:\n the current folder was not available, chosen an ")\
            +_("alternate default folder.\n")

# conf. variables in a global dictionary & start with a config_act
shelf_fl = Save_recall()

if not shelf_fl.has_key('default_config') :
    actual_dir = gtk.FileChooserDialog()
    default_config = {
        'dir_BU'    : default_fold,               # backup dir, string
        'image'     : 2,                          # image(s) to save, index in 'source'
        'extension' : 0,                          # extension of file, index in 'exten'
        'kept'      : 1,                          # number of backup to keep
        'interval(s)'  : 600.0,                   # backup interval in second
        'start'     : False                       # at start of interval, bool.
    }
    shelf_fl.save('default_config')
#TODO: permit change to 'default_config' (for 'noUI_autosave_a'?)

config = {}

# select which config to start with
if shelf_fl.has_key('laststop_config'): 
    config_act = 'laststop_config' 
else: config_act = 'default_config'

config = shelf_fl.recall(config_act)

# def the variable for the numbered preset
for i in range(5): exec('recall_config%d'%i + ' = {}')

# Preset mode messages
msga = _("Adding numbered config: a consecutive sequence of 5 .")
msgr = _("Replacing a numbered config: first, click select-box \'Recall config (number?)\'.")

# end of preset management =====================================================

# next are choices in 'Files'
source = [_("Launching one"), _("All changed"), _("All open")]
exten = [".xcf.bz2", ".xcf.gz", ".xcf"]


class Control_Autosave(gtk.Window):
    """
    GUI for interaction with autosave_a and calling repeatedly
    the timing fonction
    """

    def __init__(self, img, parent=gimp):
        global config
        self.image = [img.name, img.ID]
        self.recycle = False

        # color for starting frame labels
        span_conf = "<span foreground='dark blue' background='yellow' weight='bold' >%s</span>"
        span_stat = "<span foreground='dark green' background='#d9c9c9' weight='bold' >%s</span>"

        # Create the window
        self.win = gtk.Window.__init__(self)
        #self.connect('destroy', lambda *w: gtk.main_quit())
        self.connect('destroy', self.on_destroy)
        self.set_title(_("Periodic Saver for GIMP"))
        
        # put the instance number in the subtitle
        sup_vbox = gtk.VBox(False, 8)
        self.add(sup_vbox)
        title_line = _("Autosave_a-0.4, Panel <%d>")%(shelf['autosave']['instance']) 
        label1 = gtk.Label(title_line)
        # Change attributes of the subtitle
        attr = pango.AttrList()
        bold = pango.AttrWeight(pango.WEIGHT_ULTRABOLD, 0, len(title_line))
        attr.insert(bold)
        label1.set_attributes(attr)
        label1.set_has_tooltip(True)
        label1.set_tooltip_text(_("The number between <> is the number of actual")\
                                    +_("\ninstance of this window at opening."))
        sup_vbox.pack_start(label1, False, False, 0)

        # next code block is for the window icon -------------------------------
        icon_dir = os.path.join(os.path.dirname(os.path.abspath(__file__)), \
                                'autosave_a')
        try:
            self.set_icon_from_file( icon_dir+os.sep+'autosave-icon.png')
        except:
            print("Autosave_a warning: did not find the icon file!")
        # ----------------------------------------------------------------------

        #self.set_focus_on_map(True)
        self.set_border_width(8)

        warn_mess = ""

        ##############################
        ## Instructions+ ->
        # add warning if a 'noUI_autosave_a' is running
        self.choices = shelf_fl.list_dict()
        if shelf.has_key('noUI_autosave') and shelf['noUI_autosave']:
            warn_mess += _("Warning: 'NoUI_autosave_a' is running. ")
            noUI_config = shelf['noUI_autosave']
            if config == noUI_config: warn_mess += _("Same initial config.\n")
            else:
                for cfg in self.choices:
                    if noUI_config == shelf_fl.recall(cfg):
                        add_mess = _("Config is '%s'.\n")%cfg
                        break
                else:
                    add_mess = _("Not a preset config!\n")
                    gimp.message(_("The 'NoUI' configuration with no preset is:\n%s")\
                        %cfg2lines(noUI_config))
                warn_mess += add_mess 
        top_label1 = warn_mess+_(" Manage and choose your parameters below then press 'Start'")\
            +_(" to enter\nbackup mode. Only button active after 'Start' is 'Stop'.")
        # Change attribute of the label warning line
        self.label_ = gtk.Label()
        self.set_label_attribute(2, warn_mess, self.label_)
        self.label_.set_text(top_label1)
        sup_vbox.pack_start(self.label_, False, False, 0)

        ##############################
        ## Directory ->

        # label for frame
        self.label1 = gtk.Label()
        self.label1.set_use_markup(True)
        self.label1.set_label(span_conf%_("Directory"))

        dir_frame = gtk.Frame()
        dir_frame.set_label_widget(self.label1)
        dir_frame.set_label_align(0.0, 0.8)
        #dir_frame.set_shadow_type(gtk.SHADOW_ETCHED_OUT)
        sup_vbox.pack_start(dir_frame, padding=5)


        # button to choose dir (to launch gtk.FileChooser)
        hbox = gtk.HBox(False, 4)
        button = gtk.Button(_("Change the"))
        button.connect('clicked', self.on_choose_dir_change)
        button.set_has_tooltip(True)
        button.set_tooltip_text(_("After selecting the folder, a click on")\
            +(" close the dialog window confirm it."))
        hbox.pack_start(button, False, False, 0)
        # put the actual dir beside it
        label = gtk.Label(_("backup dir.: "))
        hbox.pack_start(label, False, False, 0)

        self.label0 = gtk.Label('')
        self.label0.set_has_tooltip(True)
        self.label0.set_tooltip_text(_("The actual folder where the backup will")\
           +_(" be.\nIf written in red, the verified free space is less then 1.5 MB."))
        hbox.pack_start(self.label0, False, False, 0)
        dir_frame.add(hbox)
        
        # possible to delete files as it is now        

        ##############################
        ## Files ->

        # label for frame in config area
        self.label2 = gtk.Label()
        self.label2.set_use_markup(True)
        self.label2.set_label(span_conf%_("Files"))

        file_frame = gtk.Frame()
        file_frame.set_label_widget(self.label2)
        sup_vbox.pack_start(file_frame, padding=5)
        table = gtk.Table(2, 3)
        table.set_row_spacings(4)
        table.set_col_spacings(4)
        file_frame.add(table)

        # column headers
        label = gtk.Label(_("image"))
        label.set_has_tooltip(True)
        label.set_tooltip_text(_("The source(s) of the backup file from:")\
                                    +_("\nthe one to all"))
        table.attach(label, 0, 1, 0, 1)
        label = gtk.Label(_("extension"))
        label.set_tooltip_text(_("To control the compression of 'xcf' file from:")\
                                    +_("\nmore to less"))
        table.attach(label, 1, 2, 0, 1)
        label = gtk.Label(_("nr kept"))
        label.set_has_tooltip(True)
        label.set_tooltip_text(_("Number of backup kept for the same ")\
                                    +_("\nsession and image ID"))
        table.attach(label, 2, 3, 0, 1)

        # autosave image source + extension: xcfbz2 or xcfgz
        source = [_("Launching one"), _("All changed"), \
                      _("All open")]
        self.combo1 = gtk.combo_box_new_text()
        [self.combo1.append_text(x) for x in source]
        self.combo1.connect("changed",  self.on_img_source_change)
        table.attach(self.combo1, 0, 1, 1, 2)

        self.combo = gtk.combo_box_new_text()
        [self.combo.append_text(x) for x in exten]
        self.combo.connect("changed",  self.on_file_ext_change)
        table.attach(self.combo, 1, 2, 1, 2)

        # number of backup to keep of current image: 1 -> 9
        self.interv = gtk.SpinButton(adjustment=None, climb_rate=1.0, digits=0)
        self.interv.set_range(1, 9)
        self.interv.set_increments(1, 1)
        self.interv.set_numeric(True)
        self.interv.connect('value-changed',  self.on_nr_kept_change)
        table.attach(self.interv, 2, 3, 1, 2)

        ##############################
        ## Time ->

        # label for frame in config area
        self.label3 = gtk.Label()
        self.label3.set_use_markup(True)
        self.label3.set_label(span_conf%_("Time"))

        time_frame = gtk.Frame()
        time_frame.set_label_widget(self.label3)
        sup_vbox.pack_start(time_frame, padding=5)
        hbox = gtk.HBox(False, 4)
        time_frame.add(hbox)

        self.rbtn = gtk.CheckButton(_("At start? "))        
        self.rbtn.connect("toggled", self.on_toggled_check_change, None)
        self.rbtn.set_has_tooltip(True)
        self.rbtn.set_tooltip_text(_("If check: begins at the interval start,")\
                                  +_("\nif not at the end."))
        hbox.pack_start(self.rbtn, False, False, 0)

        # put the header for time interval beside it
        label = gtk.Label(_("Backup interval (min): "))
        hbox.pack_start(label, False, False, 0)
        # gtk.SpinButton(adjustment=None, climb_rate=0.0, digits=0)
        self.interv1 = gtk.SpinButton(adjustment=None, climb_rate=0.0, digits=1)
        self.interv1.set_range(1, 999)
        self.interv1.set_increments(0.1, 5)
        self.interv1.set_numeric(True)
        self.interv1.connect('value-changed',  self.on_time_interval_change)
        hbox.pack_start(self.interv1, False, False, 0)

        ##############################
        ## Controls line ->

        hbox = gtk.HBox(False, 8)
        sup_vbox.pack_start(hbox, False, False, 0)

        # button to save current parameters now
        self.nr = 0     # number of numbered save_config
        for i in range(len(self.choices)):
            if self.choices[i][:5] == 'recal': self.nr += 1
        if self.nr > 4 : 
            self.recycle = True
            self.nr = 0 
        self.button = gtk.Button(_("Save config %d>")%self.nr)
        self.button.set_has_tooltip(True)
        self.button.set_tooltip_text(_("Will save all the above parameters on disk")\
                            +_("\nfor later usage with select-box 'Recall config...'"))
        self.button.connect('clicked', self.on_save_now_clicked)
        hbox.pack_start(self.button, False, False, 0)

        # button for a dirty flag
        self.button2 = gtk.Button('')
        #self.button2.set_use_markup(True) don't work for button
        self.button2.set_has_tooltip(True)
        self.button2.set_tooltip_text(_("A star here means a config change from following")\
            +_(" recall preset.\nCan after consideration click here, if a star, to reset."))
        self.button2.connect("released", self.on_reset_recall_config_clicked)
        hbox.pack_start(self.button2, 0) 
        
        # place a remenber param. combobox
        hbox2 = gtk.HBox()
        hbox.pack_start(hbox2)
        self.combo_box = gtk.combo_box_new_text()
        self.combo_box.set_wrap_width(1)
        # new 'choices' list to unable lang. translation
        self.choices_combo = [_("Default config")]
        conf_act = 0
        for i in range(len(self.choices)):
            if i > 0 :
                if self.choices[i][:5] == 'lasts':
                    self.choices_combo.append(_("LastStop config"))
                    conf_act = i
                else:
                    self.choices_combo.append(_("Recall config ")+\
                        self.choices[i][-1:])
            self.combo_box.append_text(self.choices_combo[i])
        self.combo_box.set_active(conf_act)
        self.config_sav = deepcopy(config)
        self.combo_box.set_has_tooltip(True)
        self.combo_box.set_tooltip_text(_("Recall (display above) other existing config presets"))
        hbox2.pack_start(self.combo_box)
        self.combo_box.connect("changed", self.on_recall_config_clicked)

        # icon to indicate state
        self.stock_ic = gtk.Image()
        self.stock_ic.set_from_stock(gtk.STOCK_EXECUTE, gtk.ICON_SIZE_BUTTON)
        self.stock_ic.set_has_tooltip(True)
        self.stock_ic.set_tooltip_text(_("Indicates if the widgets to the left")\
                +_("\nand above are frozen or not"))
        hbox.pack_start(self.stock_ic, False, False, 0)
        hbox.pack_start(gtk.VSeparator(), expand=False)

        # place a start-stop button
        self.button1 = gtk.Button(_("Start"))
        self.button1.connect('pressed', self.on_activate_clicked)
        hbox.pack_start(self.button1, True, True, 3)

        ##############################
        ## Status info ->

        # label for status frame
        self.label4 = gtk.Label()
        self.label4.set_use_markup(True)
        self.label4.set_label(span_stat%_("Config status"))

        info_frame = gtk.Frame()
        info_frame.set_label_widget(self.label4)
        info_frame.set_label_align(0.5, 0.5)
        sup_vbox.pack_start(info_frame, padding=5)
        vbox = gtk.VBox(False, 0)
        info_frame.add(vbox)

        if self.recycle: msgi = msgr
        else: msgi = msga
        self.label5 = gtk.Label(msgi)
        #self.label5.set_use_markup(True)
        self.label5.set_alignment(0.1, 0.2)
        vbox.pack_start(self.label5, False, False, 0)

        # self.flag is for status message in 'timer_action()'
        self.flag = True
        
        ##############################
        ## finaly for the whole window    
        self.show_all()
        self.set_config()
        timeout_add_seconds(1, self.timer_action, priority=308)

    def on_destroy(self, arg):
        end = time.strftime("%a, %d %b %Y %H:%M:%S")
        # one less in gimpshelf
        shelf['autosave'] = {'instance': shelf['autosave']['instance'] - 1}
        if active:
            end_message = _("INFO:\n  was closed on:\n %s;")%end
            if cntr > 1 : end_message += _("\nafter %d rounds of backup.")\
                %cntr
            else : end_message += _("\nafter %d round of backup.")%cntr
            
            shelf['autosaver'] = {'running' : shelf['autosaver']['running'] - 1}
            gimp.message(end_message)            
        self.destroy()
        gtk.main_quit()

    def set_config(self):
        """ put config on display """
        if not active:
            mess = config['dir_BU']
            self.label0.set_text(mess)
            self.set_label_attribute(1, mess, self.label0)
            self.combo1.set_active(config['image'])
            self.combo.set_active(config['extension'])
            self.interv.set_value(config['kept'])
            self.interv1.set_value(config['interval(s)']/60.0)
            if self.rbtn.get_active() != config['start']:
                self.rbtn.set_active(config['start'])
            return

    def manage_dirty_flag(self):
        # compare config to recall_config (in all methods '...change()')
        if config.values() == self.config_sav.values() : 
            self.button2.set_label('')
        else:
            self.button2.set_label('*')
        return

    def on_choose_dir_change(self, button):
        """
        Note: the method starting with 'on_...' is a callback and ending give the
         UI origin: if '..._change' its the config area, if '..._clicked' the action line
         and other its the whole window.
        """

        global config
        if not active:

            folder = gtk.FileChooserDialog(_('Autosave_a to Directory'),\
                    None, gtk.FILE_CHOOSER_ACTION_CREATE_FOLDER)
            folder.set_current_folder(config['dir_BU'])
            folder.set_show_hidden(True)
            response = folder.run()
            temp = folder.get_current_folder()
            config['dir_BU'] = temp
            folder.destroy()
            # problem on Windows with non ASCII char
            free_tuple = disk_usage(config['dir_BU'])
            mess = (config['dir_BU'] + "\n %s"%free_tuple[0])
            #print("The folder return is " + mess)
            self.set_label_attribute(free_tuple[1], mess, self.label0)
            self.label0.set_text(mess)
            
            # compare config to recall config to set dirty flag
            self.manage_dirty_flag()

    def on_nr_kept_change(self, spinbutton):
        global config
        if not active:
            config['kept'] = spinbutton.get_value_as_int()

            # compare config to recall config
            self.manage_dirty_flag()
        else: spinbutton.set_value(config['kept'])

    def on_toggled_check_change(self, rbtn, data=None):
        global config
        if not active :
            if self.rbtn.get_active() != config['start']:
                config['start'] = not config['start']
            else: self.rbtn.set_active(config['start'])

            # compare config to recall config
            self.manage_dirty_flag()

    def on_time_interval_change(self, spinbutton):
        global config
        if not active:
            config['interval(s)'] = spinbutton.get_value() * 60.0

            # compare config to recall config
            self.manage_dirty_flag()
        else: spinbutton.set_value(config['interval(s)']/60.0)

    def on_img_source_change(self, combo1):
        global config
        if not active:
            config['image'] = combo1.get_active()

            # compare config to recall config
            self.manage_dirty_flag()
        else: combo1.set_active(config['image'])

    def on_file_ext_change(self, combo):
        global config
        if not active:
            config['extension'] = combo.get_active()

            # compare config to recall config
            self.manage_dirty_flag()
        else: combo.set_active(config['extension'])

    def on_save_now_clicked(self, button):
        #TODO: make default config changeable?
        if not active :
            # two modes: 1) self.recycle = False, 2) self.recycle = True
            # and self.nr is given by 'on_recall_config_clicked()' for replacement

            global recall_config0, recall_config1, recall_config2, recall_config3,\
                recall_config4, msgr
            if self.nr == 0: recall_config0 = config
            elif self.nr == 1: recall_config1 = config
            elif self.nr == 2: recall_config2 = config
            elif self.nr == 3: recall_config3 = config
            elif self.nr == 4: recall_config4 = config
            else: 
                print("Warning: saving number outside available range!")
                return

            re_txt = 'recall_config%d'%self.nr
            shelf_fl.save(re_txt)
            if not self.recycle:
                trans_txt = _("Recall config %d")%self.nr
                self.choices = shelf_fl.list_dict()
                self.combo_box.append_text(trans_txt)
                self.choices_combo.append(trans_txt)
                self.combo_box.set_active(self.choices_combo.index(trans_txt))
                self.nr += 1
                if self.nr > 4 :
                    self.nr = 0
                    self.recycle = True
                    self.label5.set_text(msgr)
                save_dict = _("Save config %d>")%self.nr
            else: 
                self.label5.set_text(msgr)
                save_dict = _('Save config ')+str(self.nr)
                if self.combo_box.get_active() == self.nr + 2:
                    # reset the flag to non dirty
                    self.button2.set_label('')

            button.set_label(save_dict)

    def on_recall_config_clicked(self, combo_box):
        global config
        if not active:
            j = self.combo_box.get_active()
            config = shelf_fl.recall(self.choices[j])
            self.set_config()
            # to compare later with the actual config
            self.config_sav = deepcopy(config)
            self.button2.set_label('')

            if self.recycle and self.choices[j][:5] == 'recal':
                self.nr = int(self.choices[j][-1:])
                self.button.set_label(_('Save config %d>')%self.nr)
                self.label5.set_text(_("Second, in button 'Save ...' the selected number ")\
                    +_("'%d>' is not saved yet.\nThen set the new config values before")%self.nr\
                    +_(" clicking it, if you want a change."))

    def on_reset_recall_config_clicked(self, widget):
        global config
        if not active and self.button2.get_label() == '*':
            j = self.combo_box.get_active()
            config = shelf_fl.recall(self.choices[j])
            self.set_config()
            # reset the flag to non dirty
            self.button2.set_label('')
        return


    def on_activate_clicked(self, switch, data=None):
        """
        The button 'Start' pass from the config phase to backup phase with the displayed values in config area and 'Stop' close the plug-in.
        """
        global active, config, laststop_config, last_clock   #stop_config,
        last_cfg = {}

        if switch.get_label() == _("Start"):
            active = True
            switch.set_label(_("Stop"))
            self.stock_ic.set_from_stock(gtk.STOCK_STOP, gtk.ICON_SIZE_BUTTON)

            # change top instructions
            top_label2 = _("Only button active after 'Start' is now 'Stop'.")\
               +_("\nThen minimize (close=stop) the window until needed.")
            self.label_.set_text(top_label2)
    
            # change title and color of frame status
            self.label4.set_label("<span weight='bold'>%s</span>"%_("Backup status"))
            self.set_label_attribute(2, _("Backup status"), self.label4)
            # and gray the config area frames
            span_var = "<span foreground='#5c5858' background='light gray' weight='bold' >%s</span>"
            self.label1.set_label(span_var%_("Stop directory"))
            self.label2.set_label(span_var%_("Stop files"))
            self.label3.set_label(span_var%_("Stop time"))

            # message to warn of a running 'autosave_a', just see his window
            warn_mess = ""
            if shelf['autosaver']['running'] > 0:
                warn_mess = _("WARNING:\n  another 'autosave_a' is open!\nSee ")\
                    +_("his window for the configuration.\n****************\n")
            shelf['autosaver'] = {'running' : shelf['autosaver']['running'] + 1}

            # a message to remenber the saving dir
            last_clock = time.mktime(time.localtime())
            begin = time.strftime("%a, %d %b %Y %H:%M:%S")
            free_space = disk_usage(config['dir_BU'])[0]
            gimp.message(warn_mess + _("INFO:\n  was started on:\n ")\
                +_("%s;\nthe backup folder is\n %s")%(begin, \
                config['dir_BU']) +_("\nwith %s.")%free_space)
            #self.free_initial = free_space[:free_space.find(' ')+3]
        else:
            active = False
            end = time.strftime("%a, %d %b %Y %H:%M:%S")
            end_message = _("INFO:\n  was stopped on:\n %s;")% end
            if cntr > 1 : end_message += _("\nafter %d rounds of backup.")\
                                %cntr
            else : end_message += _("\nafter %d round of backup.")%cntr

            if not os.path.exists(sys_file(config['dir_BU'])):
                end_message += _("\nThe folder is no longer there, already!")
            else:
                free_space = disk_usage(config['dir_BU'])
                if free_space[1] != 0 and cntr != 0 :
                    end_message += _("\nWith %s left.")%free_space[0]
            # one less in gimpshelf
            shelf['autosave'] = {'instance': shelf['autosave']['instance'] - 1}
            shelf['autosaver'] = {'running' : shelf['autosaver']['running'] - 1}
            # save the stop config
            laststop_config = config
            shelf_fl.save('laststop_config')
            gimp.message(end_message)
            self.button1.connect("released", gtk.main_quit)
        return
        
    def set_label_attribute(self, color, mess, label):
        """
        Helper function, set a 'gtk.label' with a pango foreground color attribute.
        Parameter 'color' is an integer: 0: green, 1: blue, 2: red; for foreground.
        """
        color_ind = [(0, 32768, 8000), (0, 8000, 32768), (32768, 8000, 0)]
        attr = pango.AttrList()
        colo = color_ind[color]
        fg_color = pango.AttrForeground(colo[0], colo[1], colo[2], 0, len(mess))
        if color == 2 :
            # set AttrBackground for warning
            bg_color = pango.AttrBackground(50000, 65535, 48000, 0, len(mess))
            attr.insert(bg_color)
        attr.insert(fg_color)
        label.set_attributes(attr)
        return

    def timer_action(self):
        global last_clock
        if active:
            clock_cur = time.mktime(time.localtime()) - last_clock
            at_start = (cntr == 0 and config['start'])
            at_end = clock_cur > config['interval(s)']
            if at_start or at_end:
                mess = _("Saving round #%d now...")%(cntr+1)
                self.label5.set_text(mess)
                self.show_all()
                # verify the actual existence of 'config['dir_BU']'
                dir_back = sys_file(config['dir_BU'])
                if not os.path.exists(dir_back):
                    try: os.mkdir(dir_back)
                        # if true was an user mistake; this tries to rectify it!
                    except:
                        # probably a removed media
                        bail_out(_("\nunable to recreate former folder!"), 1)
                        return False

                backup_time(self.image)
                if at_end: last_clock += config['interval(s)']

                # give free space left if low, at the folder label.
                free_tuple = disk_usage(config['dir_BU'])
                if free_tuple[1] == 2:     # free space low
                    free_str = free_tuple[0][:free_tuple[0].find(' ')+3]
                    if free_str == '0 B ':
                        bail_out(_("\nno free space left on the media or disk."), 1)
                        return False
                    mess1 = config['dir_BU'] + _("\nLow free space: %s left now")\
                        %(free_str)
                    self.set_label_attribute(2, mess1, self.label0)
                    self.label0.set_text(mess1)

                self.flag = True
            else:
                if self.flag:
                    self.msgb = _("Backup round done: %d ; ")%cntr
                    self.flag = False
                msgb = self.msgb + _("next in %d s .")%(config['interval(s)']-clock_cur)
                #self.set_label_attribute(2, msgb, self.label2)
                self.label5.set_text(msgb)
                self.show_all()
            
        return True

def backup_time(image):
    """
    A modification of 'autosave.py'
    """
    global cntr, backupFiles
    cntr += 1

    img_list = gimp.image_list()
    curImages = {}
    # to find unsave change: dirty = k.dirty, is True or False
    for k in img_list :
        key = k.ID
        if config['image'] == 2 : curImages[key] = k
        elif  k.dirty and config['image'] == 1 : curImages[key] = k
        elif [k.name, key] == image and config['image'] == 0 :
            curImages[key] = k

    # if backup only the changed image, the nr kept can be surprising
    curIDs = curImages.keys()
    oldIDs = backupFiles.keys()
    newIDs = [x for x in curIDs if x not in oldIDs];
    delIDs = [x for x in oldIDs if x not in curIDs];

    # remove closed images in backups list
    for id in delIDs:
        #print("The backup file '%s', removed from the list."\
        #   %backupFiles[id])
        del(backupFiles[id])

    if curIDs == []:
        # no image to backup
        print("No image to backup in cycle %d , for autosave_a.py"%cntr)
        return

    # backupFiles is a list of filename stub and cycle counter for keeping
    for id in newIDs:
        # to avoid 'Untitle' for imported image file in GIMP-2.8!
        if gimp.version >= (2, 8, 0):
            temp_name = str(curImages[id].filename)
            if temp_name:
                start_tmp = temp_name.rfind(os.sep)+1
                end_tmp = temp_name.find('.', start_tmp)
                if end_tmp > 0:
                    cur_name = temp_name[start_tmp:end_tmp]
                else: cur_name = temp_name[start_tmp:]
            else: cur_name = curImages[id].name[:curImages[id].name.find('.')]
        # this was working in 2.6
        else: cur_name = curImages[id].name[:curImages[id].name.find('.')]

        prefix = 'BU-ID' + str(id) + '-' + cur_name + '-'
        backupFiles[id] = [config['dir_BU'] + os.sep + prefix, cntr]

    # backup images by replacing the last file if >= kept;
    for id, stub in backupFiles.iteritems():
        if cntr- stub[1] >= config['kept']:
            # failed because of user file erase: corrected in next 2 lines ===
            pre_file = sys_file(stub[0]+str(stub[1])+exten[config['extension']])
            if os.path.isfile(pre_file): os.remove(pre_file)
            backupFiles[id][1] += 1
        img = curImages[id]
        filename = stub[0] + str(cntr) + exten[config['extension']]
        try:
            pdb.gimp_file_save(img, img.active_drawable, filename, filename)
        except:
            if os.name == 'nt':
                gimp.message("ERROR in backup image: "+filename)
            else:
                print("ERROR in backup image: "+filename)

    return

def cfg2lines(dict):
    """
    Transform the items of a dictionary to a multi-lines text.
    Input: dictionary
    Output: text
    """
    global source, exten

    lines = ""
    keys_list = dict.keys()
    nr_key = len(keys_list)
    # template of line: "key: value,\n" (a period for last line)
    for i in range(nr_key):
        if keys_list[i] == 'image': value_str = source[int(dict[keys_list[i]])]
        elif keys_list[i] == 'extension': value_str = exten[int(dict[keys_list[i]])]
        else: value_str = str(dict[keys_list[i]])
        lines += keys_list[i]+': '+value_str+" ,\n"
    lines = lines[:-2]+".\n"
    
    return lines

def disk_usage(path):
    """
    Return disk usage about the given path as free space in this adaptation.
    Values are expressed in bytes.
    From author: Giampaolo Rodola' <g.rodola [AT] gmail [DOT] com>
    License: MIT
    """

    freemem_low = 1 #1 for blue text
    if hasattr(os, 'statvfs'):  # POSIX
        st = os.statvfs(path)
        free = st.f_bavail * st.f_frsize

    elif os.name == 'nt':       # Windows
        var, total, free = ctypes.c_ulonglong(), ctypes.c_ulonglong(), \
                           ctypes.c_ulonglong()
        if sys.version_info >= (3,) or isinstance(path, unicode):
            fun = ctypes.windll.kernel32.GetDiskFreeSpaceExW
        else:
            fun = ctypes.windll.kernel32.GetDiskFreeSpaceExA
        ret = fun(sys_file(path), ctypes.byref(var), ctypes.byref(total), \
              ctypes.byref(free))
        if ret == 0:
            raise ctypes.WinError()
        free = free.value
        
    else:
        return (_("check free space"), 0)   #0 for green text

    # if less then 1.5 MB warn by returning a 'freemem_low = -1'
    if free <= 1500000: freemem_low = 2    #2 for red text
    # abbreviate free space number
    symbols = ('K', 'M', 'G', 'T', 'P', 'E', 'Z', 'Y')
    prefix = {}
    for i, s in enumerate(symbols):
        prefix[s] = 1 << (i+1)*10
    for s in reversed(symbols):
        if free >= prefix[s]:
            value = float(free) / prefix[s]
            return (_('%.1f %sB free space') %(value, s), freemem_low)
    return (_("%d B free space") %free, freemem_low)

def popup_mess(message):
    flag = gtk.DIALOG_MODAL|gtk.DIALOG_DESTROY_WITH_PARENT
    msgBox = gtk.MessageDialog(None, flag, gtk.MESSAGE_WARNING, gtk.BUTTONS_OK,\
        message)
    msgBox.run()
    msgBox.destroy()
    return

def bail_out(message, source):
    """
    parameters: message refers to additional text for the cause
                source is an integer to identify the function responsible
    """
    if source == 1: # its autosave_a
        # one less in gimpshelf
        shelf['autosave'] = {'instance': shelf['autosave']['instance'] - 1}
        shelf['autosaver'] = {'running' : shelf['autosaver']['running'] - 1}
        func = 'autosave_a'
    elif source == 2: # its noUI_autosave_a
        print("Forced stop of 'noUI_autosave_a' at "+time.strftime("%a, %d %b %Y %H:%M:%S"))
        func = 'noUI_autosave_a'
        shelf['noUI_autosave'] = ''
        #del shelf['noUI_autosave'] # produce an error gimpshelf.py?
    else: return
    # quit with error and pop the message
    #if folder_prob:
        #message = (_("ERROR:\n  %s periodic backup was suspended after round #%d ;")\
            #%(func, cntr) + message)        
    #else:
    message = (_("ERROR:\n  %s periodic backup was suspended after round #%d in %s ;")\
        %(func, cntr, config['dir_BU']) + message)
    popup_mess(message)
    gimp.message(message)

    #quit()
    gtk.main_quit()
    return


class NoUI_Autosave():
    """
    For a non interactive 'autosave_a'
    """

    def __init__(self):
        timeout_add_seconds(1, self.timer_action, priority=308)
    
    def timer_action(self):
        global last_clock, active

        # start when there is an open image
        if not active:
            img_list = gimp.image_list()
            nb_img = len(img_list)
            if nb_img > 0:
                active = True
                last_clock = time.mktime(time.localtime())
                self.image = [img_list[nb_img-1].name, img_list[nb_img-1].ID]
        if active:
            clock_cur = time.mktime(time.localtime()) - last_clock
            at_start = (cntr == 0 and config['start'])
            at_end = clock_cur > config['interval(s)']
            if at_start or at_end:
                # verify the actual existence of 'config['dir_BU']'
                if not os.path.exists(sys_file(config['dir_BU'])):
                    try: os.mkdir(sys_file(config['dir_BU']))
                        # if true probably was an user mistake; this rectifies it!
                    except:
                        bail_out(_("\nunable to recreate former folder!"), 2)
                        return False
                        
                backup_time(self.image)
                free_tuple = disk_usage(config['dir_BU'])
                if free_tuple[1] == 2:     # free space low
                    free_str = free_tuple[0][:free_tuple[0].find(' ')+3]
                    if free_str == '0 B ':
                        bail_out(_("\nno free space left on the media or disk."), 2)
                        return False
                if at_end: last_clock += config['interval(s)']
                #print("Backup round done: %d in noUI_autosave_a"%cntr)
        return True


################################################################################

def autosave_a(img, item):

    # for testing
    # beep()
    
    # count instance
    if shelf.has_key('autosave'):
        shelf['autosave'] = {'instance' : shelf['autosave']['instance'] + 1}
    else:
        shelf['autosave'] = {'instance' : 1}
        shelf['autosaver'] = {'running' : 0}

    if warning: gimp.message(warning)
    co_au = Control_Autosave(img)
    gtk.main()
    pdb.gimp_displays_flush()
    #print("List of dict. in shelf_fl: "+str(shelf_fl.list_dict()))

def noUI_autosave_a():

    # with gimpshelf avoid duplicate launch
    if shelf.has_key('noUI_autosave') and shelf['noUI_autosave']:
        gimp.message(_("ERROR: a 'noUI_autosave_a' intance is already running!"))
    #elif shelf.has_key('autosave') and config == shelf['autosave']:
        #gimp.message(_("ERROR: 'autosave_a' is running with same config!"))
    else:
        if warning: gimp.message(warning)
        shelf['noUI_autosave'] = config
        print("Start of 'noUI_autosave_a' at "+time.strftime("%a, %d %b %Y %H:%M:%S"))
        NoUI_Autosave()
        gtk.main()


register(
        "autosave_a",
        _("It configures auto-backup of open images, starts and stops it.")\
            +_("\nFrom: ")+__file__,
        _("Periodically saves chosen opened images to a backup directory"),
        "R. Brizard",
        "(c) R. Brizard",
        "2012",
        _("Auto save..."),
        "*",
        [(PF_IMAGE, "img", "IMAGE:", None),
         (PF_DRAWABLE, "drawable", "DRAWABLE:", None)],
        [],
        autosave_a,
        menu = _("<Image>/Extensions/Plugins-Python/Fichier"),
        #menu = "<Image>/Plugins-Python/Fichier",
        domain = ("autosave_a", locale_directory))

register(
        "noUI_autosave_a",
        _("It accepts previous 'Laststop' configuration of autosave_a.")\
            +_("\nFrom: ")+__file__,
        _("Periodically saves opened images to a backup directory, without ")
            +_("configuration option."),
        "R. Brizard",
        "(c) R. Brizard",
        "2012",
        _("_NoUI auto save"),
        "",
        [],
        [],
        noUI_autosave_a,
        menu = _("<Image>/Extensions/Plugins-Python/Fichier"))
        #menu = "<Image>/Plugins-Python/File")

main()
