#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
Script type: Python-fu
Script Title: Luminosity Mask Drive Thru
Tested on GIMP version: 2.10.22; Windows 10, 64 bit; GTK 2.28.7

What is Does:
    Create luminosity mask from a layer.

Where The Script Installs:
    See the how-to file.

Get the latest version: github.com/gummycowboy
"""
from gimpfu import pdb
import gimpfu as fu
import gtk
import pygtk
import sys
pygtk.require("2.0")

DEBUG = 0
program_title = "Luminosity Mask Drive Thru 1.01"

LIGHTS_MASK_NAME = (
        "Lights ",
        "Lighter Lights",
        "Bright Lights",
        "Super Lights",
        "Ultra Lights"
    )

DARKS_MASK_NAME = (
        "Darks ",
        "Darker Darks",
        "Shadow Darks",
        "Super Darks",
        "Ultra Darks"
    )

MIDTONES_MASK_NAME = (
        "Basic Midtones",
        "Expanded Midtones",
        "Wide Midtones",
        "Super Midtones"
    )

# color:
MAX_COLOR = 65535
BLACK = gtk.gdk.Color(0, 0, 0)
GREEN = gtk.gdk.Color(0, MAX_COLOR, 0)
GREY = gtk.gdk.Color(MAX_COLOR / 2, MAX_COLOR / 2, MAX_COLOR / 2)
RED = gtk.gdk.Color(MAX_COLOR, 0, 0)
WHITE = gtk.gdk.Color(MAX_COLOR, MAX_COLOR, MAX_COLOR)

MARGIN = 20


def start(image, drawable):
    """
    Start the program.

    typical variable usage:
        a : part of a sequence; object, int
        b : see ‟a”
        c : see ‟a”; column
        d : dict
        e : dict
        f : float
        g : widget; window
        h : height
        i : iteration
        j : GIMP image; iteration
        k : key
        m : flag
        n : string
        p : process
        q : iterable
        r : row
        s : size
        t : size
        u : point
        v : point
        w : width
        x : coordinate; index
        y : coordinate
        z : layer

    image: GIMP image
        the active image

    drawable: drawable
        the active drawable
    """
    # Save the interface context so that it can be restored:
    pdb.gimp_context_push()

    # Make the script undo-able with one step:
    pdb.gimp_image_undo_group_start(image)

    try:
        UIMain(image, drawable)

    except Exception as ex:
        # Comm.show_err(ex)
        if DEBUG:
            print >> sys.stderr, ex

    pdb.gimp_image_undo_group_end(image)

    # Restore the interface context:
    pdb.gimp_context_pop()


def hide_all(z):
    """
    Hide all the layers in the image.

    z: layer or GIMP image
    """
    for z1 in z.layers:
        if not hasattr(z1, 'layers'):
            pdb.gimp_item_set_visible(z1, 0)

        else:
            # recursive:
            hide_all(z1)


def info_msg(n):
    """
    Use to output messages to the error console.

    n: string
        message to display
    """
    a = pdb.gimp_message_get_handler()

    pdb.gimp_message_set_handler(fu.ERROR_CONSOLE)
    fu.gimp.message(n)
    pdb.gimp_message_set_handler(a)


def invert_selection(j):
    """
    Invert a selection.

    j: GIMP image
    """
    if is_selection(j):
        pdb.gimp_selection_invert(j)


def is_selection(j):
    """
    Return true if there is a selection.

    j: GIMP image
    """
    return not pdb.gimp_selection_is_empty(j)


def pop_up(parent, x, n, title):
    """
    Display a message dialog.

    parent: GTK window
        parent window

    x: int
        message type index, (question, info)     (0..1)

    n: string
        message

    title: string
        window title

    Return true (as 1) if the user responded with yes.
    """
    g = gtk.MessageDialog(
            parent=parent,
            flags=gtk.DIALOG_MODAL,
            type=(gtk.MESSAGE_QUESTION, gtk.MESSAGE_INFO)[x],
            buttons=(gtk.BUTTONS_YES_NO, gtk.BUTTONS_CLOSE)[x],
            message_format=n
        )

    g.set_title(title)

    a = g.run()

    g.destroy()
    return int(a == gtk.RESPONSE_YES)


def restore_visibility(z, d):
    """
    Hide all the layers in the image.

    z: layer or GIMP image
    d: dict
        Has layer visibility state where key: value is
        layer name: (0..1) (hidden..visible).
    """
    for z1 in z.layers:
        if z1.name in d:
            pdb.gimp_item_set_visible(z1, d[z1.name])

        else:
            # New layers are assumed to be visible:
            pdb.gimp_item_set_visible(z1, 1)
        if hasattr(z1, 'layers'):
            # recursive:
            restore_visibility(z1, d)


def save_visibility(z, d):
    """
    Store the visibility state of all the layers in the image.

    z: layer or GIMP image
    d: dict
        Store layer state where key: value is
        layer name: (0..1) (hidden..visible).
    """
    for z1 in z.layers:
        d[z1.name] = pdb.gimp_item_get_visible(z1)
        if hasattr(z1, 'layers'):
            # recursive:
            save_visibility(z1, d)


def show_layer(z):
    """
    Ensure a layer is visible.

    z: layer
        Is either the start layer or a parent,
        as in a group layer, of the start layer.
    """
    pdb.gimp_item_set_visible(z, 1)
    if z.parent:
        show_layer(z.parent)


class Widget:
    """This is the base widget for the custom widgets."""

    def __init__(self, callback):
        """
        Has widget factored functions and attributes.

        Sub-widget classes need to init the Widget
        class before connecting events.

        callback: function
            Connect widget with event handler.
        """
        self._callback = callback

    def callback(self, *_):
        """
        The widget changed or was activated.

        Call the widget's feedback function.
        """
        self._callback(self)


class EventBox(gtk.EventBox):
    """This is a custom GTK EventBox."""

    def __init__(self, color):
        """
        color: gtk color component
            gdk.gtk color
        """
        super(gtk.EventBox, self).__init__()
        if color is not None:
            self.modify_bg(gtk.STATE_NORMAL, color)


class Button(Widget):
    """This is a custom GTK Button."""

    def __init__(
                self,
                text,
                on_action,
                padding=None,
                align=(0, 0, 1, 0)
            ):
        """
        Create a gtk.Button that is attached to an gtk.Alignment.

        Use the Widget class callback.

        text: string
            Button label

        on_action: function
            callback on action

        padding: tuple
            Alignment padding
        """
        g = self.alignment = gtk.Alignment(*align)
        g1 = self.button = gtk.Button(text)

        Widget.__init__(self, on_action)
        g1.connect('clicked', self.callback)
        g.add(g1)
        if padding:
            g.set_padding(*padding)


class CheckButton(Widget):
    """
    This is a custom GTK CheckButton.

    Is attached to its own Alignment.
    """

    def __init__(self, text, on_action, padding=None):
        """
        Create CheckButton and Alignment.

        text: string
            label text

        on_action: function
            Call on check button action.

        padding: tuple of int
            Alignment padding (top, bottom, left, right)
        """
        g = self.alignment = gtk.Alignment(0, 0, 0, 0)
        g1 = self.checkbutton = gtk.CheckButton(label=text)

        Widget.__init__(self, on_action)
        g.add(g1)
        g1.connect('clicked', self.callback)
        g1.connect('activate', self.callback)
        if padding:
            g.set_padding(*padding)

    def get_value(self):
        """
        Return the value of the CheckButton.
        """
        return int(self.checkbutton.get_active())


class Label:
    """This is a custom GTK Label."""

    def __init__(self, text, padding=None, align=(0, 0, 0, 0)):
        """
        text: string
            label text

        padding: tuple of int
            (top, bottom, left, right) margin

        align: tuple of float
            (top, bottom, left, right) space reservation (0..1)
        """
        self.label = gtk.Label(text)
        g = self.alignment = gtk.Alignment(*align)

        if padding:
            g.set_padding(*padding)
        g.add(self.label)


class UIMain():
    """This is the script's main window."""

    def __init__(self, image, drawable):
        """
        This is where the main GTK event loop takes place.

        image: GIMP image
            the active image

        drawable: drawable
            the active drawable
        """
        self.image = image
        self.layer = drawable
        self.selection = None

        if pdb.gimp_item_is_layer(self.layer):
            if not pdb.gimp_selection_is_empty(image):
                self.selection = pdb.gimp_selection_save(self.image)

            g = self.win = gtk.Window(gtk.WINDOW_TOPLEVEL)
            g.set_title(program_title)
            g.set_position(gtk.WIN_POS_CENTER)
            self.light_widgets = []
            self.dark_widgets = []
            self.midtone_widgets = []

            self._draw_window()
            g.show_all()
            g.connect('key_press_event', self.on_key_press)
            g.connect("delete_event", lambda *_: gtk.main_quit())
            self._go_button.button.set_sensitive(0)
            gtk.main()

        else:
            pop_up(
                    None,
                    1,
                    program_title + "\nrequires an image"
                    " layer to be selected.",
                    "No Active Layer Error"
                )

    def _create_midtone_mask_layers(self, j, selections):
        """
        Create the mask layer from the selections.

        j: GIMP image
        selections: iterable
            list of selections that create masks
        """
        z = self.layer
        for i in selections:
            x = i[1]
            if self.midtone_widgets[x].get_value():
                z = self.clone_layer(j, z)
                z.name = self.layer.name + ": " + MIDTONES_MASK_NAME[x]

                # Load selection for the mask:
                pdb.gimp_image_select_item(j, fu.CHANNEL_OP_REPLACE, i[0])

                # Create mask from the selection:
                mask = pdb.gimp_layer_create_mask(z, fu.ADD_MASK_SELECTION)

                pdb.gimp_layer_add_mask(z, mask)
                self._isolate_selection(z)

    def _create_midtone_mask_selections(
                self,
                j,
                light_selections,
                dark_selections
            ):
        """
        Create a midtone mask based on an index
        where zero references the first mask layer.

        j: GIMP image
        light_selections: iterable
            list of light mask selections

        dark_selections: iterable
            list of dark mask selections

        Return a list of midtone selections.
        """
        selections = []

        for i in range(3, -1, -1):
            if self.midtone_widgets[i].get_value():
                pdb.gimp_selection_all(j)

                # light alpha:
                pdb.gimp_image_select_item(
                        j,
                        fu.CHANNEL_OP_SUBTRACT,
                        light_selections[i + 1]
                    )

                # dark alpha:
                pdb.gimp_image_select_item(
                        j,
                        fu.CHANNEL_OP_SUBTRACT,
                        dark_selections[i + 1]
                    )

                # Save selection:
                selections.append((pdb.gimp_selection_save(j), i))
        return selections

    def _create_polar_mask_selections(self, j, mask_count, is_darks=False):
        """
        Create a base mask, either light or dark.

        j: GIMP image
        is_darks: flag
            If it's true, the group is for the darks-type masks.

        Return a list of polar-type selections.
        """
        n = "Darks" if is_darks else "Lights"
        z = self.clone_layer(j, self.layer)
        selections = []

        pdb.gimp_drawable_desaturate(z, fu.DESATURATE_LUMINANCE)

        if is_darks:
            pdb.gimp_drawable_invert(z, 0)

        channel = pdb.gimp_channel_new_from_component(
                j,
                fu.CHANNEL_RED,
                n,
            )

        pdb.gimp_image_insert_channel(
                j,
                channel,
                None,
                0
            )

        # Select channel to get grayscale pixels where black
        # equates to transparent and white to opaque.
        pdb.gimp_image_select_item(j, fu.CHANNEL_OP_REPLACE, channel)
        selections.append(pdb.gimp_selection_save(j))

        mask = pdb.gimp_layer_create_mask(z, fu.ADD_MASK_SELECTION)

        pdb.gimp_layer_add_mask(z, mask)

        if mask_count > 1:
            for _ in range(mask_count - 1):
                invert_selection(j)
                replace_sel = pdb.gimp_selection_save(j)

                # Restore the original selection:
                invert_selection(j)

                # Subtract inverted selection:
                pdb.gimp_image_select_item(
                        j,
                        fu.CHANNEL_OP_SUBTRACT,
                        replace_sel
                    )

                # The selection will make the mask:
                selections.append(pdb.gimp_selection_save(j))
                pdb.gimp_image_remove_channel(j, replace_sel)

        pdb.gimp_image_remove_layer(j, z)
        pdb.gimp_image_remove_channel(j, channel)
        return selections

    def _create_polar_mask_layers(self, j, selections, mask_index):
        """
        Create the mask layer from the selections.

        j: GIMP image
        selections: iterable
            list of selections that create masks

        mask_index: int
            0: light, 1: dark
        """
        z = self.layer
        mask_name = (LIGHTS_MASK_NAME, DARKS_MASK_NAME)[mask_index]
        widgets = (self.light_widgets, self.dark_widgets)[mask_index]
        for x, i in enumerate(selections):
            x1 = (4, 3, 2, 1, 0)[x]
            if widgets[x1].get_value():
                z = self.clone_layer(j, z)
                z.name = self.layer.name + ": " + mask_name[x]

                # Load selection for mask:
                pdb.gimp_image_select_item(j, fu.CHANNEL_OP_REPLACE, i)

                # Create the mask from the selection:
                mask = pdb.gimp_layer_create_mask(z, fu.ADD_MASK_SELECTION)

                pdb.gimp_layer_add_mask(z, mask)
                self._isolate_selection(z)

    def _draw_darks_widgets(self, table):
        """
        Draw the darks-mask-type widgets.

        table: GTK table
            container for widgets
        """
        w = MARGIN
        w1 = w / 2
        pad = w1, w1, w, w
        g = Label("Darks", padding=pad, align=(1, 1, 1, 1))
        event_box = EventBox(BLACK)

        g.label.modify_fg(gtk.STATE_NORMAL, WHITE)
        event_box.add(g.alignment)
        table.attach(event_box, 2, 3, 0, 1)
        for i in range(5):
            x = (4, 3, 2, 1, 0)[i]
            g = CheckButton(
                    DARKS_MASK_NAME[x],
                    self.on_widget_change,
                    padding=pad
                )

            table.attach(g.alignment, 2, 3, i + 1, i + 2)
            self.dark_widgets.append(g)

    def _draw_lights_widgets(self, table):
        """
        Draw the lights-mask-type widgets.

        table: GTK Table
            container for widgets
        """
        w = MARGIN
        w1 = w / 2
        pad = w1, w1, w, w
        g = Label("Lights", padding=pad, align=(1, 1, 1, 1)).alignment
        event_box = EventBox(WHITE)

        event_box.add(g)
        table.attach(event_box, 0, 1, 0, 1)
        for i in range(5):
            x = (4, 3, 2, 1, 0)[i]
            g = CheckButton(
                    LIGHTS_MASK_NAME[x],
                    self.on_widget_change,
                    padding=pad
                )

            table.attach(g.alignment, 0, 1, i + 1, i + 2)
            self.light_widgets.append(g)

    def _draw_midtone_widgets(self, table):
        """
        Draw the lights-mask-type widgets.

        table: GTK Table
            container for widgets
        """
        w = MARGIN
        w1 = w / 2
        pad = w1, w1, w, w
        g = Label("Midtones", padding=pad, align=(1, 1, 1, 1)).alignment
        event_box = EventBox(GREY)

        event_box.add(g)
        table.attach(event_box, 1, 2, 0, 1)
        for i in range(5):
            if i:
                g = CheckButton(
                        MIDTONES_MASK_NAME[i - 1],
                        self.on_widget_change,
                        padding=pad
                    )

                self.midtone_widgets.append(g)
                table.attach(g.alignment, 1, 2, i + 1, i + 2)

    def _draw_window(self):
        """Draw the window's widgets."""
        w = MARGIN
        w1 = w / 2
        box = gtk.VBox()

        # Table has 6 rows, 3 columns:
        table = gtk.Table(6, 3)

        table.set_homogeneous(True)
        box.add(table)
        self.win.add(box)
        self._draw_lights_widgets(table)
        self._draw_darks_widgets(table)
        self._draw_midtone_widgets(table)

        table = gtk.Table(1, 2)
        box.add(table)

        event_box = EventBox(RED)
        g = self._stop_button = Button(
                    "Stop",
                    self.close,
                    padding=(w1, w1, w, w1)
                )

        event_box.add(g.alignment)
        table.attach(event_box, 0, 1, 0, 1)

        event_box = EventBox(GREEN)
        g = self._go_button = Button(
                "Go",
                self.do_job,
                padding=(w1, w1, w1, w)
            )

        event_box.add(g.alignment)
        table.attach(event_box, 1, 2, 0, 1)

    def _get_mask_counts(self):
        """
        Return the number of selections that are needed to make masks.
        """
        dark_count = light_count = 0
        midtone_count = -1

        for x, g in enumerate(self.light_widgets):
            if g.get_value():
                x1 = (5, 4, 3, 2, 1)[x]
                light_count = x1
                break

        for x, g in enumerate(self.dark_widgets):
            if g.get_value():
                x1 = (5, 4, 3, 2, 1)[x]
                dark_count = x1
                break

        for x, g in enumerate(self.midtone_widgets):
            if g.get_value():
                midtone_count = x + 1

        light_count = max(light_count, midtone_count + 1)
        dark_count = max(dark_count, midtone_count + 1)
        midtone_count = max(0, midtone_count)
        return light_count, dark_count, midtone_count

    def _isolate_selection(self, z):
        """
        If there's was a selection at the start,
        then isolate the selection on the mask layer.

        z: layer
            mask layer
        """
        if self.selection:
            pdb.gimp_image_select_item(
                    self.image,
                    fu.CHANNEL_OP_REPLACE,
                    self.selection
                )

            invert_selection(self.image)
            pdb.gimp_edit_clear(z)

    def clone_layer(self, j, z):
        """
        Duplicate a layer.

        j: GIMP image
        z: layer

        Return the duplicated layer.
        """
        pdb.gimp_selection_all(j)

        is_start_layer = 1 if z == self.layer else 0
        z1 = pdb.gimp_layer_copy(z, 1)

        # Insert the layer above the active layer:
        pdb.gimp_image_set_active_layer(j, z)
        pdb.gimp_image_insert_layer(j, z1, z.parent, -1)

        # Perceptual space creates smoother masks:
        pdb.gimp_layer_set_composite_space(
                z,
                fu.LAYER_COLOR_SPACE_RGB_PERCEPTUAL
            )

        if is_start_layer and z.mask:
            # Apply the start layers mask to the clone's alpha:
            z2 = pdb.gimp_layer_copy(z, 1)

            pdb.gimp_image_insert_layer(j, z2, z.parent, -1)
            pdb.gimp_edit_clear(z1)
            z1 = pdb.gimp_image_merge_down(j, z2, fu.CLIP_TO_IMAGE)

        # Copies the mask, but it's not needed:
        if z1.mask:
            z1.remove_mask(fu.MASK_DISCARD)

        pdb.gimp_selection_none(j)
        return z1

    def close(self, *_):
        """
        Close the window.
        Exit the main event-handler loop.
        """
        self.win.emit("delete-event", gtk.gdk.Event(gtk.gdk.DELETE))

    def do_job(self, *_):
        """Create the luminosity masks."""
        self.win.hide()

        light_count, dark_count, midtone_count = self._get_mask_counts()
        j = self.image
        d = {}
        light_selections = []
        midtone_selections = []
        dark_selections = []

        save_visibility(j, d)
        hide_all(j)
        show_layer(self.layer)

        # Do light selections:
        if light_count:
            # Create the lights base mask needed to create lights-type masks:
            light_selections = self._create_polar_mask_selections(
                    j,
                    light_count,
                    is_darks=False
                )

        # Do dark selections:
        if dark_count:
            # Create the darks base mask needed to create darks-type masks:
            dark_selections = self._create_polar_mask_selections(
                    j,
                    dark_count,
                    is_darks=True
                )

        # Do midtone selections:
        if midtone_count:
            midtone_selections = self._create_midtone_mask_selections(
                    j,
                    light_selections,
                    dark_selections
                )

        restore_visibility(j, d)

        # Make mask layers:
        if light_count:
            self._create_polar_mask_layers(j, light_selections, 0)

        if midtone_count:
            self._create_midtone_mask_layers(j, midtone_selections)

        if dark_count:
            self._create_polar_mask_layers(j, dark_selections, 1)

        # Clean up the polar selections:
        for q in (light_selections, dark_selections):
            for sel in q:
                pdb.gimp_image_remove_channel(j, sel)

        # Clean up the midtone selections:
        for q in midtone_selections:
            pdb.gimp_image_remove_channel(j, q[0])

        self.close()
        pdb.gimp_selection_none(j)
        pdb.gimp_image_set_active_layer(j, self.layer)

        # Restore the original selection:
        if self.selection:
            pdb.gimp_image_select_item(
                    j,
                    fu.CHANNEL_OP_REPLACE,
                    self.selection
                )

    def on_key_press(self, _, a):
        """
        Check to see if the user pressed the Esc key.
        If so, then close the window.

        a: key-press event
        """
        n = gtk.gdk.keyval_name(a.keyval)
        if n == 'Escape':
            return self.close()

        elif n == 'Return':
            if self._go_button.button.get_sensitive():
                self.win.get_focus()
                self.do_job()

    def on_widget_change(self, _):
        """
        Call when a widget changes.

        Update the viability of the Go Button
        per the state of the checkbuttons.
        """
        m = False
        for q in (self.light_widgets, self.dark_widgets, self.midtone_widgets):
            for g in q:
                if g.get_value():
                    m = True
                    break
            if m:
                break

        if m:
            self._go_button.button.set_sensitive(1)

        else:
            self._go_button.button.set_sensitive(0)


fu.register(
        # name
        # becomes dialog title as python-fu + name
        # Space character is not allowed.
        # ‟name” is case-sensitive:
        "Luminosity-Mask-Drive-Thru",

        # tool-tip and window-tip text:
        "Creates luminosity masks from a layer and/or its selection.",

        # Help (describe how-to, exceptions and dependencies).
        # This info will display in the plug-in browser:
        "Creates a stack of optional masked layers. Requires an open image.",

        # The author is displayed in the plug-in browser:
        "Charles Bartley",

        # The copyright is displayed in the plug-in browser:
        "Charles Bartley",

        # The date is displayed in the plug-in browser:
        "2021",

        # menu item descriptor with short-cut key ‟_L”:
        "<Image>/Layer/Mask/_Luminosity Mask Drive Thru...",

        # image types
        # An empty string equates to no image needed:
        "RGB*, GRAY*",

        # dialog parameters:
        [],

        # results:
        [],

        # dialog function handler
        start,
    )


if __name__ == "__main__":
    if DEBUG:
        # Trace errors by creating a file if it doesn't already exist that
        # will accept and append error messages
        # and piped (">>") print statements.
        import time
        sys.stderr = open("D:\\LMDT_error.txt", 'a')
        print >> sys.stderr, "\nLMDT,", time.ctime()
    fu.main()
