; offset_path.scm
; by Rob Antonishen
; http://www.silent9.com

; Version 1.0 (20130417)

; Description
;
; Fractalizes a path using a midpoint desplace algorithm
; with either uniform or gaussian distributions.
;

; Changelog
; 1.0 - original version

; License:
;
; This program is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version. 
;
; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; The GNU Public License is available at
; http://www.gnu.org/copyleft/gpl.html
(define (offset_path img inPath inOffset)

  ; helper functions:
  
  ; TinyScheme lacks pi.
  (define pi (acos -1))

  ;main script
  ;-----------------------------------------------------------------------------

  (let*
    ((width (car (gimp-image-width img)))
	   (height (car (gimp-image-height img)))
     (varVector inPath)
     (varName (car (gimp-vectors-get-name varVector)))
     (newVector 0)
     (temp 0))
    
    (gimp-image-undo-group-start img)
	(gimp-progress-set-text "Offsetting Path...")
    (set! newVector (car (gimp-vectors-new img (string-append (car (gimp-vectors-get-name varVector)) " offset"))))

    ;there is an active vector/path
    (unless (equal? varVector -1)
      (let 
        ((strokelist (cadr (gimp-vectors-get-strokes varVector)))
         (numstrokes (vector-length (cadr (gimp-vectors-get-strokes varVector))))
         (points ())
         (numpoints 0)
         (counter 0))

         (while (< counter numstrokes)  ; for each stroke in the path
           (gimp-progress-set-text (string-append "Analyzing Segment " (number->string (+ counter 1)) " of " (number->string numstrokes)))
           (set! points (caddr (gimp-vectors-stroke-get-points varVector (vector-ref strokelist counter))))
           (set! numpoints (/ (cadr (gimp-vectors-stroke-get-points varVector (vector-ref strokelist counter))) 6))

           (let
              ((pcount 0)
              (pointlist ())
              (newpointlist ())
              (divcounter 0))
              (while (< pcount numpoints)
                 (set! pointlist (append pointlist (list (list (vector-ref points (+ (* pcount 6) 2))
                                                               (vector-ref points (+ (* pcount 6) 3))))))
                 (set! pcount (+ pcount 1))
               )

               ;add two start point to end
               (set! pointlist (append pointlist (list (car pointlist)) (list (cadr pointlist))))
               
               ;pointlist is now a list like ((x1 y1) (x2 y2) ... (xn yn) (x1 y1) (x2 y2)) use list-ref to get each pair
               (gimp-progress-set-text (string-append "Ofsetting " (number->string (+ counter 1)) " of " (number->string numstrokes)))

			   ; do the offset magic here!
			   (set! pcount 0)
			   (while (< pcount (- (length pointlist) 2))
			      (let* (
				       (x1 (car (list-ref pointlist pcount)))
				       (y1 (cadr (list-ref pointlist pcount)))
				       (x2 (car (list-ref pointlist (+ pcount 1))))
				       (y2 (cadr (list-ref pointlist (+ pcount 1))))
				       (x3 (car (list-ref pointlist (+ pcount 2))))
				       (y3 (cadr (list-ref pointlist (+ pcount 2))))
					   (a12 (atan (- y1 y2) (- x1 x2)))
					   (a32 (atan (- y3 y2) (- x3 x2)))
					   (an (if (> a32 a12) (/ (- a32 a12) 2) (/ (- (* 2 pi) (- a12 a32)) 2)))
					   (len (/ inOffset (sin an)))
					   (am (+ an a12))
					   (xoff (* len (cos am)))
					   (yoff (* len (sin am)))
					   )
										
					   (set! newpointlist (append newpointlist (list (list 
			  	             (+ x2 xoff) ;x
			  				 (+ y2 yoff) ;y
			          ))))
				  )
			  	  (set! pcount (+ pcount 1))
			   )
			 
               (set! pointlist newpointlist)

             ;pointlist should be a list with the new points
             ;turn this new list into a set of points and replace the stoke in the array.
             (set! pcount 0)
             (set! numpoints (length pointlist))
             (set! newpointlist ())
             (gimp-progress-set-text (string-append "Remapping " (number->string (+ counter 1)) " of " (number->string numstrokes)))
             (while (< pcount numpoints)
               (if (= (modulo pcount 10) 0) (gimp-progress-pulse))
               (set! temp (list-ref pointlist pcount))
               ; no idea why but appending all of this in one line would cause some to get dropped
               (set! newpointlist (append newpointlist temp))
               (set! newpointlist (append newpointlist temp))
               (set! newpointlist (append newpointlist temp))
               (set! pcount (+ pcount 1))
             )

             (gimp-vectors-stroke-new-from-points newVector 0 (length newpointlist)
                (list->vector newpointlist) TRUE)
           )
           (set! counter (+ counter 1))
         )
      )   
    )
    (gimp-progress-set-text " ")
    (gimp-progress-end)
    (gimp-image-add-vectors img newVector (car (gimp-image-get-vectors-position img varVector)))
    (gimp-vectors-set-visible newVector TRUE)
  	(gimp-displays-flush)
  	(gimp-image-undo-group-end img)
  )
)

(script-fu-register "offset_path"
        		    "Offset Path..."
                    "Offset the points of a path."
                    "Rob Antonishen"
                    "Rob Antonishen"
                    "April 2013"
                    ""
                    SF-IMAGE      "image"              0
                    SF-VECTORS    "path"               0
                    SF-ADJUSTMENT "Offset"       (list 10 -200 200 1 10 0.1 SF-SLIDER)			
)

(script-fu-menu-register "offset_path"
                         "<Vectors>")